/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.auto;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;

import org.ametys.cms.filter.ContentFilter;
import org.ametys.cms.filter.ContentFilterExtensionPoint;
import org.ametys.cms.filter.ContentFilterHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.cms.workflow.SendMailFunction;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.workflow.CreateNewsletterFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebConstants;
import org.ametys.web.filter.WebContentFilter;
import org.ametys.web.filter.WebContentFilterHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Runnable engine that creates the automatic newsletter contents.
 */
public class AutomaticNewslettersSchedulable extends AbstractStaticSchedulable
{
    
    /** The newsletter content type. */
    protected static final String _NEWSLETTER_CONTENT_TYPE = "org.ametys.plugins.newsletter.Content.newsletter";
        
    /** The instant the engine was started. */
    protected Date _runDate;
    
    /** The newsletter workflow name. */
    protected String _workflowName;
    
    /** The workflow initial action ID. */
    protected int _wfInitialActionId;
    
    /** A list of action IDs to validate a newsletter from initial step. */
    protected List<Integer> _wfValidateActionIds;
    
    /** A map of the content IDs by filter, reset on each run. */
    protected Map<String, List<String>> _filterContentIdCache;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The site manager. */
    protected SiteManager _siteManager;
    
    /** The workflow provider. */
    protected WorkflowProvider _workflowProvider;
    
    /** The automatic newsletter extension point. */
    protected AutomaticNewsletterExtensionPoint _autoNewsletterEP;
    
    /** The newsletter category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryEP;
    
    /** The content filter extension point. */
    protected ContentFilterExtensionPoint _contentFilterEP;
    
    /** The content filter helper. */
    protected WebContentFilterHelper _contentFilterHelper;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;    
        
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        // Lookup the needed components.
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _autoNewsletterEP = (AutomaticNewsletterExtensionPoint) manager.lookup(AutomaticNewsletterExtensionPoint.ROLE);
        _categoryEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _contentFilterEP = (ContentFilterExtensionPoint) manager.lookup(ContentFilterExtensionPoint.ROLE);
        _contentFilterHelper = (WebContentFilterHelper) manager.lookup(ContentFilterHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        
        _filterContentIdCache = new HashMap<>();
    }
    
    /**
     * Configure the engine (to be called by the scheduler or the action).
     * @param configuration the component configuration.
     * @throws ConfigurationException if an error occurs in the configuration.
     */
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        Configuration workflowConf = configuration.getChild("workflow");
        _workflowName = workflowConf.getAttribute("name");
        _wfInitialActionId = workflowConf.getAttributeAsInteger("initialActionId");
        
        String[] validateActionIds = StringUtils.split(workflowConf.getAttribute("validateActionIds"), ", ");
        _wfValidateActionIds = new ArrayList<>(validateActionIds.length);
        for (String actionId : validateActionIds)
        {
            try
            {
                _wfValidateActionIds.add(Integer.valueOf(actionId));
            }
            catch (NumberFormatException e)
            {
                throw new ConfigurationException("Invalid validation action ID.", e);
            }
        }
    }

    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        // Store the date and time.
        _runDate = new Date();
        
        // Reset the cache.
        _filterContentIdCache.clear(); 
        
        try (AmetysObjectIterable<Site> sites = _siteManager.getSites();)
        {
            for (Site site : sites)
            {
                try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps();)
                {
                    for (Sitemap sitemap : sitemaps)
                    {
                        createAutomaticNewsletters(site.getName(), sitemap.getName());
                    }
                }
            }
        }
    }
    
    /**
     * Test each category in a site and sitemap and launch the newsletter creation if needed.
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     */
    protected void createAutomaticNewsletters(String siteName, String sitemapName)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_INTERNAL_ALLOWED, true);
        request.setAttribute("siteName", siteName);
        
        for (String providerId : _categoryEP.getExtensionsIds())
        {
            CategoryProvider provider = _categoryEP.getExtension(providerId);
            
            // Browse all categories for this site and sitemap. 
            for (Category category : provider.getAllCategories(siteName, sitemapName))
            {
                // Get the automatic newsletter assigned to this category.
                Collection<String> automaticIds = provider.getAutomaticIds(category.getId());
                
                for (String autoNewsletterId : automaticIds)
                {
                    AutomaticNewsletter autoNewsletter = _autoNewsletterEP.getExtension(autoNewsletterId);
                    
                    // Test if an automatic newsletter content has to be created today.
                    if (autoNewsletter != null && createNow(autoNewsletter))
                    {
                        createAndValidateAutomaticNewsletter(siteName, sitemapName, category, provider, autoNewsletter);
                    }
                }
            }
            
        }
    }
    
    /**
     * Create an automatic newsletter content in a category.
     * @param sitemapName the sitemap name.
     * @param siteName the site name.
     * @param category the newsletter category.
     * @param provider the category provider.
     * @param autoNewsletter the associated automatic newsletter.
     */
    protected void createAndValidateAutomaticNewsletter(String siteName, String sitemapName, Category category, CategoryProvider provider, AutomaticNewsletter autoNewsletter)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Preparing to create an automatic newsletter for category " + category.getId() + " in " + siteName + " and sitemap " + sitemapName);
        }
        
        // Get the list of content IDs by filter name.
        Map<String, AutomaticNewsletterFilterResult> contentsByFilter = getFilterResults(siteName, sitemapName, autoNewsletter);
        
        try
        {
            if (hasResults(contentsByFilter.values()))
            {
                // Compute the next newsletter number in this category.
                long nextNumber = getNextNumber(category, provider, siteName, sitemapName);
                
                // Create newsletter content.
                WorkflowAwareContent content = createNewsletterContent(siteName, sitemapName, category, autoNewsletter, nextNumber, contentsByFilter);
                
                // Validate and send.
                validateNewsletter(content);
            }
            else
            {
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("No content has been returned by the filters for the automatic newsletter in category " + category.getId() + " in site " + siteName + " and sitemap " + sitemapName + ": no newsletter has been created.");
                }
            }
        }
        catch (InvalidActionException | WorkflowException e)
        {
            getLogger().error("Unable to create and validate an automatic newsletter for category " + category.getId() + " in site " + siteName + " and sitemap " + sitemapName, e);
        }
    }
    
    /**
     * Get the list of contents for the automatic newsletter filters.
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     * @param autoNewsletter the automatic newsletter.
     * @return the results, indexed by filter name (in the auto newsletter).
     */
    protected Map<String, AutomaticNewsletterFilterResult> getFilterResults(String siteName, String sitemapName, AutomaticNewsletter autoNewsletter)
    {
        Map<String, AutomaticNewsletterFilterResult> contentsByFilter = new HashMap<>();
        
        Request request = ContextHelper.getRequest(_context);
        
        Map<String, String> filters = autoNewsletter.getFilters();
        
        for (String name : filters.keySet())
        {
            String filterId = filters.get(name);
            
            AutomaticNewsletterFilterResult result = new AutomaticNewsletterFilterResult();
            contentsByFilter.put(name, result);
            
            List<String> contentIds = new ArrayList<>();
            
            ContentFilter filter = _contentFilterEP.getExtension(filterId);
            
            if (filter != null && filter instanceof WebContentFilter)
            {
                WebContentFilter webFilter = (WebContentFilter) filter;
                result.setViewName(filter.getView());
                
                String cacheKey = siteName + "/" + sitemapName + "/" + filterId;
                
                if (_filterContentIdCache.containsKey(cacheKey))
                {
                    contentIds = _filterContentIdCache.get(cacheKey);
                }
                else
                {
                    // Get the contents in the live workspace.
                    String originalWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
                    RequestAttributeWorkspaceSelector.setForcedWorkspace(request, WebConstants.LIVE_WORKSPACE);
                    
                    contentIds = _contentFilterHelper.getMatchingContentIds(webFilter, siteName, sitemapName, null);
                    
                    // Set the workspace back to its original value.
                    RequestAttributeWorkspaceSelector.setForcedWorkspace(request, originalWorkspace);
                    
                    // Cache the results.
                    _filterContentIdCache.put(cacheKey, contentIds);
                }
            }
            
            result.setContentIds(contentIds);
        }
        
        return contentsByFilter;
    }
    
    /**
     * Create the newsletter content.
     * @param siteName the site name.
     * @param language the language.
     * @param category the category.
     * @param autoNewsletter the automatic newsletter.
     * @param newsletterNumber the newsletter number.
     * @param filterResults the filter results (content IDs for each filter).
     * @return The newly created newsletter content.
     * @throws WorkflowException if a workflow error occurs.
     */
    protected WorkflowAwareContent createNewsletterContent(String siteName, String language, Category category, AutomaticNewsletter autoNewsletter, long newsletterNumber, Map<String, AutomaticNewsletterFilterResult> filterResults) throws WorkflowException
    {
        String contentName = category.getName() + "-" + newsletterNumber;
        
        String title = getNewsletterTitle(language, category, autoNewsletter, newsletterNumber);
        
        Map<String, Object> params = new HashMap<>();
        
        // Workflow result.
        Map<String, Object> workflowResult = new HashMap<>();
        params.put(AbstractWorkflowComponent.RESULT_MAP_KEY, workflowResult);
        
        // Workflow parameters.
        params.put("workflowName", _workflowName);
        params.put(org.ametys.web.workflow.CreateContentFunction.SITE_KEY, siteName);
        params.put(CreateContentFunction.CONTENT_NAME_KEY, contentName);
        params.put(CreateContentFunction.CONTENT_TITLE_KEY, title);
        params.put(CreateContentFunction.CONTENT_TYPES_KEY, new String[]{_NEWSLETTER_CONTENT_TYPE});
        params.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, language);
        params.put(CreateNewsletterFunction.NEWSLETTER_CATEGORY_KEY, category.getId());
        params.put(CreateNewsletterFunction.NEWSLETTER_NUMBER_KEY, Long.valueOf(newsletterNumber));
        params.put(CreateNewsletterFunction.NEWSLETTER_DATE_KEY, _runDate);
        params.put(CreateNewsletterFunction.NEWSLETTER_IS_AUTOMATIC_KEY, "true");
        params.put(CreateNewsletterFunction.NEWSLETTER_PROCESS_AUTO_SECTIONS_KEY, "true");
        params.put(CreateNewsletterFunction.NEWSLETTER_CONTENT_ID_MAP_KEY, filterResults);
        
        // Trigger the creation.
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
        workflow.initialize(_workflowName, _wfInitialActionId, params);
        
        // Get the content in the results and return it.
        WorkflowAwareContent content = (WorkflowAwareContent) workflowResult.get(AbstractContentWorkflowComponent.CONTENT_KEY);
        
        return content;
    }
    
    /**
     * Validate the newly created newsletter.
     * @param newsletterContent the newsletter content, must be in draft state.
     * @throws WorkflowException if a workflow error occurs.
     */
    protected void validateNewsletter(WorkflowAwareContent newsletterContent) throws WorkflowException
    {
        long workflowId = newsletterContent.getWorkflowId();
        
        Map<String, Object> inputs = new HashMap<>();
        
        inputs.put(AbstractContentWorkflowComponent.CONTENT_KEY, newsletterContent);
        // Do not send workflow mail notifications.
        inputs.put(SendMailFunction.SEND_MAIL, "false");
        
        inputs.put(CheckRightsCondition.FORCE, true);
        
        // Without this attribute, the newsletter is not sent to subscribers.
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute("send", "true");
        
        // Successively execute all the configured actions.
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(newsletterContent);
        for (Integer actionId : _wfValidateActionIds)
        {
            workflow.doAction(workflowId, actionId, inputs);
        }
    }
    
    /**
     * Compute the newsletter title.
     * @param language the language.
     * @param category the newsletter category.
     * @param autoNewsletter the automatic newsletter.
     * @param newsletterNumber the newsletter number.
     * @return the newsletter title.
     */
    protected String getNewsletterTitle(String language, Category category, AutomaticNewsletter autoNewsletter, long newsletterNumber)
    {
        String title = "";
        
        I18nizableText newsletterTitle = autoNewsletter.getNewsletterTitle();
        if (newsletterTitle == null || StringUtils.isEmpty(newsletterTitle.toString()))
        {
            // The newsletter title is not set in the auto newsletter:
            // create the newsletter title from the category title.
            String categoryTitle = _i18nUtils.translate(category.getTitle(), language);
            title = categoryTitle + " " + newsletterNumber;
        }
        else if (newsletterTitle.isI18n())
        {
            // The newsletter title is set as a parametrizable I18nizableText in the auto newsletter.
            Map<String, I18nizableTextParameter> params = Collections.singletonMap("number", new I18nizableText(Long.toString(newsletterNumber)));
            I18nizableText titleI18n = new I18nizableText(newsletterTitle.getCatalogue(), newsletterTitle.getKey(), params);
            title = _i18nUtils.translate(titleI18n, language);
        }
        else
        {
            // The newsletter title is set as a non-i18n I18nizableText.
            title = newsletterTitle.getLabel();
            if (title.contains("{number}"))
            {
                title = title.replaceAll("\\{number\\}", String.valueOf(newsletterNumber));
            }
            else
            {
                title += " " + newsletterNumber;
            }
        }
        
        return title;
    }
    
    /**
     * Compute the newsletter number.
     * @param category the newsletter category.
     * @param provider the category provider.
     * @param siteName the site name.
     * @param language the language.
     * @return the newsletter number.
     */
    protected long getNextNumber(Category category, CategoryProvider provider, String siteName, String language)
    {
        long number = 0;
        
        // Browse all existing numbers to get the highest number.
        try (AmetysObjectIterable<Content> newsletters = provider.getNewsletters(category.getId(), siteName, language);)
        {
            for (Content newsletterContent : newsletters)
            {
                long contentNumber = newsletterContent.getValue("newsletter-number", false, 0L);
                
                // Keep the number if it's higher.
                number = Math.max(number, contentNumber);
            }
            
            // Return the next newsletter number.
            return number + 1;
        }
    }
    
    /**
     * Test if there is at least one content in a collection of filter results.
     * @param results a collection of filter results.
     * @return true if at least one filter yielded a result, false otherwise.
     */
    protected boolean hasResults(Collection<AutomaticNewsletterFilterResult> results)
    {
        boolean hasResults = false;
        
        for (AutomaticNewsletterFilterResult result : results)
        {
            if (result.hasResults())
            {
                hasResults = true;
            }
        }
        
        return hasResults;
    }
    
    /**
     * Test if an automatic newsletter content has to be created now.
     * @param autoNewsletter the automatic newsletter.
     * @return true if an automatic newsletter content has to be created now, false otherwise.
     */
    protected boolean createNow(AutomaticNewsletter autoNewsletter)
    {
        boolean createToday = false;
        
        // The time the engine was launched.
        ZonedDateTime runDate = _runDate.toInstant().atZone(ZoneId.systemDefault());
        
        // The days at which the newsletter has to be created.
        Collection<Integer> dayNumbers = autoNewsletter.getDayNumbers();
        
        switch (autoNewsletter.getFrequencyType())
        {
            case MONTH:
                // Test with a month frequency.
                createToday = testMonth(dayNumbers, runDate);
                break;
            case WEEK:
                // Test with a week frequency.
                createToday = testWeek(dayNumbers, runDate);
                break;
            default:
                break;
        }
        
        return createToday;
    }
    
    /**
     * Test if we are in the configured month creation period.
     * @param dayNumbers the days in the month on which a newsletter is to be created.
     * @param runDate the instant the engine was launched.
     * @return true if we are in the configured month creation period, false otherwise.
     */
    protected boolean testMonth(Collection<Integer> dayNumbers, ZonedDateTime runDate)
    {
        boolean createToday = false;
        
        int dayOfMonth = runDate.getDayOfMonth();
        int lastDayOfMonth = runDate.with(TemporalAdjusters.lastDayOfMonth()).getDayOfMonth();
        
        for (Integer dayNumber : dayNumbers)
        {
            if (dayNumber.intValue() == dayOfMonth)
            {
                createToday = true;
            }
            else if (dayNumber.intValue() > lastDayOfMonth && dayOfMonth == lastDayOfMonth)
            {
                // If the configured day is outside the current month (for instance, if "31" is configured),
                // run the last day of the current month.
                createToday = true;
            }
        }
        
        return createToday;
    }
    
    /**
     * Test if we are in the configured week creation period.
     * @param dayNumbers the days in the month on which a newsletter is to be created.
     * @param runDate the instant the engine was launched.
     * @return true if we are in the configured week creation period, false otherwise.
     */
    protected boolean testWeek(Collection<Integer> dayNumbers, ZonedDateTime runDate)
    {
        boolean createToday = false;
        
        int dayOfWeek = runDate.getDayOfWeek().getValue();
        
        for (Integer dayNumber : dayNumbers)
        {
            if (dayNumber.intValue() == dayOfWeek)
            {
                createToday = true;
            }
        }
        
        return createToday;
    }
    
}
