/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.Collection;
import java.util.List;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This interface represents a category provider. <br>
 */
public interface CategoryProvider
{
    /**
     * Determines if the categories can be modified
     * @return true if the categories can be modified
     */
    public boolean isWritable ();
    /**
     * Returns the provider's id.
     * @return the provider's id.
     */
    public String getId();
    
    /**
     * Returns the provider's label.
     * @return the provider's label.
     */
    public I18nizableText getLabel();
    
    /**
     * Returns the provider's description.
     * @return the provider's description.
     */
    public I18nizableText getDescription ();
    
    /**
     * Returns the provider's root categories.
     * @param siteName The site name. Can be null for all sites
     * @param lang The languages. Can be null for all languages
     * @return the provider's root categories.
     */
    public List<Category> getCategories (String siteName, String lang);
    
    /**
     * Returns all the provider's categories, recursively.
     * @param siteName The site name.
     * @param lang The languages.
     * @return all the provider's categories.
     */
    public Collection<Category> getAllCategories(String siteName, String lang);
    
    /**
     * Returns the provider's category.
     * @param categoryID The category id
     * @return the provider's category.
     */
    public Category getCategory(String categoryID);
    
    
    /**
     * Determines if the category exists.
     * @param categoryID The category id
     * @return true if the category exists.
     */
    public boolean hasCategory (String categoryID);
    
    /**
     * Determines if the category has sub-categories
     * @param categoryID The category id
     * @return true if the category have sub-categories
     */
    public boolean hasChildren (String categoryID);
    
    /**
     * Get the sub-categories of a category
     * @param categoryID The category id
     * @return the sub-categories
     */
    public List<Category> getCategories (String categoryID);
    
    /**
     * Determines if a category has categories linked with
     * @param categoryID The category id
     * @param siteName The site name
     * @param lang The language name
     * @return if a category has categories linked with
     */
    public boolean hasNewsletters (String categoryID, String siteName, String lang);
    
    /**
     * Get the newsletter contents linked to a category
     * @param categoryID The category id
     * @param siteName The site name
     * @param lang The language name
     * @return the newsletter contents
     */
    public AmetysObjectIterable<Content> getNewsletters (String categoryID, String siteName, String lang);
    
    /**
     * Get the root id
     * @param siteName The site name
     * @param lang The language name
     * @return the root id
     */
    public String getRootId (String siteName, String lang);
    
    /**
     * Affect a template to category
     * @param category The category  
     * @param templateName The template name
     */
    public void setTemplate (Category category, String templateName);
    
    /**
     * Get automatic property.
     * @param categoryId The category id.
     * @return the list of automatic newsletter IDs.
     */
    public Collection<String> getAutomaticIds(String categoryId);
    
    /**
     * Set automatic property.
     * @param categoryId The category id.
     * @param automaticNewsletterIds The automatic newsletter IDs or empty to set non-automatic.
     */
    public void setAutomatic(String categoryId, Collection<String> automaticNewsletterIds);
    
}
