/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.site.SiteManager;

/**
 * Class representing a JCR categories provider
 */
public class JCRCategoryProvider implements LogEnabled, CategoryProvider, Serviceable, Configurable, PluginAware
{
    /** The id */
    protected String _id;
    /** The label */
    protected I18nizableText _label;
    /** The description */
    protected I18nizableText _description;
    /** The plugin name */
    protected String _pluginName;
    /** The feature name */
    protected String _featureName;
    /** The Logger */
    protected Logger _logger;
    
    private AmetysObjectResolver _resolver;
    private SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = configureLabel(configuration);
        _description = configureDescription(configuration);
    }
    
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public boolean isWritable()
    {
        return true;
    }
    
    @Override
    public List<Category> getCategories(String siteName, String lang)
    {
        if (StringUtils.isEmpty(siteName))
        {
            return null;
        }
        
        List<Category> categories = new ArrayList<>();
        
        try
        {
            DefaultTraversableAmetysObject rootNode = (DefaultTraversableAmetysObject) _getRootNode(siteName, lang);
            
            AmetysObjectIterable<AmetysObject> it = rootNode.getChildren();
            for (AmetysObject object : it)
            {
                if (object instanceof JCRCategory)
                {
                    JCRCategory jcrCategory = (JCRCategory) object;
                    Category category = new Category(jcrCategory.getId(), jcrCategory.getName(), rootNode.getId(), new I18nizableText(jcrCategory.getTitle()), new I18nizableText(jcrCategory.getDescription()), jcrCategory.getTemplate(), siteName, lang);
                    
                    categories.add(category);
                }
            }
        }
        catch (RepositoryException e)
        {
            _logger.error("Unable to list the newsletter categories for site " + siteName + " and language " + lang, e);
        }
        
        return categories;
    }
    
    @Override
    public Collection<Category> getAllCategories(String siteName, String lang)
    {
        if (StringUtils.isEmpty(siteName))
        {
            return null;
        }
        
        Set<Category> categories = new HashSet<>();
        
        try
        {
            TraversableAmetysObject rootNode = _getRootNode(siteName, lang);
            
            // Recursively add categories from the root node.
            addCategories(categories, rootNode, siteName, lang);
        }
        catch (RepositoryException e)
        {
            _logger.error("Unable to list the newsletter categories for site " + siteName + " and language " + lang, e);
        }
        
        return categories;
    }

    /**
     * Recursively find {@link JCRCategory} objects in the given TraversableAmetysObject.
     * @param categories the Set to fill with categories.
     * @param parentObject the parent {@link TraversableAmetysObject}.
     * @param siteName the site name.
     * @param lang the language.
     */
    protected void addCategories(Set<Category> categories, TraversableAmetysObject parentObject, String siteName, String lang)
    {
        AmetysObjectIterable<AmetysObject> it = parentObject.getChildren();
        for (AmetysObject object : it)
        {
            if (object instanceof JCRCategory)
            {
                JCRCategory jcrCategory = (JCRCategory) object;
                Category category = new Category(jcrCategory.getId(), jcrCategory.getName(), parentObject.getId(), new I18nizableText(jcrCategory.getTitle()), new I18nizableText(jcrCategory.getDescription()), jcrCategory.getTemplate(), siteName, lang);
                
                categories.add(category);
                
                // 
                addCategories(categories, jcrCategory, siteName, lang);
            }
        }
    }
    
    @Override
    public Category getCategory(String categoryID)
    {
        try
        {
            JCRCategory category = _resolver.resolveById(categoryID);
            
            AmetysObject parent = category.getParent();
            String parentId = parent.getId();
            if (!(parent instanceof JCRCategory))
            {
                parentId = "provider_" + getId();
            }
            
            return new Category(category.getId(), category.getName(), parentId, new I18nizableText(category.getTitle()), new I18nizableText(category.getDescription()), category.getTemplate(), category.getSiteName(), category.getLang());
        }
        catch (UnknownAmetysObjectException e) 
        {
            _logger.warn("Unable to retrieve newsletter category of id " + categoryID, e);
            return null;
        }
    }
    
    @Override
    public void setTemplate(Category category, String templateName)
    {
        JCRCategory jcrCategory = _resolver.resolveById(category.getId());
        jcrCategory.setTemplate(templateName);
        jcrCategory.saveChanges();
    }
    
    @Override
    public Collection<String> getAutomaticIds(String categoryId)
    {
        JCRCategory jcrCategory = _resolver.resolveById(categoryId);
        return jcrCategory.getAutomaticIds();
    }
    
    @Override
    public void setAutomatic(String categoryId, Collection<String> automaticNewsletterIds)
    {
        JCRCategory jcrCategory = _resolver.resolveById(categoryId);
        jcrCategory.setAutomaticIds(automaticNewsletterIds);
        jcrCategory.saveChanges();
    }
    
    @Override
    public boolean hasCategory(String categoryID)
    {
        try
        {
            AmetysObject object = _resolver.resolveById(categoryID);
            return object instanceof JCRCategory;
        }
        catch (UnknownAmetysObjectException e)
        {
            _logger.debug("Unable to retrieve newsletter category of id " + categoryID, e);
            return false;
        }
        catch (AmetysRepositoryException e) 
        {
            _logger.debug("Unable to retrieve newsletter category of id " + categoryID, e);
            return false;
        }
    }
    
    @Override
    public List<Category> getCategories(String categoryID)
    {
        List<Category> categories = new ArrayList<>();
        
        try
        {
            JCRCategory category = _resolver.resolveById(categoryID);
            AmetysObjectIterable<AmetysObject> it = category.getChildren();
            for (AmetysObject object : it)
            {
                if (object instanceof JCRCategory)
                {
                    JCRCategory jcrCategory = (JCRCategory) object;
                    Category child = new Category(jcrCategory.getId(), jcrCategory.getName(), category.getId(), new I18nizableText(jcrCategory.getTitle()), new I18nizableText(jcrCategory.getDescription()), jcrCategory.getTemplate(), jcrCategory.getSiteName(), jcrCategory.getLang());
                    
                    categories.add(child);
                }
            }
        }
        catch (AmetysRepositoryException e)
        {
            _logger.error("Unable to list the newsletter categories for category of id " + categoryID, e);
        }
        
        
        return categories;
    }
    
    
    @Override
    public boolean hasChildren(String categoryID)
    {
        try
        {
            JCRCategory category = _resolver.resolveById(categoryID);
            return category.getChildren().iterator().hasNext();
        }
        catch (AmetysRepositoryException e)
        {
            _logger.error("Unable to retrieve newsletter category of id " + categoryID, e);
            return false;
        }
        
    }
    
    @Override
    public AmetysObjectIterable<Content> getNewsletters(String categoryID, String siteName, String lang)
    {
        ExpressionContext expressionContextForInternal = ExpressionContext.newInstance().withInternal(true);
        Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, "org.ametys.plugins.newsletter.Content.newsletter");
        Expression siteExpr = new StringExpression("site", Operator.EQ, siteName, expressionContextForInternal);
        Expression catExpr = new StringExpression("category", Operator.EQ, categoryID, expressionContextForInternal);
        Expression expr = new AndExpression(cTypeExpr, siteExpr, catExpr);
        
        String xpathQuery = org.ametys.plugins.repository.query.QueryHelper.getXPathQuery(null, "ametys:content", expr);
        return _resolver.query(xpathQuery);
    }
    
    @Override
    public boolean hasNewsletters(String categoryID, String siteName, String lang)
    {
        ExpressionContext expressionContextForInternal = ExpressionContext.newInstance().withInternal(true);
        Expression expr = new ContentTypeExpression(Operator.EQ, "org.ametys.plugins.newsletter.Content.newsletter");
        Expression siteExpr = new StringExpression("site", Operator.EQ, siteName, expressionContextForInternal);
        expr = new AndExpression(expr, siteExpr);
        Expression catExpr = new StringExpression("category", Operator.EQ, categoryID, expressionContextForInternal);
        expr = new AndExpression(expr, catExpr);
        
        String xpathQuery = org.ametys.plugins.repository.query.QueryHelper.getXPathQuery(null, "ametys:content", expr);
        
        return _resolver.query(xpathQuery).iterator().hasNext();
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }

    @Override
    public String getId()
    {
        return _id;
    }
    
    /**
     * Configure label from the passed configuration
     * @param configuration The configuration
     * @return The label
     * @throws ConfigurationException If an error occurred
     */
    protected I18nizableText configureLabel (Configuration configuration) throws ConfigurationException
    {
        Configuration labelConfiguration = configuration.getChild("label");
        
        if (labelConfiguration.getAttributeAsBoolean("i18n", false))
        {
            return new I18nizableText("plugin." + _pluginName, labelConfiguration.getValue(""));
        }
        else
        {
            return new I18nizableText(labelConfiguration.getValue(""));
        }
    }
    
    /**
     * Configure description from the passed configuration
     * @param configuration The configuration
     * @return The description
     * @throws ConfigurationException If an error occurred
     */
    protected I18nizableText configureDescription (Configuration configuration) throws ConfigurationException
    {
        Configuration descConfiguration = configuration.getChild("description");
        
        if (descConfiguration.getAttributeAsBoolean("i18n", false))
        {
            return new I18nizableText("plugin." + _pluginName, descConfiguration.getValue(""));
        }
        else
        {
            return new I18nizableText(descConfiguration.getValue(""));
        }
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _featureName = featureName;
        _id = id;
    }
    
    /**
     * Get the plugin name
     * @return the plugin name
     */
    public String getPluginName()
    {
        return _pluginName;
    }
    
    @Override
    public String getRootId(String siteName, String lang)
    {
        try
        {
            return _getRootNode(siteName, lang).getId();
        }
        catch (RepositoryException e) 
        {
            _logger.error("Unable to retrieve the root node of newsletter categories for site " + siteName + " and language " + lang, e);
            return null;
        }
    }
    
    
    private TraversableAmetysObject _getRootNode (String sitename, String lang) throws RepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(sitename).getRootPlugins();
        
        ModifiableTraversableAmetysObject categoriesNode = null;
        if (!pluginsNode.hasChild("newsletter"))
        {
            categoriesNode = ((ModifiableTraversableAmetysObject) pluginsNode.createChild("newsletter", "ametys:unstructured")).createChild("ametys:categories", "ametys:unstructured");
        }
        else
        {
            categoriesNode = pluginsNode.getChild("newsletter/ametys:categories");
        }
        
        if (!categoriesNode.hasChild(lang))
        {
            categoriesNode.createChild(lang, "ametys:unstructured");
            ((JCRAmetysObject) pluginsNode).getNode().getSession().save();
        }
        
        return pluginsNode.getChild("newsletter/ametys:categories/" + lang);
    }

}
