/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.generators;

import java.io.IOException;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryGenerator;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.web.WebConstants;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItemException;
import org.ametys.web.site.SiteConfigurationExtensionPoint;

/**
 * Newsletter service
 *
 */
public class NewsletterServiceGenerator extends CategoryGenerator
{
    /** The site configuration */
    protected SiteConfigurationExtensionPoint _siteConfiguration;
    /** The user manager */
    protected UserManager _userManager;
    /** The user helper */
    protected UserHelper _userHelper;
    /** Page helper */
    protected PageHelper _pageHelper;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _siteConfiguration = (SiteConfigurationExtensionPoint) smanager.lookup(SiteConfigurationExtensionPoint.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _pageHelper = (PageHelper) smanager.lookup(PageHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "NewsletterService");
        
        if (zoneItem != null)
        {
            XMLUtils.startElement(contentHandler, "Categories");
            String[] categoryIDs = zoneItem.getServiceParameters().getValue("categories", false, "").split(",");
            Set<String> ids = _categoryProviderEP.getExtensionsIds();
            
            for (String categoryID : categoryIDs)
            {
                boolean found = false;
                
                for (String id : ids)
                {
                    CategoryProvider provider = _categoryProviderEP.getExtension(id);
                    if (provider.hasCategory(categoryID))
                    {
                        Category category = provider.getCategory(categoryID);
                        saxCategory (provider, category, true, false);
                        found = true;
                        break;
                    }
                }
                
                if (!found)
                {
                    throw new ZoneItemException("The newsletter category with id '" + categoryID + "' was not found");
                }
            }
            XMLUtils.endElement(contentHandler, "Categories");
        }
        
        XMLUtils.startElement(contentHandler, "FormResult");
        // SAX message
        String msg = parameters.getParameter("msg", "");
        if (msg.length() != 0)
        {
            XMLUtils.createElement(contentHandler, "msg", msg);
        }
        
        String email = parameters.getParameter("email", "");
        if (email.length() != 0)
        {
            XMLUtils.createElement(contentHandler, "email", email);
        }
        
        String[] ids = parameters.getParameter("subscribeTo", "").split(",");
        for (String id : ids)
        {
            if (id.length() > 0)
            {
                XMLUtils.createElement(contentHandler, "subscribeTo", id);
            }
        }
        
        ids = parameters.getParameter("alreadySubscribeTo", "").split(",");
        for (String id : ids)
        {
            if (id.length() > 0)
            {
                XMLUtils.createElement(contentHandler, "alreadySubscribeTo", id);
            }
            
        }
        
        Page page = (Page) request.getAttribute(Page.class.getName());
        XMLUtils.createElement(contentHandler, "has-captcha", String.valueOf(_pageHelper.isCaptchaRequired(page)));
        
        XMLUtils.endElement(contentHandler, "FormResult");
        
        _saxInfoUser();
        
        XMLUtils.endElement(contentHandler, "NewsletterService");
        contentHandler.endDocument();
    }
    
    private void _saxInfoUser() throws SAXException
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        if (userIdentity != null)
        {
            User user = _userManager.getUser(userIdentity);
            _userHelper.saxUser(user, contentHandler);
        }
    }
    
    
}
