/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.newsletter.daos.SubscribersDAO.UnsubscribeOrigin;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Unsubscribes user thanks to his unique token
 * 
 */
public class UnsubscribeAction extends ServiceableAction
{
    private SubscribersDAO _subscribersDao;
    private CategoryProviderExtensionPoint _categoryProviderEP;
    private SiteManager _siteManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscribersDao = (SubscribersDAO) smanager.lookup(SubscribersDAO.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();

        Request request = ObjectModelHelper.getRequest(objectModel);

        boolean unsubscribe = request.getParameter("unsubscribe") != null;
        String token = request.getParameter("token");
        result.put("token", "token");
        
        if (unsubscribe)
        {
            _subscribersDao.unsubscribe(token, UnsubscribeOrigin.SUBSCRIBER);
            String categoryID = request.getParameter("categoryId");
            Category category = _getCategory(categoryID);
            if (category != null)
            {
                result.put("categoryId", category.getId());
                result.put("categoryTitle", category.getTitle().getLabel());
            }
            result.put("unsubcribe", "true");
        }
        else
        {
            Subscriber subscriber = _subscribersDao.getSubscriberByToken(token);
            if (subscriber != null)
            {
                String email = subscriber.getEmail();
                result.put("email", email);

                String sitename = subscriber.getSiteName();
                Site site = _siteManager.getSite(sitename);
                if (site != null)
                {
                    result.put("siteTitle", site.getTitle());
                }

                String categoryID = subscriber.getCategoryId();
                Category category = _getCategory(categoryID);
                if (category != null)
                {
                    result.put("categoryId", category.getId());
                    result.put("categoryTitle", category.getTitle().getLabel());
                }
                else
                {
                    result.put("unknown-category", "true");
                }
            }
            else
            {
                result.put("unknown-subscriber", "true");
            }
        }

        return result;
    }
    
    private Category _getCategory (String categoryID)
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        for (String id : ids)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            if (provider.hasCategory(categoryID))
            {
                return provider.getCategory(categoryID);
            }
        }
        
        return null;
    }

}
