/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.userpref;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesStorage;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.newsletter.daos.SubscribersDAO.UnsubscribeOrigin;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Retrieves and stores newsletter user preferences values as subscriptions.
 */
public class NewsletterUserPreferencesStorage extends AbstractLogEnabled implements UserPreferencesStorage, Component, Serviceable
{
    /** The front-office users manager. */
    protected UserManager _foUserManager;
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryEP;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The subscribers DAO. */
    protected SubscribersDAO _subscribersDao;

    /** The population context helper */
    protected PopulationContextHelper _populationContextHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _foUserManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _categoryEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
        _populationContextHelper = (PopulationContextHelper) serviceManager.lookup(PopulationContextHelper.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _subscribersDao = (SubscribersDAO) serviceManager.lookup(SubscribersDAO.ROLE);
    }
    
    public Map<UserIdentity, Map<String, String>> getAllUnTypedUserPrefs(String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        return _getUnTypedUserPrefs(null, storageContext, contextVars);
    }
    
    @Override
    public Map<String, String> getUnTypedUserPrefs(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        if (userIdentity == null)
        {
            return new HashMap<>();
        }
        
        Map<UserIdentity, Map<String, String>> unTypedUserPrefs = _getUnTypedUserPrefs(userIdentity, storageContext, contextVars);
        return unTypedUserPrefs.getOrDefault(userIdentity, new HashMap<>());
    }
    
    private Map<UserIdentity, Map<String, String>> _getUnTypedUserPrefs(UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        Map<UserIdentity, Map<String, String>> userPrefs = new HashMap<>();

        try
        {
            String siteName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME);
            if (siteName != null)
            {
                for (User u : _getUsers(siteName, user))
                {
                    Map<String, String> preferenceValues = new HashMap<>();
                    
                    List<Subscriber> subscriptions = _subscribersDao.getSubscriptions(u.getEmail(), siteName);
                    for (Subscriber subscription : subscriptions)
                    {
                        preferenceValues.put(subscription.getCategoryId(), "true");
                    }
                    
                    userPrefs.put(u.getIdentity(), preferenceValues);
                }
            }
            return userPrefs;
        }
        catch (Exception e)
        {
            String message = user == null 
                ? "Error getting newsletter all user's preferences in context " + storageContext
                : "Error getting newsletter user preferences for login " + user + " and context " + storageContext;
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
    }
    
    private List<User> _getUsers(String siteName, UserIdentity userIdentity)
    {
        if (userIdentity != null)
        {
            User user = _foUserManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
            return user != null ? List.of(user) : List.of();
        }
        
        Set<String> userPopulationsOnSite = _populationContextHelper.getUserPopulationsOnContexts(Arrays.asList("/sites/" + siteName, "/sites-fo/" + siteName), false, false);
        return _subscribersDao.getSubscribers()
                    .stream()
                    .map(s -> {
                        try
                        {
                            return _foUserManager.getUserByEmail(userPopulationsOnSite, s.getEmail());
                        }
                        catch (NotUniqueUserException e)
                        {
                            getLogger().warn("Can get user preference fo user with mail '" + s.getEmail() + "' because several users is found", e);
                            return null;
                        }
                    })
                    .filter(Objects::nonNull)
                    .toList();
    }
    
    @Override
    public void setUserPreferences(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, Map<String, String> preferences) throws UserPreferencesException
    {
        try
        {
            User user = _foUserManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
            String siteName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME);
            
            if (user != null && siteName != null)
            {
                List<Subscriber> newSubscribers = new ArrayList<>();
                Set<String> removeTokens = new HashSet<>();
                
                Map<String, String> existingCategoryIds = getExistingCategoryIds(user.getEmail(), siteName);
                
                for (String categoryId : preferences.keySet())
                {
                    String value = preferences.get(categoryId);
                    
                    if (value.equals("true") && _categoryEP.hasCategory(categoryId) && !existingCategoryIds.containsKey(categoryId))
                    {
                        Subscriber subscriber = getSubscription(siteName, user, categoryId);
                        newSubscribers.add(subscriber);
                    }
                    else if (!value.equals("true") && _categoryEP.hasCategory(categoryId) && existingCategoryIds.containsKey(categoryId))
                    {
                        String token = existingCategoryIds.get(categoryId);
                        removeTokens.add(token);
                    }
                }
                
                // Modify the subscriptions.
                _subscribersDao.modifySubscriptions(newSubscribers, removeTokens, UnsubscribeOrigin.SUBSCRIBER);
            }
        }
        catch (Exception e)
        {
            String message = "Error setting newsletter user preferences for login " + userIdentity + " and context " + storageContext;
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
    }
    
    @Override
    public void removeUserPreferences(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        try
        {
            User user = _foUserManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
            
            if (user != null)
            {
                String email = user.getEmail();
                if (email != null)
                {
                    String siteName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME);
                    if (siteName != null)
                    {
                        // Remove the subscriptions.
                        _subscribersDao.unsubscribe(email, storageContext, UnsubscribeOrigin.DATAPOLICY);
                    }
                    else
                    {
                        try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
                        {
                            for (Site site : sites)
                            {
                                // Before unsubscribing, check that no other user use the same email on the site
                                siteName = site.getName();
                                Set<String> userPopulationsOnSite = _populationContextHelper.getUserPopulationsOnContexts(Arrays.asList("/sites/" + siteName, "/sites-fo/" + siteName), false, false);
                                try
                                {
                                    if (_foUserManager.getUserByEmail(userPopulationsOnSite, email) == null)
                                    {
                                        _subscribersDao.unsubscribe(email, siteName, UnsubscribeOrigin.DATAPOLICY);
                                    }
                                }
                                catch (NotUniqueUserException e)
                                {
                                    // Do nothing, this email is still matching existing users
                                }
                            }
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            String message = "Error removing newsletter subscriptions for login " + userIdentity + " and context " + storageContext;
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
    }
    
    @Override
    public String getUserPreferenceAsString(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        String value = null;
        
        Boolean booleanValue = getUserPreferenceAsBoolean(userIdentity, storageContext, contextVars, id);
        
        if (booleanValue != null)
        {
            value = booleanValue.toString();
        }
        
        return value;
    }
    
    @Override
    public Long getUserPreferenceAsLong(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }
    
    @Override
    public ZonedDateTime getUserPreferenceAsDate(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }
    
    @Override
    public Boolean getUserPreferenceAsBoolean(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        try
        {
            Boolean value = null;
            
            User user = _foUserManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
            String siteName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME);
            
            if (user != null && siteName != null)
            {
                Subscriber subscriber = _subscribersDao.getSubscriber(user.getEmail(), siteName, id);
                
                value = Boolean.valueOf(subscriber != null);
            }
            
            return value;
        }
        catch (Exception e)
        {
            throw new UserPreferencesException("Error getting newsletter user preferences for login " + userIdentity + " and context " + storageContext, e);
        }
    }
    
    @Override
    public Double getUserPreferenceAsDouble(UserIdentity userIdentity, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }
    
    /**
     * Create a subscriber object from the given input.
     * @param siteName the site name.
     * @param user the user.
     * @param categoryId the category ID.
     * @return the Subscriber object.
     */
    protected Subscriber getSubscription(String siteName, User user, String categoryId)
    {
        return getSubscription(siteName, user, categoryId, true);
    }
    
    /**
     * Create a subscriber object from the given input.
     * @param siteName the site name.
     * @param user the user.
     * @param categoryId the category ID.
     * @param generateDateAndToken true to generate a token and set the subscription date, false otherwise.
     * @return the Subscriber object.
     */
    protected Subscriber getSubscription(String siteName, User user, String categoryId, boolean generateDateAndToken)
    {
        Subscriber subscriber = new Subscriber();
        subscriber.setEmail(user.getEmail());
        subscriber.setSiteName(siteName);
        subscriber.setCategoryId(categoryId);
        
        if (generateDateAndToken)
        {
            subscriber.setSubscribedAt(new Date());
            
            // Generate unique token.
            String token = UUID.randomUUID().toString();
            subscriber.setToken(token);
        }
        
        return subscriber;
    }
    
    /**
     * Get the existing subscriptions for a user in a given site.
     * @param email the user e-mail address.
     * @param siteName the site name.
     * @return a Set of category IDs, to which user has subscribed.
     */
    protected Map<String, String> getExistingCategoryIds(String email, String siteName)
    {
        Map<String, String> existingCategoryIds = new HashMap<>();
        
        List<Subscriber> existingSubscriptions = _subscribersDao.getSubscriptions(email, siteName);
        for (Subscriber subscription : existingSubscriptions)
        {
            existingCategoryIds.put(subscription.getCategoryId(), subscription.getToken());
        }
        
        return existingCategoryIds;
    }
}
