/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.workflow;

import java.io.IOException;
import java.util.Date;
import java.util.Map;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.web.analytics.AbstractWebAnalyticsProvider;

import jakarta.mail.MessagingException;

/**
 * Sends mails in a thread
 */
public class SendMailEngine implements Runnable
{
    private static Logger _logger = LoggerFactory.getLogger(SendMailEngine.class);

    private static final Random _RNG = new Random((long) Math.random() * System.nanoTime());
    
    private String _subject;
    private String _htmlBody;
    private String _textBody;
    private Map<String, String> _recipients;
    private String _sender;
    
    private boolean _parametrized;
    
    /**
     * Parameterize engine
     * @param subject The mail subject
     * @param htmlBody The mail body in HTML format
     * @param textBody The mail body in text format
     * @param recipients The recipient addresses
     * @param sender The sender address
     */
    public void parameterize (String subject, String htmlBody, String textBody, Map<String, String> recipients, String sender)
    {
        _subject = subject;
        _htmlBody = SendMailHelper.inlineCSS(htmlBody);
        _textBody = textBody;
        _recipients = recipients;
        _sender = sender;
        
        _parametrized = true;
    }
    
    private void _checkParametrization()
    {
        if (!_parametrized)
        {
            String message = "The mail engine component has to be parameterized before it can be run.";
            _logger.error(message);
            throw new IllegalStateException(message);
        }
    }
    
    public void run()
    {
        _checkParametrization ();
        
        if (_logger.isInfoEnabled())
        {
            _logger.info("Try to send " + _recipients.size() + " mails at " + new Date());
        }

        for (String recipient : _recipients.keySet())
        {
            try
            {
                String htmlBody = _htmlBody.replaceAll("#token#", _recipients.get(recipient));
                String textBody = _textBody.replaceAll("#token#", _recipients.get(recipient));
                
                Pattern pattern = Pattern.compile("\\" + AbstractWebAnalyticsProvider.RANDOM_NUMBER_TOKEN);
                Matcher m = pattern.matcher(htmlBody);
                StringBuffer sb = new StringBuffer();
                while (m.find()) 
                {
                    m.appendReplacement(sb, Integer.toString(_RNG.nextInt(Integer.MAX_VALUE)));
                }
                m.appendTail(sb);

                SendMailHelper.newMail()
                              .withSubject(_subject)
                              .withHTMLBody(sb.toString())
                              .withInlineCSS(false) // We inline the CSS manually once and for all to avoid doing it multiple times
                              .withTextBody(textBody)
                              .withSender(_sender)
                              .withRecipient(recipient)
                              .sendMail();
                
                Thread.sleep(1000);
            }
            catch (MessagingException | IOException e)
            {
                _logger.error("Mail failed !", e);
            }
            catch (InterruptedException e)
            {
                _logger.error("Error while waiting for sending mails", e);
            }
        }
        
        if (_logger.isInfoEnabled())
        {
            _logger.info("All mails are sent at " + new Date());
        }
    }
}
