/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.export.OSEConstants;
import org.ametys.odf.ose.export.impl.odf.db.EtapeHelper;
import org.ametys.odf.ose.export.utils.ElementRetriever;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.EmptyIterable;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Exporter of steps
 */
public class StepExporter extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = StepExporter.class.getName();
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The retriever of elements from ODF */
    protected ElementRetriever _elementRetriever;
    /** The helper for ODF contents */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _elementRetriever = (ElementRetriever) manager.lookup(ElementRetriever.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Get all the {@link Container} that are steps (nature is "annee") and are exportable (attribute "noOseExport" is not <code>true</code>)
     * @param ametysCatalog The Ametys catalog
     * @return The list of exportable steps
     */
    public AmetysObjectIterable<Container> getStepsToExport(String ametysCatalog)
    {
        // Add values for steps (container with nature equals to year)
        return _odfHelper.getYearId()
            .map(anneeId -> _matchingStepsExpression(anneeId, ametysCatalog))
            .map(ContentQueryHelper::getContentXPathQuery)
            .map(_resolver::<Container>query)
            .orElseGet(EmptyIterable::new);
    }
    
    private Expression _matchingStepsExpression(String anneeId, String ametysCatalog)
    {
        Expression contentTypeExpr = new ContentTypeExpression(Operator.EQ, ContainerFactory.CONTAINER_CONTENT_TYPE);
        String lang = Config.getInstance().getValue("odf.programs.lang");
        Expression langExpr = new LanguageExpression(Operator.EQ, lang);
        Expression catalogExpr = new StringExpression(ProgramItem.CATALOG, Operator.EQ, ametysCatalog);
        Expression natureExpr = new StringExpression(Container.NATURE, Operator.EQ, anneeId);
        Expression doExportExpression = new NotExpression(new BooleanExpression(OSEConstants.NO_OSE_EXPORT_ATTRIBUTE_NAME, true));
        
        return new AndExpression(contentTypeExpr, langExpr, catalogExpr, natureExpr, doExportExpression);
    }

    /**
     * Get queries to export the given step.
     * @param step The step to export
     * @param oseCatalog The OSE catalog
     * @return A {@link List} of {@link ParameterizableQuery} to export the {@link Container}, it can be empty if there is a problem (see logs)
     */
    public List<ParameterizableQuery> getQueries(Container step, Long oseCatalog)
    {
        // Degree : Remonter aux formations parentes et prendre les degrees, s'il n'y en a qu'une prendre son code, sinon erreur
        Set<OdfReferenceTableEntry> degrees = _elementRetriever.retrieveDegree(step);
        
        // Structure : Remonter parent par parent pour obtenir le ou les structures rattachées (sur abstractProgram)
        Set<OrgUnit> orgUnits = _elementRetriever.retrieveOrgUnits(step);
        
        if (degrees.size() != 1 || orgUnits.size() != 1)
        {
            LogUtils.stepImpossibilityDegreesOrOrgunits(getLogger(), step, degrees, orgUnits);
            return List.of();
        }
        
        OdfReferenceTableEntry degree = _getFirstDegree(degrees);
        String degreeCode = degree.getCode();
        String orgUnitCode = _getFirstOrgUnitCode(orgUnits);
        Optional<String> functionalDomain = _getFunctionalDomain(degree);
        
        if (functionalDomain.isEmpty())
        {
            LogUtils.stepImpossibilityFunctDomain(getLogger(), step);
            return List.of();
        }
        
        return _getQueries(step, oseCatalog, degreeCode, orgUnitCode, functionalDomain.get());
    }
    
    private Optional<String> _getFunctionalDomain(OdfReferenceTableEntry degree)
    {
        Expression expr = new AndExpression(
            new ContentTypeExpression(Operator.EQ, "odf-enumeration.DegreeCategory"),
            new StringExpression("degrees", Operator.EQ, degree.getId())
        );
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        
        return contents
                    .stream()
                    .findFirst()
                    .map(c -> c.<String>getValue("functionalDomain"));
    }
    
    private OdfReferenceTableEntry _getFirstDegree(Set<OdfReferenceTableEntry> degrees)
    {
        return degrees.stream()
                .findFirst()
                .get();
    }
    
    private String _getFirstOrgUnitCode(Set<OrgUnit> orgUnits)
    {
        return orgUnits.stream()
                .findFirst()
                .map(OrgUnit::getUAICode)
                .get();
    }
    
    private List<ParameterizableQuery> _getQueries(Container container, Long oseCatalog, String degree, String orgUnit, String functionalDomain)
    {
        String title = container.getTitle();
        if (ModelHelper.hasModelItem("etpCode", container.getModel()))
        {
            String apogee = container.getValue("etpCode");
            if (StringUtils.isNotEmpty(apogee))
            {
                title += " (" + apogee + ")";
            }
        }
        String code = container.getCode();
        
        return EtapeHelper.insertInto(code, title, oseCatalog, degree, orgUnit, functionalDomain);
    }
}

