/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.clientsideelement.SmartContentClientSideElement;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper.PilotageStatus;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for pilotage status button
 */
public class PilotageStatusButtonClientSideElement extends SmartContentClientSideElement
{
    /** The pilotage status helper */
    protected PilotageStatusHelper _pilotageStatusHelper; 
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _pilotageStatusHelper = (PilotageStatusHelper) manager.lookup(PilotageStatusHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Get informations on contents' state
     * @param contentsId the ids of contents
     * @param buttonStatus the button status
     * @return informations on contents' state
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(List<String> contentsId, String buttonStatus)
    {
        Map<String, Object> results = super.getStatus(contentsId);
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        if (!allrightContents.isEmpty())
        {
            String contentId = (String) allrightContents.get(0).get("id");
            Content content = _resolver.resolveById(contentId);
            PilotageStatus pilotageStatus = _pilotageStatusHelper.getPilotageStatus(content);
            
            PilotageStatus buttonPilotageStatus = StringUtils.isNotBlank(buttonStatus) ? PilotageStatus.valueOf(buttonStatus) : PilotageStatus.NONE;
            
            results.put("isEnabled", isEnable(buttonPilotageStatus, pilotageStatus));
            results.put("isPressed", isToggle(buttonPilotageStatus, pilotageStatus));
            
            results.put("go-back", _getMainlyGoBackDescription(content));
            results.put("go-back-disabled", _getMainlyGoBackDisabledDescription(content));
            results.put("go-next", _getMainlyGoNextDescription(content));
            results.put("go-next-disabled", _getMainlyGoBackDisabledDescription(content));
        }
        
        return results;
    }
    
    @Override
    protected boolean _hasReadAccess(Content content)
    {
        return _rightManager.currentUserHasRight("ODF_Pilotage_See_History_Rights", content) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Get the validation date from content id
     * @param contentId the contentId
     * @return the validation date
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public LocalDate getValidationDate(String contentId)
    {
        Content content = _resolver.resolveById(contentId);
        PilotageStatus pilotageStatus = _pilotageStatusHelper.getPilotageStatus(content);
        
        switch (pilotageStatus)
        {
            case MENTION_VALIDATED:
                return _pilotageStatusHelper.getMentionValidationDate(content);
            case ORGUNIT_VALIDATED:
                return _pilotageStatusHelper.getOrgUnitValidationDate(content);
            case CFVU_VALIDATED:
                return _pilotageStatusHelper.getCFVUValidationDate(content);
            default:
                break;
        }
        
        return null;
    }
    
    /**
     * Get the list of shared first child name of the content
     * @param contentId the contentId
     * @return the list of child name
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public List<String> getSharedChildName(String contentId)
    {
        List<String> names = new ArrayList<>();
        
        ProgramItem programItem = _resolver.resolveById(contentId);
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            if (_odfHelper.getParentAbstractPrograms(child).size() > 1)
            {
                names.add(_contentHelper.getTitle((Content) child));
            }
        }
        
        return names;
    }
    
    /**
     * True if the button is enable
     * @param buttonPilotageStatus the button status
     * @param contentPilotageStatus the content status
     * @return true if the button is enable
     */
    public boolean isEnable(PilotageStatus buttonPilotageStatus, PilotageStatus contentPilotageStatus)
    {
        switch (contentPilotageStatus)
        {
            case NONE:
                return buttonPilotageStatus.equals(PilotageStatus.MENTION_VALIDATED);
            case MENTION_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.MENTION_VALIDATED)
                    || buttonPilotageStatus.equals(PilotageStatus.ORGUNIT_VALIDATED);
            case ORGUNIT_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.ORGUNIT_VALIDATED)
                    || buttonPilotageStatus.equals(PilotageStatus.CFVU_VALIDATED);
            case CFVU_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.CFVU_VALIDATED);
            default:
                break;
        }
        
        return false;
    }
    
    /**
     * True if the button is toggle
     * @param buttonPilotageStatus the button status
     * @param contentPilotageStatus the content status
     * @return true if the button is toggle
     */
    public boolean isToggle(PilotageStatus buttonPilotageStatus, PilotageStatus contentPilotageStatus)
    {
        switch (contentPilotageStatus)
        {
            case NONE:
                return false;
            case MENTION_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.MENTION_VALIDATED);
            case ORGUNIT_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.MENTION_VALIDATED)
                    || buttonPilotageStatus.equals(PilotageStatus.ORGUNIT_VALIDATED);
            case CFVU_VALIDATED:
                return buttonPilotageStatus.equals(PilotageStatus.MENTION_VALIDATED)
                    || buttonPilotageStatus.equals(PilotageStatus.ORGUNIT_VALIDATED)
                    || buttonPilotageStatus.equals(PilotageStatus.CFVU_VALIDATED);
            default:
                break;
        }
        
        return false;
    }
    
    /**
     * Set the pilotage status to the content
     * @param contentId the content id
     * @param status the pilotage status
     * @param validationDateAsString the validation date as string
     * @param comment the comment
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public void setPilotageStatus(String contentId, String status, String validationDateAsString, String comment)
    {
        ModifiableDefaultContent content = _resolver.resolveById(contentId);
        
        if (!_hasRight(content))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "'tried to edit pilotage status of content '" + contentId + "' without sufficient right");
        }
        
        PilotageStatus pilotageStatus = StringUtils.isNotBlank(status) ? PilotageStatus.valueOf(status) : PilotageStatus.NONE;
        
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate validationDate = DateUtils.parseLocalDate(validationDateAsString);
        if (validationDate == null)
        {
            validationDate = getValidationDate(contentId);
            if (validationDate == null || validationDate.isBefore(LocalDate.now()))
            {
                validationDate = LocalDate.now();
            }
        }
        _pilotageStatusHelper.setValidationAttribute(content, validationDate, user, comment, pilotageStatus);
    }
    
    /**
     * Remove the pilotage status from the content
     * @param contentId the content id
     * @param status the pilotage status
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public void removePilotageStatus(String contentId, String status)
    {
        ModifiableDefaultContent content = _resolver.resolveById(contentId);
        
        if (!_hasRight(content))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "'tried to edit pilotage status of content '" + contentId + "' without sufficient right");
        }
        
        PilotageStatus pilotageStatus = StringUtils.isNotBlank(status) ? PilotageStatus.valueOf(status) : PilotageStatus.NONE;
        
        _pilotageStatusHelper.removePilotageStatus(content, pilotageStatus);
    }
    
    /**
     * Get content i18n description when we go back to previous step in pilotage status
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getMainlyGoBackDescription(Content content)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("go-back-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
    /**
     * Get content i18n description when we can't go back to previous step in pilotage status
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getMainlyGoBackDisabledDescription(Content content)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("go-back-disabled-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
    /**
     * Get content i18n description when we go to next step in pilotage status
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getMainlyGoNextDescription(Content content)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("go-next-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
    /**
     * Get content i18n description when we can't go to next step in pilotage status
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getMainlyGoNextDisabledDescription(Content content)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("go-next-disabled-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
}
