/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.runtime.config.Config;

/**
 * This class store caches of the CostComputationDataComponent
 */
public class CostComputationDataCache
{
    /** Initialized maps */
    private Map<String, String> _eqTDComputationByNature;
    private Map<String, Double> _eqTDByNature;
    private Map<String, NormDetails> _effectiveMinMaxByNature;
    private Map<String, Map<String, NormDetails>> _normDetailsByNormAndNature;
    private String _yearNature;
    
    /** Computed maps */
    private CostComputationData _costData;
    private Set<String> _exploredItems;
   
    private OverriddenData _overriddenData;
    
    /**
     * The constructor
     * @param programs the programs concerned by the current cost computation
     * @param norms list of norms
     * @param teachingNatures list of teaching natures
     * @param yearNature identifier for the container's nature with year ("annee") value
     * @param overriddenData overridden data by the user
     */
    public CostComputationDataCache(List<Program> programs, List<OdfReferenceTableEntry> norms, List<OdfReferenceTableEntry> teachingNatures, String yearNature, OverriddenData overriddenData)
    {
        _costData = new CostComputationData(programs);
        _exploredItems = new HashSet<>();
        
        _initializeNatureData(teachingNatures);
        _initializeNormData(norms);
        _yearNature = yearNature;
        
        _overriddenData = overriddenData;
    }

    /**
     * Retrieve norm values created and initialize informations about them in the cache
     * @param norms list of norms
     */
    protected void _initializeNormData(List<OdfReferenceTableEntry> norms)
    {
        _normDetailsByNormAndNature = new HashMap<>();
        
        for (OdfReferenceTableEntry normeEntry : norms)
        {
            Map<String, NormDetails> normDetailsByNature = new HashMap<>();
            if (normeEntry.getContent().hasValue("groupEffectives"))
            {
                ModelAwareRepeater groupEffectives = normeEntry.getContent().getRepeater("groupEffectives");
                for (ModelAwareRepeaterEntry groupEffectivesEntry : groupEffectives.getEntries())
                {
                    NormDetails normDetails = new NormDetails(
                        groupEffectivesEntry.getValue("effectifMax"),
                        groupEffectivesEntry.getValue("effectifMinSup"),
                        normeEntry.getLabel(Config.getInstance().getValue("odf.programs.lang"))
                    );
                    normDetailsByNature.put(ContentDataHelper.getContentIdFromContentData(groupEffectivesEntry, "natureEnseignement"), normDetails);
                }
            }
            _normDetailsByNormAndNature.put(normeEntry.getId(), normDetailsByNature);
        }
    }
    
    /**
     * Retrieve teaching nature values created and initialize informations about them in the cache
     * @param teachingNatures list of teaching natures
     */
    protected void _initializeNatureData(List<OdfReferenceTableEntry> teachingNatures)
    {
        _eqTDComputationByNature = new HashMap<>();
        _eqTDByNature = new HashMap<>();
        _effectiveMinMaxByNature = new HashMap<>();
        
        for (OdfReferenceTableEntry teachingNature : teachingNatures)
        {
            String id = teachingNature.getId();
            
            // eqTD computation
            String eqTDComputationMode = teachingNature.getContent().getValue("eqTDComputationMode", true, "org.ametys.plugins.odfpilotage.cost.eqtd.GroupsMode");
            _eqTDComputationByNature.put(id, eqTDComputationMode);
            
            // eqTD coef
            Double eqTD = PilotageHelper.transformEqTD2Double(teachingNature.getContent().getValue("eqTD"));
            if (eqTD != null)
            {
                _eqTDByNature.put(id, eqTD);
            }
            
            // effective max and min sup
            NormDetails normDetails = new NormDetails(
                teachingNature.getContent().getValue("effectifMax"),
                teachingNature.getContent().getValue("effectifMinSup")
            );
            _effectiveMinMaxByNature.put(id, normDetails);
        }
    }
    
    /**
     * Add the item to the list of explored items
     * @param exploredItem the item to explore
     * @return <code>false</code> if the item has already been added
     */
    public boolean addExploredItem(String exploredItem)
    {
        return _exploredItems.add(exploredItem);
    }
    
    /**
     * Get the computation data, useful at the end of the computation to return the final results.
     * @return A {@link CostComputationData} object with volumes, eqTD, groups, effectives, etc.
     */
    public CostComputationData getCostComputationData()
    {
        return _costData;
    }

    /**
     * Get an eqTD computation mode for a natureId
     * @param natureId the nature id
     * @return the eqTD computation mode
     */
    public String getEqTDComputationByNature(String natureId)
    {
        return _eqTDComputationByNature.get(natureId);
    }
    
    /**
     * Get an eqTD value for a natureId
     * @param natureId the nature id
     * @return the eqTD value
     */
    public Double getEqTDByNature(String natureId)
    {
        return _eqTDByNature.get(natureId);
    }
    
    /**
     * Get an effectiveMinMax for a nature id
     * @param natureId the nature id
     * @return the effective min max
     */
    public NormDetails getEffectiveMinMaxByNature(String natureId)
    {
        return _effectiveMinMaxByNature.get(natureId);
    }
    
    /**
     * Get the norm retrieved by a nature id
     * @param normId the norm id
     * @param natureId the nature id
     * @return the norm
     */
    public NormDetails getNormDetailsForNature(String normId, String natureId)
    {
        return _normDetailsByNormAndNature.get(normId).get(natureId);
    }
    
    /**
     * Check if the norms map contains the norm id
     * @param normId the norm id
     * @return true if the norms map contains the norm id
     */
    public boolean normExists(String normId)
    {
        return _normDetailsByNormAndNature.containsKey(normId);
    }
    
    /**
     * Check if the norms map contains the nature id
     * @param normId the norm id
     * @param natureId the nature id
     * @return true if the norms map contains the nature id
     */
    public boolean natureInNormExists(String normId, String natureId)
    {
        return _normDetailsByNormAndNature.get(normId).containsKey(natureId);
    }
    
    /**
     * Get the identifier of the year nature.
     * @return the identifier of the year ("annee") container's nature, can be null
     */
    public String getYearNature()
    {
        return _yearNature;
    }
    
    /**
     * Get the full program item (or course part, or orgunit) computed data
     * @param programItemId the identifier of the object
     * @return a {@link ProgramItemData} containing all useful informations for a given program item (or course part, or orgunit)
     */
    public ProgramItemData getProgramItemData(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .orElse(null);
    }
    
    /**
     * Get a computed effective for a program item id
     * @param programItemId the program item (or course part) identifier
     * @return the computed effective
     */
    public Effectives getEffectives(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getEffectives)
                .orElse(null);
    }

    /**
     * Put a computed effective in the map
     * @param programItemId the program item (or course part) identifier
     * @param effectives the effectives
     */
    public void putEffectives(String programItemId, Effectives effectives)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setEffectives(effectives);
    }
    
    /**
     * Get a weight for a program item
     * @param programItemId the program item id
     * @return the weight
     */
    public Double getWeight(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getWeight)
                .orElse(null);
    }
    
    /**
     * Put a weight in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param weight the weight
     */
    public void putWeight(String programItemId, Double weight)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setWeight(weight);
    }
    
    /**
     * Get the pathes to programs for a program item
     * @param programItemId the program item id
     * @return the pathes
     */
    public Set<String> getPathes(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getPathes)
                .orElse(null);
    }
    
    /**
     * Set the pathes in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param pathes the pathes to the programs
     */
    public void putPathes(String programItemId, Set<String> pathes)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setPathes(pathes);
    }
    
    /**
     * Get the volumes of hours for a program item id
     * @param programItemId the program item (or course part) identifier
     * @return the volumes of hours
     */
    public VolumesOfHours getVolumesOfHours(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getVolumesOfHours)
                .orElse(null);
    }

    /**
     * Put volumes of hours in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param volumesOfHours the volumes of hours
     */
    public void putVolumesOfHours(String programItemId, VolumesOfHours volumesOfHours)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setVolumesOfHours(volumesOfHours);
    }

    /**
     * Get the groups for a program item id
     * @param programItemId the program item (or course part) identifier
     * @return the groups
     */
    public Groups getGroups(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getGroups)
                .orElse(null);
    }

    /**
     * Put groups in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param groups the groups
     */
    public void putGroups(String programItemId, Groups groups)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setGroups(groups);
    }

    /**
     * Get the td equivalent for a program item id
     * @param programItemId the program item (or course part) identifier
     * @return the td equivalent
     */
    public EqTD getEqTD(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getEqTD)
                .orElse(null);
    }

    /**
     * Put td equivalent in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param eqTD the td equivalent
     */
    public void putEqTD(String programItemId, EqTD eqTD)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setEqTD(eqTD);
    }

    /**
     * Tell if the given item id references a course part
     * @param programItemId the program item (or course part) identifier
     * @return <code>true</code> if it is a court part
     */
    public boolean isCoursePart(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::isCoursePart)
                .orElse(false);
    }
    
    /**
     * Set the item id as a course part or not
     * @param programItemId the program item (or course part) identifier
     * @param isCoursePart <code>true</code> if it is a course part
     */
    public void setIsCoursePart(String programItemId, boolean isCoursePart)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setIsCoursePart(isCoursePart);
    }
    
    /**
     * Get the norm details for a program item id
     * @param programItemId the program item (or course part) identifier
     * @return the norm details
     */
    public NormDetails getNormDetails(String programItemId)
    {
        return Optional.of(programItemId)
                .map(_costData::get)
                .map(ProgramItemData::getNormDetails)
                .orElse(null);
    }

    /**
     * Put norm details in the cache for a program item (or course part)
     * @param programItemId the program item (or course part) identifier
     * @param normDetails the norm details
     */
    public void putNormDetails(String programItemId, NormDetails normDetails)
    {
        ProgramItemData data = _costData.computeIfAbsent(programItemId, __ -> new ProgramItemData());
        data.setNormDetails(normDetails);
    }

    /**
     * Get the overridden effective of a program item
     * @param programItemId the program item id
     * @return the overridden effective
     */
    public Long getOverriddenEffective(String programItemId)
    {
        return _overriddenData.getEffective(programItemId);
    }
    
    /**
     * Get the overridden number groups of a program item
     * @param programItemId the program item id
     * @return the overridden number of groups
     */
    public Long getOverriddenGroups(String programItemId)
    {
        return _overriddenData.getNumberOfGroups(programItemId);
    }
    
    /**
     * Get the overridden volume of hours of a program item
     * @param programItemId the program item id
     * @return the overridden volume of hours
     */
    public Double getOverriddenVolumeOfHours(String programItemId)
    {
        return _overriddenData.getVolumeOfHours(programItemId);
    }
}
