/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Class representing an {@link EqTD}
 */
public class EqTD
{
    /** The global eqTD value by coursePart */
    private Map<String, Double> _globalEqTD;
    
    /** The pro-rated eqTD values */
    private Map<String, Double> _proratedEqTD;
    
    /** The local (or held) eqTD values */
    private Map<String, Double> _localEqTD;
    
    /** List of held occurrences (for course parts only) */
    private Set<String> _heldList;
    
    /**
     * The constructor
     */
    public EqTD()
    {
        _globalEqTD = new HashMap<>();
        _proratedEqTD = new HashMap<>();
        _localEqTD = new HashMap<>();
        _heldList = new HashSet<>();
    }
    
    /**
     * Get the global eqTD value
     * @return the global eqTD value
     */
    public Double getGlobalEqTD()
    {
        return _globalEqTD.values()
                .stream()
                .reduce(0D, Double::sum);
    }

    /**
     * Get the global eqTD value by course part
     * @return all global eqTD values by course part
     */
    public Map<String, Double> getAllGlobalEqTD()
    {
        return _globalEqTD;
    }
    
    /**
     * Set the global eqTD value
     * @param coursePartId The course part identifier
     * @param eqTD the value
     */
    public void addGlobalEqTD(String coursePartId, Double eqTD)
    {
        _globalEqTD.put(coursePartId, eqTD);
    }
    
    /**
     * Set the pro-rated eqTD value
     * @param programItemPath the program item path
     * @param eqTD the value
     */
    public void addProratedEqTD(String programItemPath, Double eqTD)
    {
        _proratedEqTD.put(programItemPath, eqTD);
    }
    
    /**
     * Get the pro-rated eqTD value
     * @param programItemPath the program item path
     * @return the pro-rated eqTD value
     */
    public Double getProratedEqTD(String programItemPath)
    {
        return _proratedEqTD.getOrDefault(programItemPath, 0d);
    }
    
    /**
     * Get all pro-rated eqTD values
     * @return all pro-rated eqTD values
     */
    public Map<String, Double> getProratedEqTD()
    {
        return _proratedEqTD;
    }

    /**
     * Set the local eqTD value
     * @param programItemPath the program item path
     * @param eqTD the value
     */
    public void addLocalEqTD(String programItemPath, Double eqTD)
    {
        _localEqTD.put(programItemPath, eqTD);
    }
    
    /**
     * Set the local eqTD value for a course part
     * @param itemPath the course part path
     * @param held <code>true</code> if the course part is held by the current step
     */
    public void addLocalEqTD4CoursePart(String itemPath, boolean held)
    {
        if (held)
        {
            _heldList.add(itemPath);
            
            Double coursePartLocalEqTD = getGlobalEqTD() / _heldList.size();
            
            for (String heldPath : _heldList)
            {
                _localEqTD.put(heldPath, coursePartLocalEqTD);
            }
        }
        else
        {
            _localEqTD.put(itemPath, 0D);
        }
    }
    
    /**
     * Get the local eqTD value
     * @param programItemPath the program item path
     * @return the local eqTD value
     */
    public Double getLocalEqTD(String programItemPath)
    {
        return _localEqTD.getOrDefault(programItemPath, 0d);
    }

    /**
     * Get all local eqTD values
     * @return all local eqTD values
     */
    public Map<String, Double> getLocalEqTD()
    {
        return _localEqTD;
    }
    
    /**
     * Sum pro-rated and local eqTD values
     * @param eqTD The second eqTD object to sum
     */
    public void sum(EqTD eqTD)
    {
        _globalEqTD.putAll(eqTD.getAllGlobalEqTD());
    }
}
