/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentAccessDeniedException;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.contentstree.TreeConfiguration;
import org.ametys.plugins.odfpilotage.cost.CostComputationComponent;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.Effectives;
import org.ametys.plugins.odfpilotage.cost.entity.EqTD;
import org.ametys.plugins.odfpilotage.cost.entity.Groups;
import org.ametys.plugins.odfpilotage.cost.entity.NormDetails;
import org.ametys.plugins.odfpilotage.cost.entity.OverriddenData;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.plugins.odfpilotage.cost.entity.VolumesOfHours;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * This component handle the content of the cost modeling tool
 */
public class CostComputationTreeHelper extends ODFContentsTreeHelper implements Contextualizable
{
    private static int _ACTION = 2;
    
    private static final String __COST_DATA_KEY = "cost-data-key";
    
    /** The cost computation component */
    protected CostComputationComponent _costComputationComponent;
   
    /** Workflow helper component */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The I18N utils */
    protected I18nUtils _i18nUtils;
    
    /** The pilotage helper component */
    protected PilotageHelper _pilotageHelper;
    
    /** The ODF helper component */
    protected ODFHelper _odfHelper;

    /** The context */
    protected Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _costComputationComponent = (CostComputationComponent) smanager.lookup(CostComputationComponent.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Get the cost data from user session
     * @return the cost data
     */
    protected CostComputationData _getCostData()
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession();
        
        CostComputationData costData = (CostComputationData) session.getAttribute(__COST_DATA_KEY);
        if (costData == null)
        {
            throw new UnsupportedOperationException("Cost data is not initalized in session attribute '" + __COST_DATA_KEY + "'");
        }
        
        return costData;
    }
    
    /**
     * Set the cost data to user session
     * @param costData the cost data
     */
    protected void _setCostData(CostComputationData costData)
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession();
        
        session.setAttribute(__COST_DATA_KEY, costData);
    }
    
    /**
     * Launch the cost computation component algorithm
     * @param content the content to compute
     * @param catalog the catalog
     * @param lang the lang
     * @param overriddenData overridden data by the user
     */
    protected void _launchCostComputation(Content content, String catalog, String lang, OverriddenData overriddenData)
    {
        CostComputationData costData = content instanceof OrgUnit orgUnit
            ? _costComputationComponent.computeCostsOnOrgUnit(orgUnit, catalog, lang, false, overriddenData, true)
            : (content instanceof Program program
                ? _costComputationComponent.computeCostsOnProgram(program, false, overriddenData, true)
                : null
            );
        
        if (costData != null)
        {
            _setCostData(costData);
        }
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param parentContent the root content
     * @param treeConfiguration the tree configuration
     * @param catalog the catalog
     * @param lang the lang
     * @return the children content for each child attributes
     */
    protected Map<String, List<Content>> _getChildrenContent(Content parentContent, TreeConfiguration treeConfiguration, String catalog, String lang)
    {
        Map<String, List<Content>> children = new HashMap<>();
        if (parentContent instanceof OrgUnit orgUnit)
        {
            List<Content> contents = orgUnit.getSubOrgUnits()
                    .stream()
                    .map(this::_resolveSilently)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            if (orgUnit.getParentOrgUnit() == null && !contents.isEmpty())
            {
                children.put("childOrgUnits", contents);
            }
            else
            {
                List<Program> programs = _odfHelper.getProgramsFromOrgUnit((OrgUnit) parentContent, catalog, lang);
                contents = programs.stream()
                        .filter(Content.class::isInstance)
                        .map(Content.class::cast)
                        .collect(Collectors.toList());

                // _programsLink n'est pas un vrai attribut, cet identifiant n'est jamais utilisé
                children.put("_programsLink", contents);
            }
        }
        else
        {
            children = super.getChildrenContent(parentContent, treeConfiguration);
        }
        return children;
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param contentId the parent content
     * @param path the content path
     * @param treeId the tree configuration
     * @param contentPath the content path
     * @param catalog the catalog
     * @param lang the lang
     * @return the children content
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Object> getChildrenContent(String contentId, List<String> path, String treeId, String contentPath, String catalog, String lang)
    {
        TreeConfiguration treeConfiguration = _getTreeConfiguration(treeId);
        Content parentContent = _getParentContent(contentId);
        Map<String, List<Content>> children = _getChildrenContent(parentContent, treeConfiguration, catalog, lang);
   
        Map<String, Object> infos = new HashMap<>();

        List<Map<String, Object>> childrenInfos = new ArrayList<>();
        infos.put("children", childrenInfos);
        
        boolean lastLevel = parentContent instanceof Course parentCourse && !parentCourse.hasCourseLists();

        for (String attributePath : children.keySet())
        {
            for (Content childContent : children.get(attributePath))
            {
                // Ignore course part not on last level
                if (lastLevel || !(childContent instanceof CoursePart))
                {
                    ArrayList<String> subcontentList = new ArrayList<>(path);
                    path.add(childContent.getId());
                    
                    Map<String, Object> childInfo = content2Json(childContent, subcontentList, contentPath + ModelItem.ITEM_PATH_SEPARATOR + childContent.getName());
                    childInfo.put("metadataPath", attributePath);
                    
                    if (!hasChildrenContent(childContent, treeConfiguration))
                    {
                        childInfo.put("children", Collections.EMPTY_LIST);
                    }
                    
                    childrenInfos.add(childInfo);
                }
            }
        }
        return infos;
    }
    
    /**
     * Get the root node informations
     * @param contentId The content
     * @param catalog the catalog
     * @param lang the lang
     * @param overriddenData overridden data by the user
     * @return The informations
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Object> getRootNodeInformations(String contentId, String catalog, String lang, Map<String, Map<String, String>> overriddenData)
    {
        return getNodeInformations(contentId, List.of(contentId), catalog, lang, overriddenData);
    }
    
    /**
     * Get the node informations
     * @param contentId The content
     * @param path The path to the content in the current tree
     * @param catalog the catalog
     * @param lang the lang
     * @param overriddenData Overridden data by the user
     * @return The informations
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Object> getNodeInformations(String contentId, List<String> path, String catalog, String lang, Map<String, Map<String, String>> overriddenData)
    {
        Content content = _ametysResolver.resolveById(contentId);
        _launchCostComputation(content, catalog, lang, new OverriddenData(overriddenData));
        Map<String, Object> json = content2Json(content, path, content.getName());
        if (content instanceof Program program)
        {
            json.put("lang", program.getLanguage());
            json.put("catalog", program.getCatalog());
        }
        return json;
    }
    
    /**
     * Get the default JSON representation of a content of the tree
     * @param content the content
     * @param path The path of the content in the current tree
     * @param contentPath the content path
     * @return the content as JSON
     */
    protected Map<String, Object> content2Json(Content content, List<String> path, String contentPath)
    {
        Map<String, Object> infos = super.content2Json(content, path);
       
        CostComputationData costData = _getCostData();
        ProgramItemData data = costData.get(content.getId());
        
        if (data != null && !(content instanceof CourseList))
        {
            String realContentPath =
                    !(content instanceof OrgUnit) && _ametysResolver.query(QueryHelper.getXPathQuery(StringUtils.substringBefore(contentPath, ModelItem.ITEM_PATH_SEPARATOR), OrgUnitFactory.ORGUNIT_NODETYPE, null)).getSize() > 0
                    ? StringUtils.substringAfter(contentPath, ModelItem.ITEM_PATH_SEPARATOR)
                    : contentPath;
            infos.putAll(_programItemData2json(data, realContentPath));
            infos.put("previousData", _previousData2json(content));
        }
        
        return infos;
    }
    
    private Map<String, Object> _programItemData2json(ProgramItemData data, String contentPath)
    {
        Map<String, Object> json = new HashMap<>();
        json.put("effectives", _effectives2json(data.getEffectives(), contentPath));
        json.put("groups", _groups2json(data.getGroups()));
        json.put("norm", _normDetails2json(data.getNormDetails()));
        json.put("volumesOfHours", _volumesOfHours2json(data.getVolumesOfHours()));
        json.put("eqTD", _eqTD2json(data.getEqTD(), contentPath));
        data.getHeRatio(contentPath).ifPresent(heRatio -> json.put("heRatio", heRatio));
        return json;
    }
    
    private Map<String, Object> _previousData2json(Content content)
    {
        Map<String, Object> json = new HashMap<>();
        if (content instanceof CourseListContainer)
        {
            json.put("currentYearEffectives", content.getValue("numberOfStudentsCurrentYear"));
            json.put("precedingYearEffectives", content.getValue("numberOfStudentsPrecedingYear"));
        }
        else if (content instanceof CoursePart)
        {
            json.put("currentYearGroups", content.getValue("currentYearGroups"));
            json.put("precedingYearGroups", content.getValue("precedingYearGroups"));
        }
        
        return json;
    }

    private Map<String, Object> _effectives2json(Effectives effectives, String contentPath)
    {
        Map<String, Object> json = new HashMap<>();
        if (effectives != null)
        {
            effectives.getOverriddenEffective().map(Math::round).ifPresent(eff -> json.put("overridden", eff));
            effectives.getEnteredEffective().map(Math::round).ifPresent(eff -> json.put("entered", eff));
            effectives.getEstimatedEffective().map(Math::round).ifPresent(eff -> json.put("estimated", eff));
            effectives.getComputedEffective().map(Math::round).ifPresent(eff -> json.put("computed", eff));
            json.put("inconsistent", effectives.isInconsistentWithParent());
            Optional.of(contentPath).map(effectives::getLocalEffective).map(Math::round).ifPresent(eff -> json.put("local", eff));
            
            // Fill the distribution
            List<Map<String, Object>> distribution = new ArrayList<>();
            for (Entry<String, Double> estimatedEffectiveByPath : effectives.getEstimatedEffectiveByPath().entrySet())
            {
                List<Map<String, String>> path = _pilotageHelper.getContentsFromPath(estimatedEffectiveByPath.getKey())
                                                    .map(c -> Map.of("label", c.getTitle(), "id", c.getId()))
                                                    .toList();
                distribution.add(Map.of(
                        "value", estimatedEffectiveByPath.getValue(),
                        "path", path
                ));
            }
            distribution.sort(new Comparator<Map<String, Object>>() {
                public int compare(Map<String, Object> o1, Map<String, Object> o2)
                {
                    @SuppressWarnings("unchecked")
                    String path1 = ((List<Map<String, String>>) o1.get("path")).stream().map(p -> p .get("label")).collect(Collectors.joining(" > "));
                    @SuppressWarnings("unchecked")
                    String path2 = ((List<Map<String, String>>) o2.get("path")).stream().map(p -> p .get("label")).collect(Collectors.joining(" > "));
                    
                    return path1.compareTo(path2);
                }
            });
            json.put("distribution", distribution);
        }
        return json;
    }
    
    private Map<String, Object> _groups2json(Groups groups)
    {
        Map<String, Object> json = new HashMap<>();
        if (groups != null)
        {
            json.put("overridden", groups.getOverriddenGroups());
            json.put("entered", groups.getGroupsToOpen());
            json.put("computed", groups.getComputedGroups());
        }
        return json;
    }
    
    private Map<String, Object> _normDetails2json(NormDetails normDetails)
    {
        Map<String, Object> json = new HashMap<>();
        if (normDetails != null)
        {
            json.put("effMax", normDetails.getEffectiveMax());
            json.put("effMinSup", normDetails.getEffectiveMinSup());
            json.put("label", normDetails.getNormLabel());
        }
        return json;
    }
    
    private Map<String, Object> _volumesOfHours2json(VolumesOfHours volumesOfHours)
    {
        Map<String, Object> json = new HashMap<>();
        if (volumesOfHours != null)
        {
            json.put("original", volumesOfHours.getOriginalVolumeOfHours());
            json.putAll(volumesOfHours.getVolumes());
        }
        return json;
    }
    
    private Map<String, Object> _eqTD2json(EqTD eqTD, String contentPath)
    {
        Map<String, Object> json = new HashMap<>();
        if (eqTD != null)
        {
            json.put("global", eqTD.getGlobalEqTD());
            json.put("local", eqTD.getLocalEqTD(contentPath));
            json.put("prorated", eqTD.getProratedEqTD(contentPath));
        }
        return json;
    }
    
    private Content _resolveSilently(String contentId)
    {
        try
        {
            return _ametysResolver.resolveById(contentId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Launch the cost computation component algorithm with overridden data by the user
     * @param contentsToRefresh all open contents in the tool to refresh
     * @param contentId the root node
     * @param catalog the catalog
     * @param lang the lang
     * @param overriddenData overridden data by the user
     * @return new values associated with their path
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Map<String, Object>> refresh(List<Map<String, Object>> contentsToRefresh, String contentId, String catalog, String lang, Map<String, Map<String, String>> overriddenData)
    {
        Map<String, Map<String, Object>> refreshedData = new HashMap<>();
        Content content = _ametysResolver.resolveById(contentId);
        OverriddenData data = new OverriddenData(overriddenData);
        
        _launchCostComputation(content, catalog, lang, data);
        for (Map<String, Object> entry : contentsToRefresh)
        {
            String namePath = (String) entry.get("namePath");
            @SuppressWarnings("unchecked")
            List<String> path = (List<String>) entry.get("path");
            String subContentId = (String) entry.get("contentId");
            
            Content item = _ametysResolver.resolveById(subContentId);
            refreshedData.put(namePath, content2Json(item, path, namePath));
        }
        return refreshedData;
    }
    
    /**
     * Retrieve old data to compare with the new computed one
     * @return all old data to compare
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Object> getOldData()
    {
        Map<String, Object> data = new HashMap<>();

        CostComputationData costData = _getCostData();
        for (String itemId : costData.keySet())
        {
            Map<String, Object> itemMap = new HashMap<>();
            ProgramItemData itemData = costData.get(itemId);
            EqTD eqTD = itemData.getEqTD();
            if (eqTD != null)
            {
                itemMap.put("globalEqTD", eqTD.getGlobalEqTD());
                itemMap.put("proratedEqTD", eqTD.getProratedEqTD());
                itemMap.put("localEqTD", eqTD.getLocalEqTD());
            }
            
            // Replace empty double optional by 0 (only for old data)
            itemMap.put(
                "heRatio",
                itemData.getHeRatios()
                .entrySet()
                .stream()
                .collect(
                    Collectors.toMap(
                        Entry::getKey,
                        entry -> entry.getValue().orElse(0D)
                    )
                )
            );
            
            data.put(itemId, itemMap);
        }
        
        return data;
    }
    
    /**
     * Save overridden data to the contents
     * @param overriddenData overridden data by the user
     * @return true if the saving is successful
     */
    @Callable(rights = "ODF_Rights_Pilotage_SimulateurDeCout")
    public Map<String, Object> saveOverriddenData(Map<String, Map<String, String>> overriddenData)
    {
        Map<String, Object> result = new HashMap<>();
        for (Entry<String, Map<String, String>> data : overriddenData.entrySet())
        {
            String contentId = data.getKey();
            Map<String, String> contentValues = data.getValue();
            
            if (!contentValues.isEmpty())
            {
                String status = null;
                Map<String, String> json = new HashMap<>();
                json.put("id", contentId);
                
                try
                {
                    ModifiableDefaultContent content = _ametysResolver.resolveById(data.getKey());
                    json.put("title", content.getTitle());
                    if (content.hasValue("code"))
                    {
                        json.put("code", content.getValue("code"));
                    }
                    
                    Map<String, Object> values = new HashMap<>();
                    
                    // If it is a course part
                    if (content instanceof CoursePart)
                    {
                        Optional.of("groups")
                            .map(contentValues::get)
                            .map(Long::parseLong)
                            .filter(v -> !Objects.equals(content.getValue("groupsToOpen"), v))
                            .ifPresent(v -> values.put("groupsToOpen", v));
                        
                        Optional.of("nbHours")
                            .map(contentValues::get)
                            .map(Double::parseDouble)
                            .filter(v -> !Objects.equals(content.getValue("nbHours"), v))
                            .ifPresent(v -> values.put("nbHours", v));
                    }
                    // If it is a course list container
                    else if (content instanceof CourseListContainer)
                    {
                        Optional.of("effectivesGlobal")
                            .map(contentValues::get)
                            .map(Long::parseLong)
                            .filter(v -> !Objects.equals(content.getValue("numberOfStudentsEstimated"), v))
                            .ifPresent(v -> values.put("numberOfStudentsEstimated", v));
                    }
                    
                    if (!values.isEmpty())
                    {
                        _contentWorkflowHelper.editContent(content, values, _ACTION, View.of(content.getModel(), values.keySet().toArray(String[]::new)));
                        status = "updated";
                        if (getLogger().isDebugEnabled())
                        {
                            getLogger().debug("[SimulatorSaveData] Content {} has been updated from cost simulator", content.getTitle());
                        }
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    status = "unknown";
                    getLogger().warn("[SimulatorSaveData] Content '{}' doesn't exists anymore", contentId);
                }
                catch (InvalidActionException e)
                {
                    status = "invalidAction";
                    getLogger().warn("[SimulatorSaveData] Invalid edit action (could be a lack of rights) on content '{}'", contentId, e);
                }
                catch (WorkflowException e)
                {
                    status = Optional.of(e)
                        .map(WorkflowException::getRootCause)
                        .filter(EditContentAccessDeniedException.class::isInstance)
                        .map(EditContentAccessDeniedException.class::cast)
                        .map(
                            cause ->
                            {
                                json.put("attributeLabel", _i18nUtils.translate(cause.getModelItem().getLabel()));
                                json.put("attributePath", cause.getModelItem().getPath());
                                return "attributeRight";
                            }
                        )
                        .orElse("workflow");
                    getLogger().warn("[SimulatorSaveData] Workflow problem on content '{}'", contentId, e);
                }
                catch (Exception e)
                {
                    status = "error";
                    getLogger().error("[SimulatorSaveData] Unknown error while updating content '{}'", contentId, e);
                }
                
                if (status != null)
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, String>> jsonContents = (List<Map<String, String>>) result.computeIfAbsent(status, __ -> new ArrayList<>());
                    jsonContents.add(json);
                }
            }
        }
        return result;
    }
}
