/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The pilotage helper.
 */
public class PilotageHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role */
    public static final String ROLE = PilotageHelper.class.getName();
    
    /** MCC Course nature */
    public static final String MCC_MODALITE_SESSION1 = "odf-enumeration.MccModaliteSession1";
    
    /** MCC Course nature */
    public static final String MCC_MODALITE_SESSION2 = "odf-enumeration.MccModaliteSession2";
    
    /** MCC Course nature */
    public static final String MCC_SESSION_NATURE = "odf-enumeration.MccSessionNature";
    
    /** Attribute name for compatible regimes for MCC modalities */
    public static final String COMPATIBLE_REGIMES_ATTRIBUTE_NAME = "compatibleRegimes";
    
    /** Norme */
    public static final String NORME = "odf-enumeration.Norme";

    /** Container MCC Regime */
    public static final String CONTAINER_MCC_REGIME = "mccRegime";
    
    /** Container MCC Number of sessions */
    public static final String CONTAINER_MCC_NUMBER_OF_SESSIONS = "mccNbSessions";
    
    /** Course MCC Regime */
    public static final String COURSE_MCC_REGIME = "mccRegime";
    
    /** The course attribute holding the excluded from MCC state */
    public static final String EXCLUDED_FROM_MCC = "excludedFromMCC";
    
    /** The cache id for step holders by program item */
    private static final String __STEP_HOLDERS_BY_ITEM_CACHE_ID = PilotageHelper.class.getName() + "$stepHoldersByItem";
    
    private static final String __STEPS_BY_ITEM_CACHE_ID = PilotageHelper.class.getName() + "$stepsByItem";
    
    private static final String __CONTENTS_BY_NAME_CACHE_ID = PilotageHelper.class.getName() + "$contentsByName";

    /** The helper for ODF contents */
    protected ODFHelper _odfHelper;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The ametys object resolver instance */
    protected AmetysObjectResolver _ametysResolver;
    /** The right manager */
    protected RightManager _rightManager;

    private ModelItemTypeExtensionPoint _contentAttribute;
    
    /**
     * Enumeration for the step holder status
     */
    public enum StepHolderStatus
    {
        /** No step holder */
        NONE,
        /** Single step holder - Nominal behavior */
        SINGLE,
        /** Multiple steps holder */
        MULTIPLE,
        /** Unknown year id */
        NO_YEAR,
        /** Year is not an ascendant of the given element */
        WRONG_YEAR;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _ametysResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentAttribute = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_CONTENT_ATTRIBUTE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__STEPS_BY_ITEM_CACHE_ID))
        {
            _cacheManager.createRequestCache(__STEPS_BY_ITEM_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_STEPS_BY_ITEM_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_STEPS_BY_ITEM_DESCRIPTION"),
                false
            );
        }
        if (!_cacheManager.hasCache(_getStepsHolderByItemCacheId()))
        {
            _cacheManager.createRequestCache(_getStepsHolderByItemCacheId(),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_STEP_HOLDERS_BY_ITEM_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_STEP_HOLDERS_BY_ITEM_DESCRIPTION"),
                false
            );
        }
        
        if (!_cacheManager.hasCache(__CONTENTS_BY_NAME_CACHE_ID))
        {
            _cacheManager.createRequestCache(__CONTENTS_BY_NAME_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTENTS_BY_NAME_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTENTS_BY_NAME_DESCRIPTION"),
                false
            );
        }
    }
    
    /**
     * Invalidate all caches
     */
    public void invalidateAllCaches()
    {
        _cacheManager.get(_getStepsHolderByItemCacheId()).invalidateAll();
        _cacheManager.get(__STEPS_BY_ITEM_CACHE_ID).invalidateAll();
    }

    /**
     * Get the step holder of a program item
     * @param programItem The program item
     * @return {@link Pair} with the step holder status as a key, and the step holder (if found) as a value
     */
    public Pair<StepHolderStatus, Container> getStepHolder(ProgramItem programItem)
    {
        Optional<String> yearId = _odfHelper.getYearId();
        
        if (yearId.isEmpty())
        {
            return Pair.of(StepHolderStatus.NO_YEAR, null);
        }
        
        Set<Container> stepsHolder = _getStepsHolder(programItem, yearId.get());

        switch (stepsHolder.size())
        {
            case 0:
                return Pair.of(StepHolderStatus.NONE, null);
            case 1:
                Container stepHolder = stepsHolder.stream().findFirst().get();
                StepHolderStatus status = getSteps(programItem).contains(stepHolder)
                        ? StepHolderStatus.SINGLE
                        : StepHolderStatus.WRONG_YEAR;
                return Pair.of(status, stepHolder);
            default:
                return Pair.of(StepHolderStatus.MULTIPLE, null);
        }
    }
    
    /**
     * Get the parent containers of type 'year' holding this program item or the program item it self if it is a 'year' container
     * @param programItem the program item
     * @return all containers of type 'year' holding this program item
     */
    public Set<Container> getParentYears(ProgramItem programItem)
    {
        return _odfHelper.getYearId()
                .map(yearId -> _getStepsHolder(programItem, yearId))
                .orElseGet(() -> Set.of());
    }

    /**
     * Get the potential steps holder (step or field "etapePorteuse" in courses) of the {@link ProgramItem}.
     * @param programItem The program item
     * @param yearId The identifier of the year container nature
     * @return The list of potential steps holder linked to the programItem. It there are several, there is no defined step holder.
     */
    protected Set<Container> _getStepsHolder(ProgramItem programItem, String yearId)
    {
        Cache<String, Set<Container>> cache = _cacheManager.get(_getStepsHolderByItemCacheId());
        return cache.get(
            programItem.getId(),
            __ -> _getStepHolderFromCourse(programItem) // If the element is a course and has a step holder
                .or(() -> _getStepHolderFromContainer(programItem, yearId)) // If the element is a step (container of type year)
                .map(Collections::singleton)
                .orElseGet(() -> _getStepsHolderFromParentElements(programItem, yearId)) // In all other cases, search in the parent elements);
        );
    }
    
    private Optional<Container> _getStepHolderFromCourse(ProgramItem programItem)
    {
        return Optional.of(programItem)
                .filter(Course.class::isInstance)
                .map(Course.class::cast)
                .map(c -> c.<ContentValue>getValue("etapePorteuse"))
                .flatMap(ContentValue::getContentIfExists)
                .map(Container.class::cast);
    }
    
    private Optional<Container> _getStepHolderFromContainer(ProgramItem programItem, String yearId)
    {
        return Optional.of(programItem)
                .filter(pi -> _odfHelper.isContainerOfNature(pi, yearId))
                .map(Container.class::cast);
    }

    /**
     * Filter the program item to keep only container with the given nature.
     * @param programItem The program item
     * @param natureId The container nature identifier
     * @return <code>true</code> if it is a container of the given nature, <code>false</code> otherwise
     * @deprecated use {@link ODFHelper#isContainerOfNature(ProgramItem, String)} instead
     */
    @Deprecated
    public boolean isContainerOfNature(ProgramItem programItem, String natureId)
    {
        return _odfHelper.isContainerOfNature(programItem, natureId);
    }
    
    private Set<Container> _getStepsHolderFromParentElements(ProgramItem programItem, String yearId)
    {
        return _odfHelper.getParentProgramItems(programItem)
                .stream()
                .map(pi -> _getStepsHolder(pi, yearId))
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    /**
     * Get the steps of a program item
     * @param programItem The program item
     * @return The steps of the given program item
     */
    public Set<Container> getSteps(ProgramItem programItem)
    {
        return _odfHelper.getYearId()
                .map(yearId -> _getSteps(programItem, yearId))
                .orElseGet(() -> Set.of());
    }
    
    /**
     * Internal method to get the steps of a program item, can be directly called by subclasses.
     * @param programItem The program item
     * @param yearId The identifier of the year container nature
     * @return The steps of the given program item
     */
    protected Set<Container> _getSteps(ProgramItem programItem, String yearId)
    {
        Cache<String, Set<Container>> cache = _cacheManager.get(__STEPS_BY_ITEM_CACHE_ID);
        return cache.get(programItem.getId(), __ -> _getStepsToCache(programItem, yearId));
    }
    
    /**
     * Get the steps to add to the cache.
     * @param programItem The program item to search on
     * @param yearId The identifier of the year container nature
     * @return A {@link Set} of {@link Container} corresponding to the steps of the given program item.
     */
    protected Set<Container> _getStepsToCache(ProgramItem programItem, String yearId)
    {
        Set<Container> containers = new HashSet<>();
        
        if (_odfHelper.isContainerOfNature(programItem, yearId))
        {
            containers.add((Container) programItem);
        }
        else
        {
            for (ProgramItem parent : _odfHelper.getParentProgramItems(programItem))
            {
                containers.addAll(_getSteps(parent, yearId));
            }
        }
        
        return containers;
    }
    
    /**
     * Determine if the content is a container of nature equals to "annee"
     * @param content The content
     * @return <code>true</code> if the current content item is a container of nature equals to "annee"
     * @deprecated use {@link ODFHelper#isContainerOfTypeYear(Content)} instead
     */
    @Deprecated
    public boolean isContainerOfTypeYear(Content content)
    {
        return _odfHelper.isContainerOfTypeYear(content);
    }
    
    /**
     * Determine if the container nature equals to "annee"
     * @param containerId The container id
     * @return <code>true</code> if the current container nature equals to "annee"
     * @deprecated use {@link ODFHelper#isContainerOfTypeYear(String)} instead
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    @Deprecated
    public boolean isContainerOfTypeYear(String containerId)
    {
        return _odfHelper.isContainerOfTypeYear(containerId);
    }
    
    /**
     * Determine if the container nature equals to "annee"
     * @param container The container
     * @return <code>true</code> if the current container nature equals to "annee"
     * @deprecated use {@link ODFHelper#isContainerOfTypeYear(Container)} instead
     */
    @Deprecated
    public boolean isContainerOfTypeYear(Container container)
    {
        return _odfHelper.isContainerOfTypeYear(container);
    }
    
    /**
     * Get the year container nature identifier.
     * @return an {@link Optional} of the year identifier
     * @deprecated use {@link ODFHelper#getYearId()} instead
     */
    @Deprecated
    public synchronized Optional<String> getYearId()
    {
        return _odfHelper.getYearId();
    }
    
    /**
     * Transform eqTD (can be a quotient) to a double value.
     * @param eqTD The eqTD to parse
     * @return The double value equivalent to the given eqTD
     */
    public static Double transformEqTD2Double(String eqTD)
    {
        Double eqTDDouble = null;
        
        if (StringUtils.isNotEmpty(eqTD))
        {
            // fraction
            if (eqTD.indexOf('/') != -1)
            {
                Double numerator = Double.valueOf(eqTD.substring(0, eqTD.indexOf('/')));
                Double denominator = Double.valueOf(eqTD.substring(eqTD.indexOf('/') + 1));
                eqTDDouble = numerator / denominator;
            }
            // decimal ou entier
            else
            {
                eqTDDouble = Double.valueOf(eqTD);
            }
        }
        
        return eqTDDouble;
    }
    
    /**
     * Get a displayable path from a path A/B/C, it displays Title A > Title B > Title C.
     * @param path A path of contents
     * @return a displayable path with titles
     */
    public String getDisplayablePath(String path)
    {
        return getContentsFromPath(path)                // Get all contents in the path of the current element
                .map(Content::getTitle)                 // Get the content title
                .collect(Collectors.joining(" > "));    // Join the results with a ' > ' separator
    }
    
    /**
     * Get all the contents from a path: program/container/course/coursePart will return a stream with the resolved contents for program, container, course and course part.
     * @param path the path to resolve
     * @return a stream of contents
     */
    public Stream<Content> getContentsFromPath(String path)
    {
        return Stream.of(path.split(ModelItem.ITEM_PATH_SEPARATOR)) // Split the path
            .sequential()                                           // Take care of the order
            .filter(StringUtils::isNotEmpty)                        // Remove empty names
            .map(this::getContentFromName)                          // Get the content from its name
            .flatMap(Optional::stream);                             // Ignore unexisting elements, it shouldn't happen, and get the optional element
    }
    
    /**
     * Get the content from its name
     * @param name the name to resolve
     * @return a content
     */
    public Optional<Content> getContentFromName(String name)
    {
        Cache<String, Optional<Content>> cache = _cacheManager.get(__CONTENTS_BY_NAME_CACHE_ID);
        return cache.get(
            name,
            __ -> _ametysResolver.<Content>query(QueryHelper.getXPathQuery(name, "ametys:content", null))
                .stream()       // Stream the results
                .findFirst()   // Keep only the first result, it shouldn't have more than one);
        );
    }
    
    /**
     * Get the compatible regimes for given MCC modality
     * @param contentId the content id of MCC modality
     * @return the compatible regimes of empty if this modality is not restricted
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public List<String> getCompatibleRegimes(String contentId)
    {
        Content content = _ametysResolver.resolveById(contentId);
        if (content.hasValue(COMPATIBLE_REGIMES_ATTRIBUTE_NAME))
        {
            ContentValue[] contentValues = content.getValue(COMPATIBLE_REGIMES_ATTRIBUTE_NAME);
            return Arrays.stream(contentValues)
                    .map(ContentValue::getContentId)
                    .toList();
        }
        
        return List.of();
    }
    
    /**
     * Get the id for steps holder by item cache.
     * @return the cache id
     */
    protected String _getStepsHolderByItemCacheId()
    {
        return __STEP_HOLDERS_BY_ITEM_CACHE_ID;
    }
    
    
    /**
     * Get the compatible modality in session 2 corresponding to the one from session 1
     * @param contentIds the content ids of MCC modality in session 1. key is a external id, value is the content id
     * @return the content ids of MCC modality in session 1 associated to the compatible modalities of session 2 or null
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED) // Used by DoubleMCCSearchGridRepeater
    public Map<String, Map<String, Object>> getCompatibleModalityInSession2(List<String> contentIds)
    {
        return contentIds.stream()
                .map(this::_findCorrespondingModality)
                .collect(Collectors.toMap(Pair::getLeft, Pair::getRight));
    }
    
    @SuppressWarnings("unchecked")
    private Pair<String, Map<String, Object>> _findCorrespondingModality(String modality1)
    {
        Map<String, Object> matching = null;
        
        Content content = _ametysResolver.resolveById(modality1);
        if (content.hasValue("code"))
        {
            String code = content.getValue("code");
            
            if (StringUtils.isNotBlank(code))
            {
                ContentTypeExpression ctypeExp = new ContentTypeExpression(Operator.EQ, MCC_MODALITE_SESSION2);
                StringExpression sExpr = new StringExpression("code", Operator.EQ, code);
                Expression notArchivedExpr = new NotExpression(new BooleanExpression("archived", true));
                
                AmetysObjectIterable<Content> answers = _ametysResolver.query("//element(*, ametys:content)[" + new AndExpression(ctypeExp, sExpr, notArchivedExpr).build() + "]");
                if (answers.getSize() != 1)
                {
                    getLogger().error("Cannot find a matching code to " + code + " in " + MCC_MODALITE_SESSION2);
                }
                else
                {
                    Content matchingContent = answers.iterator().next();
                    if (matchingContent.hasValue("code"))
                    {
                        matching = (Map<String, Object>) _contentAttribute.getExtension("content").valueToJSONForClient(matchingContent, DataContext.newInstance());
                    }
                }
            }
        }
        
        if (matching == null)
        {
            matching = new HashMap<>();
        }
        
        return Pair.of(modality1, matching);
    }
    
    /**
     * Check if the attribute with the given path can be written on the given content
     * @param content the content
     * @param attributePath the attribute's path
     * @return <code>true</code> if the attribute can be written, <code>false</code> otherwise
     */
    public boolean canWriteMccRestrictions(Content content, String attributePath)
    {
        // Only for courses
        if (content != null && content instanceof Course course)
        {
            // If the attribute path to test is mccRegime
            // And if the regime policy is set to FORCE
            // Then it cannot be written if the regime of the step holder is set
            if (attributePath.equals(COURSE_MCC_REGIME) && getMCCRegimePolicy().equals("FORCE"))
            {
                return this.<ContentValue>getValueFromSteps(course, CONTAINER_MCC_REGIME)
                        .map(ContentValue::getContentIfExists)
                        .map(Optional::isEmpty)
                        .orElse(true);
            }
        }
        return true;
    }
    
    /**
     * Retrieves the value of the given attribute from course's steps
     * @param <T> Type of the retrieved value
     * @param course the course
     * @param attributeName the name of the attribute to retrieve
     * @return the value of the given attribute from course's steps
     */
    public <T> Optional<T> getValueFromSteps(Course course, String attributeName)
    {
        return this.<T>_getValueFromStepHolder(course, attributeName)
            .or(() -> this.<T>_getCommonValueFromSteps(course, attributeName));
    }
    
    private <T> Optional<T> _getValueFromStepHolder(Course course, String attributeName)
    {
        return Optional.of(getStepHolder(course))
            .filter(p -> p.getLeft() == StepHolderStatus.SINGLE)
            .map(p -> p.getRight())
            .map(c -> c.<T>getValue(attributeName));
    }
    
    private <T> Optional<T> _getCommonValueFromSteps(Course course, String attributeName)
    {
        // Get all values from steps including empty values
        List<T> potentialValues = getSteps(course)
            .stream()
            .map(c -> c.<T>getValue(attributeName))
            .distinct()
            .toList();
        
        // Several values are possible, cannot be discriminant
        if (potentialValues.size() > 1)
        {
            return Optional.empty();
        }
        
        // Empty if no value, otherwise the unique value
        return potentialValues.stream().filter(Objects::nonNull).findFirst();
    }
    
    /**
     * Get the MCC restriction policy for the regime field.
     * @return the configured MCC restriction policy, by default "INFO"
     */
    public String getMCCRegimePolicy()
    {
        return Config.getInstance().<String>getValue("odf.mcc.restrictions.policy.regime", true, "INFO");
    }
    
    /**
     * Get the MCC restriction policy for the nbSessions field.
     * @return the configured MCC restriction policy, by default "INFO"
     */
    public String getMCCNbSessionsPolicy()
    {
        return Config.getInstance().<String>getValue("odf.mcc.restrictions.policy.nbSessions", true, "INFO");
    }
    
    /**
     * Get the MCC configuration for client
     * @return the MCC configuration
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getMCCConfiguration()
    {
        return Map.of(
            "rulesEnabled", true,
            "regimeForced", getMCCRegimePolicy().equals("FORCE"),
            "nbSessionBlocked", getMCCNbSessionsPolicy().equals("BLOCK")
        );
    }
    
    /**
     * Get all the years of a program part, search in children.
     * @param programPart The program part
     * @return A set of {@link Container} with the year nature.
     * @deprecated use {@link ODFHelper#getYears(TraversableProgramPart)} instead
     */
    @Deprecated
    public Set<Container> getYears(TraversableProgramPart programPart)
    {
        return _odfHelper.getYears(programPart);
    }
    
    /**
     * Set the excluded from MCC state of courses
     * @param courseIds the course ids
     * @param isExcluded <code>true</code> to exclude courses from MCC
     * @return The result map
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setExcludedMCCState(List<String> courseIds, boolean isExcluded)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> noRightCourses = new ArrayList<>();
        List<String> alrightCourseIds = new ArrayList<>();
        
        for (String id : courseIds)
        {
            Course course = _ametysResolver.resolveById(id);
            
            if (_rightManager.currentUserHasRight("ODF_Rights_Pilotage_Exclude_From_MCC", course) == RightResult.RIGHT_ALLOW)
            {
                excludeFromMCC(course, isExcluded);
                course.saveChanges();
                alrightCourseIds.add(id);
            }
            else
            {
                noRightCourses.add(Map.of("id", id, "title", course.getTitle()));
            }
        }
        
        result.put("alrightCourseIds", alrightCourseIds);
        result.put("noRightCourses", noRightCourses);
        return result;
    }
    
    /**
     * FIXME Probably put this method in Course with ODF fusion plugins in v5
     * <code>true</code> if the course is excluded from MCC
     * @param course the course
     * @return <code>true</code> if the course is excluded from MCC
     */
    public boolean isExcludedFromMCC(Course course)
    {
        return course.getInternalDataHolder().getValue(EXCLUDED_FROM_MCC, false);
    }
    
    /**
     * FIXME Probably put this method in Course with ODF fusion plugins in v5
     * Exclude to course from MCC
     * @param course the course
     * @param isExcluded <code>true</code> to exclude the course from MCC
     */
    public void excludeFromMCC(Course course, boolean isExcluded)
    {
        course.getInternalDataHolder().setValue(EXCLUDED_FROM_MCC, isExcluded);
    }
    
    /**
     * <code>true</code> if the program item parent is excluded in MCC
     * @param programItem the program item
     * @return <code>true</code> if the parent is excluded in MCC
     */
    public boolean isParentExcludedInMCC(ProgramItem programItem)
    {
        if (programItem instanceof Course course)
        {
            List<CourseList> parentCourseLists = course.getParentCourseLists();
            return !parentCourseLists.isEmpty() && parentCourseLists.stream().allMatch(this::isParentExcludedInMCC);
        }
        else if (programItem instanceof CourseList courseList)
        {
            List<Course> parentCourses = courseList.getParentCourses();
            return !parentCourses.isEmpty() && parentCourses.stream().allMatch(c -> isExcludedFromMCC(c) || this.isParentExcludedInMCC(c));
        }
        
        return false;
    }
}
