/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.content.CopyContentComponent;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.RepeaterEntry;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The rules manager
 */
public class RulesManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon ROLE. */
    public static final String ROLE = RulesManager.class.getName();

    /** The thematic content type */
    public static final String THEMATIC_CONTENT_TYPE = "odf-enumeration.Thematic";
    
    /** The attribute name for the thematics repeater in container */
    public static final String CONTAINER_THEMATICS = "thematics";
    
    /** The attribute name for the degree restriction in thematic content */
    public static final String THEMATIC_DEGREE = "degree";
    
    /** The attribute name for the regime restriction in thematic content */
    public static final String THEMATIC_REGIME = "regime";
    
    /** The attribute name for the nb sessions restriction in thematic content */
    public static final String THEMATIC_NB_SESSIONS = "nbSessions";
    
    /** The attribute name for the rules repeater in thematic content */
    public static final String THEMATIC_RULES = "rules";
    
    /** The attribute name for the rules repeater in container */
    public static final String CONTAINER_RULES = "rules";
    
    /** The attribute name for the thematic code */
    public static final String THEMATIC_CODE = "code";
    
    /** The attribute name for the rule thematic code */
    public static final String RULE_THEMATIC_CODE = "thematicCode";
    
    /** The attribute name for the rule code */
    public static final String RULE_CODE = "code";
    
    /** The attribute name for the rule help text */
    public static final String RULE_HELP_TEXT = "helpTextDerogation";
    
    /** The attribute name for the rule text */
    public static final String RULE_TEXT = "text";
    
    /** The attribute name for the rule status */
    public static final String RULE_STATUS = "status";
    
    /** The attribute name for the rule derogable */
    public static final String RULE_DEROGABLE = "isDerogable";
    
    /** The attribute name for the rule help motivation */
    public static final String RULE_HELP_MOTIVATION = "helpTextMotivation";
    
    /** The attribute name for the rule motivation */
    public static final String RULE_MOTIVATION = "motivation";
    
    /** The thematic prefix */
    public static final String THEMATICS_PREFIX = "TS";
    
    /** The additional thematic prefix */
    public static final String ADDITIONAL_THEMATICS_PREFIX = "TA";
    
    /** The rules number suffix, used for additional rules and rules in additional thematics */
    public static final String RULES_NB_SUFFIX = "-ruleNumber";
    
    /** The handle rules right */
    public static final String HANDLE_RULES_RIGHT = "ODF_Rights_Rules_Handle";
    
    /** The name of the specific rules view */
    public static final String SPECIFIC_RULE_VIEW = "specific-rules";
    
    /** The name of the complementary rules view */
    public static final String COMPLEMENTARY_RULE_VIEW = "thematics-edition";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    /** The copy content component */
    protected CopyContentComponent _copyContentComponent;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _thematicsHelper = (ThematicsHelper) manager.lookup(ThematicsHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _copyContentComponent = (CopyContentComponent) manager.lookup(CopyContentComponent.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    /**
     * Determines if rules are enabled
     * @return <code>true</code> if rules are enabled
     */
    public static boolean isRulesEnabled()
    {
        return Config.getInstance().getValue("odf.pilotage.rules.enabled", false, false);
    }
    
    /**
     * Determines if the rules can be modified based on container's MCC status
     * @param container the container
     * @return <code>true</code> if MCC status allow edition of rules
     */
    public boolean isRulesModificationAvailable(Container container)
    {
        if (_mccWorkflowHelper.isMCCCFVUValidated(container))
        {
            // MCC CFVU validated, so rules can't be edited
            return _rightManager.currentUserHasRight(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW;
        }
        else if (_mccWorkflowHelper.isMCCOrgUnitValidated(container))
        {
            // Parent program is MCC Orgunit validated, so rules can be edited only if current user has the MCC orgunit super right
            return _rightManager.currentUserHasRight(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW || _rightManager.currentUserHasRight(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW;
        }
        else if (_mccWorkflowHelper.isRulesValidated(container))
        {
            // Parent program is rules validated, so rules can be edited only if current user has the MCC orgunit super right or the rules mention super right
            return _rightManager.currentUserHasRight(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(MCCWorkflowHelper.RULES_VALIDATED_SUPER_RIGHT_ID, container) == RightResult.RIGHT_ALLOW;
        }
        
        return true;
    }
    
    /**
     * <code>true</code> if the current user has right on the container and its nature equals to "annee"
     * @param containerId the container id
     * @return results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> isMccCompatible(String containerId)
    {
        Container container = _resolver.resolveById(containerId);
        
        if (!hasHandleRulesRight(container))
        {
            return Map.of("error", "no-right");
        }
        
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            return Map.of("error", "no-year");
        }
        
        return Map.of();
    }
    
    /**
     * <code>true</code> if has handle rules right
     * @param container the container
     * @return <code>true</code> if has handle rules right
     */
    public boolean hasHandleRulesRight(Container container)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        RightResult hasRight = _rightManager.hasRight(currentUser, HANDLE_RULES_RIGHT, container);
        return hasRight == RightResult.RIGHT_ALLOW;
    }
    
    private void _checkContainer(Container container)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new IllegalArgumentException("The container with id '" + container.getId() + "' must be of nature 'annee'");
        }
    }
    
    private void _checkStatus(Container container)
    {
        if (!isRulesModificationAvailable(container))
        {
            throw new IllegalArgumentException("MCC status of container with id '" + container.getId() + "' does not allow to edit rules");
        }
    }
    
    /**
     * Add an additional rule to the container for a given thematic
     * @param containerId the container id
     * @param thematicCode the code of the thematic
     * @param text the text rule
     * @param motivation the motivation rule
     * @return the map of results
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> addAdditionalRule(String containerId, String thematicCode, Object text, Object motivation)
    {
        return _addAdditionalRule(containerId, thematicCode, text, motivation, 2);
    }
    
    /**
     * Add an additional rule to the container for a given thematic
     * @param containerId the container id
     * @param thematicCode the code of the thematic
     * @param text the text rule
     * @param motivation the motivation rule
     * @param actionId the action id
     * @return the map of results
     */
    protected Map<String, Object> _addAdditionalRule(String containerId, String thematicCode, Object text, Object motivation, int actionId)
    {
        Map<String, Object> results = new HashMap<>();
        
        try
        {
            Container container = _resolver.resolveById(containerId);
            _checkContainer(container);
            _checkStatus(container);
            
            String ruleCode = thematicCode + "-A" + _getUniqueRuleNumber(container, thematicCode);
            Map<String, Object> repeaterValues = new HashMap<>();
            repeaterValues.put(RULE_THEMATIC_CODE, thematicCode);
            repeaterValues.put(RULE_CODE, ruleCode);
            repeaterValues.put(RULE_TEXT, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_TEXT, text));
            repeaterValues.put(RULE_MOTIVATION, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_MOTIVATION, motivation));
            
            Map<String, Object> values = Map.of(CONTAINER_RULES, SynchronizableRepeater.appendOrRemove(List.of(repeaterValues), Set.of()));
            _contentWorkflowHelper.editContent(container, values, actionId);
            
            results.put("ruleId", ruleCode);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred adding additional rule for container '{}'", containerId, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Edit a container additional rule
     * @param containerId the container id
     * @param ruleCode the code of the rule
     * @param text the text rule
     * @param motivation the motivation rule
     * @return the map of results
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> editAdditionalRule(String containerId, String ruleCode, String text, String motivation)
    {
        Map<String, Object> results = new HashMap<>();
        
        try
        {
            Container container = _resolver.resolveById(containerId);
            _checkContainer(container);
            _checkStatus(container);
            
            Optional< ? extends ModifiableIndexableRepeaterEntry> ruleEntry = _getRuleEntry(container, CONTAINER_RULES, ruleCode);
            if (ruleEntry.isPresent())
            {
                Map<String, Object> repeaterValues = new HashMap<>();
                repeaterValues.put(RULE_TEXT, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_TEXT, text));
                repeaterValues.put(RULE_MOTIVATION, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_MOTIVATION, motivation));
                
                Map<String, Object> values = Map.of(CONTAINER_RULES, SynchronizableRepeater.replace(List.of(repeaterValues), List.of(ruleEntry.get().getPosition())));
                _contentWorkflowHelper.editContent(container, values, 2);
                
                results.put("ruleId", ruleCode);
            }
            else
            {
                getLogger().warn("Can't find the additional rule with code '{}' for container '{}'", ruleCode, containerId);
                results.put("error", "not-exist");
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred editing additional rule with code '{}' for container '{}'", ruleCode, containerId, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Delete a container rule
     * @param containerId the container id
     * @param ruleCode the code of the rule
     * @return the map of results
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> deleteRule(String containerId, String ruleCode)
    {
        Map<String, Object> results = new HashMap<>();
        
        Container container = _resolver.resolveById(containerId);
        _checkContainer(container);
        _checkStatus(container);
        
        Optional< ? extends ModifiableIndexableRepeaterEntry> ruleEntry = _getRuleEntry(container, CONTAINER_RULES, ruleCode);
        if (ruleEntry.isPresent())
        {
            ModifiableIndexableRepeaterEntry entry = ruleEntry.get();
            String thematicCode = entry.getValue(RULE_THEMATIC_CODE);

            boolean success = deleteRules(container, Set.of(ruleEntry.get().getPosition()));
            results.put("thematicId", thematicCode);
            if (!success)
            {
                results.put("error", true);
            }
        }
        else
        {
            getLogger().warn("Can't find the additional rule with code '{}' for container '{}'", ruleCode, containerId);
            results.put("error", "not-exist");
        }
        
        return results;
    }
    
    /**
     * Delete the rules of container
     * @param container the container
     * @param rulePositions the rule positions
     * @return <code>true</code> if the deletion is a success
     */
    public boolean deleteRules(Container container, Set<Integer> rulePositions)
    {
        if (rulePositions.isEmpty())
        {
            return true;
        }
        
        boolean success = false;
        try
        {
            Map<String, Object> values = Map.of(CONTAINER_RULES, SynchronizableRepeater.appendOrRemove(List.of(), rulePositions));
            _contentWorkflowHelper.editContent(container, values, 2);
            success = true;
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred deleting rules for content '{}'", container.getId(), e);
        }
        
        return success;
    }
    
    /**
     * Get the values of given rule in the given container
     * @param containerId the container id
     * @param ruleCode the rule code
     * @return the rule values
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getRuleValues(String containerId, String ruleCode)
    {
        Map<String, Object> results = new HashMap<>();
        
        Container container = _resolver.resolveById(containerId);
        _checkContainer(container);
        
        Optional< ? extends ModifiableIndexableRepeaterEntry> ruleEntry = _getRuleEntry(container, CONTAINER_RULES, ruleCode);
        if (ruleEntry.isPresent())
        {
            ModifiableIndexableRepeaterEntry entry = ruleEntry.get();
            Map<String, Object> values = new HashMap<>();
            values.put("text", _getRichTextToJSONForEdition(container, entry, CONTAINER_RULES, RULE_TEXT));
            values.put("motivation", _getRichTextToJSONForEdition(container, entry, CONTAINER_RULES, RULE_MOTIVATION));
            
            results.put("values", values);
        }
        
        return results;
    }
    
    /**
     * Get the information of given rule in the given container
     * @param containerId the container id
     * @param thematicCode the thematic code
     * @param ruleCode the rule code
     * @return the rule information
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getRuleInfo(String containerId, String thematicCode, String ruleCode)
    {
        Map<String, Object> results = new HashMap<>();
        
        Container container = _resolver.resolveById(containerId);
        _checkContainer(container);
        
        Content thematic = _thematicsHelper.getThematic(container.getCatalog(), thematicCode);
        if (thematic == null)
        {
            results.put("error", true);
            return results;
        }
        
        Optional< ? extends ModifiableIndexableRepeaterEntry> ruleEntry = _getRuleEntry(thematic, THEMATIC_RULES, ruleCode);
        if (ruleEntry.isPresent())
        {
            ModifiableIndexableRepeaterEntry entry = ruleEntry.get();
            results.put("text", _getRichTextToJSONForEdition(thematic, entry, THEMATIC_RULES, RULE_TEXT));
            results.put("helpTextDerogation", _getRichTextToJSONForEdition(thematic, entry, THEMATIC_RULES, RULE_HELP_TEXT));
            results.put("helpTextMotivation", _getRichTextToJSONForEdition(thematic, entry, THEMATIC_RULES, RULE_HELP_MOTIVATION));
        }
        
        Optional< ? extends ModifiableIndexableRepeaterEntry> derogatedRuleEntry = _getRuleEntry(container, CONTAINER_RULES, ruleCode);
        if (derogatedRuleEntry.isPresent())
        {
            results.put("derogationText", _getRichTextToJSONForEdition(container, derogatedRuleEntry.get(), CONTAINER_RULES, RULE_TEXT));
            results.put("derogationMotivation", _getRichTextToJSONForEdition(container, derogatedRuleEntry.get(), CONTAINER_RULES, RULE_MOTIVATION));
        }
        
        return results;
    }
    
    /**
     * Derogate a rule for a given container
     * @param containerId the container
     * @param thematicCode the thematic code
     * @param ruleCode the rule code
     * @param text the text of the derogation
     * @param motivation the motivation of the derogation
     * @return the rule information
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> derogateRule(String containerId, String thematicCode, String ruleCode, Object text, Object motivation)
    {
        return _derogateRule(containerId, thematicCode, ruleCode, text, motivation, 2);
    }
    
    /**
     * Derogate a rule for a given container
     * @param containerId the container
     * @param thematicCode the thematic code
     * @param ruleCode the rule code
     * @param text the text of the derogation
     * @param motivation the motivation of the derogation
     * @param actionId the action id
     * @return the rule information
     */
    protected Map<String, Object> _derogateRule(String containerId, String thematicCode, String ruleCode, Object text, Object motivation, int actionId)
    {
        Map<String, Object> results = new HashMap<>();
        
        try
        {
            Container container = _resolver.resolveById(containerId);
            _checkContainer(container);
            _checkStatus(container);
            
            Set<Integer> entryToRemove = _getRuleEntry(container, CONTAINER_RULES, ruleCode)
                    .map(RepeaterEntry::getPosition)
                    .map(Set::of)
                    .orElseGet(Set::of);
            
            Map<String, Object> repeaterValues = new HashMap<>();
            repeaterValues.put(RULE_CODE, ruleCode);
            repeaterValues.put(RULE_TEXT, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_TEXT, text));
            repeaterValues.put(RULE_MOTIVATION, _object2RichText(container, CONTAINER_RULES + ModelItem.ITEM_PATH_SEPARATOR + RULE_MOTIVATION, motivation));
            Map<String, Object> values = Map.of(CONTAINER_RULES, SynchronizableRepeater.appendOrRemove(List.of(repeaterValues), entryToRemove));
            
            _contentWorkflowHelper.editContent(container, values, actionId);
            
            results.put("ruleId", ruleCode);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred derogating rule for content '{}'", containerId, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Get the containers matching the same thematic restriction of the given container
     * @param params the given parameters
     * @return the allowed containers
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getAllowedContainers(Map<String, Object> params)
    {
        Map<String, Object> result = new HashMap<>();
        
        String containerId = (String) params.get("containerId");
        Container currentContainer = _resolver.resolveById(containerId);
        String catalog = currentContainer.getCatalog();
        
        String ruleCode = (String) params.get("ruleId");
        
        Predicate<Container> canEditRulesPredicate = c -> isRulesModificationAvailable(c);
        Predicate<Container> rightPredicate = c -> hasHandleRulesRight(c);
        Predicate<Container> notItSelfPredicate = c -> !c.getId().equals(containerId);
        
        Stream<Container> containers = StringUtils.isEmpty(ruleCode)
            // If no rule, get containers corresponding to the same restrictions as the current container
            ? _thematicsHelper.getSimilarContainers(
                    currentContainer,
                    List.of(rightPredicate, notItSelfPredicate, canEditRulesPredicate)
                )
            // If there is a rule, get containers corresponding to restrictions of the thematic of the rule
            : _thematicsHelper.getCompatibleContainers(
                    _thematicsHelper.getThematic(catalog, _thematicsHelper.getThematicCode(ruleCode)),
                    List.of(
                        rightPredicate,
                        notItSelfPredicate,
                        canEditRulesPredicate
                    )
                );
        
        // Map container informations
        List<Map<String, Object>> containersInfo = containers
            .map(c ->
                Map.<String, Object>of(
                    "id", c.getId(),
                    "code", c.getCode(),
                    "title", c.getTitle(),
                    "isDerogated", ArrayUtils.contains(c.getValue(CONTAINER_RULES + "/" + RULE_CODE, true), ruleCode),
                    "hasComplementaryRules", c.hasValue(CONTAINER_THEMATICS)
                )
            )
            .toList();
        
        result.put("containers", containersInfo);
        
        return result;
    }
    
    /**
     * Copy the rule to the given containers
     * @param srcContainerId the source container id
     * @param thematicCode the thematic code
     * @param ruleCode the rule code
     * @param destContainerIds the ids of dest container
     * @return the results
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> copyRule(String srcContainerId, String thematicCode, String ruleCode, List<String> destContainerIds)
    {
        Map<String, Object> results = new HashMap<>();
        
        Container container = _resolver.resolveById(srcContainerId);
        _checkContainer(container);
        
        Optional< ? extends ModifiableIndexableRepeaterEntry> entry = _getRuleEntry(container, CONTAINER_RULES, ruleCode);
        List<Map<String, String>> containersInError = new ArrayList<>();
        List<Map<String, String>> updatedContainers = new ArrayList<>();
        if (entry.isPresent())
        {
            ModifiableIndexableRepeaterEntry ruleEntry = entry.get();
            Object text = ruleEntry.getValue(RULE_TEXT);
            Object motivation = ruleEntry.getValue(RULE_MOTIVATION);
            boolean isDerogated  = _isDerogatedRule(ruleEntry);
            for (String destContainerId : destContainerIds)
            {
                Container destContainer = _resolver.resolveById(destContainerId);
                boolean isEditionAvailable = isRulesModificationAvailable(destContainer);
                boolean hasRight = hasHandleRulesRight(destContainer);
                if (isEditionAvailable && hasRight)
                {
                    Map<String, Object> res = new HashMap<>();
                    if (isDerogated)
                    {
                        res = _derogateRule(destContainerId, thematicCode, ruleCode, text, motivation, 222);
                    }
                    else
                    {
                        res = _addAdditionalRule(destContainerId, thematicCode, text, motivation, 222);
                    }
                    
                    if (res.containsKey("error"))
                    {
                        containersInError.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                    }
                    else
                    {
                        updatedContainers.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                    }
                }
                else
                {
                    containersInError.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                    if (!hasRight)
                    {
                        getLogger().error("Unable to copy rule on container '{}' because user has no sufficient right", destContainerId);
                    }
                    else
                    {
                        getLogger().error("Unable to copy rule on container '{}' because MCC status does not allow to edit rules", destContainerId);
                    }
                }
            }
        }
        else
        {
            getLogger().error("No rule exists with id '{}' for container '{}'", ruleCode, srcContainerId);
            results.put("error", "not-exist");
        }
        
        if (!containersInError.isEmpty())
        {
            results.put("size", destContainerIds.size());
            results.put("error-containers", containersInError);
        }
        
        results.put("updated-containers", updatedContainers);
        return results;
    }
    
    /**
     * Copy all rule of the source container to the dest containers
     * @param srcContainerId the source container id
     * @param destContainerIds the destination container ids
     * @return the copy result
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> copyAllRules(String srcContainerId, List<String> destContainerIds)
    {
        Map<String, Object> results = new HashMap<>();
        Container container = _resolver.resolveById(srcContainerId);
        _checkContainer(container);
        
        Map<String, Content> thematics = _thematicsHelper.getThematics(container.getCatalog())
            .collect(
                Collectors.toMap(
                    c -> c.getValue(RulesManager.THEMATIC_CODE),
                    Function.identity()
                )
            );
        
        List<Map<String, String>> containersInError = new ArrayList<>();
        List<Map<String, String>> updatedContainers = new ArrayList<>();
        for (String destContainerId : destContainerIds)
        {
            Container destContainer = _resolver.resolveById(destContainerId);
            boolean isEditionAvailable = isRulesModificationAvailable(destContainer);
            boolean hasRight = hasHandleRulesRight(destContainer);
            if (isEditionAvailable && hasRight)
            {
                Map<String, Boolean> thematicCompatibility = new HashMap<>();
                
                List<Map<String, Object>> destRepeaterValues = new ArrayList<>();
                Set<Integer> entryToRemove = new HashSet<>();
                ModifiableModelAwareRepeater repeater = container.getRepeater(CONTAINER_RULES);
                if (repeater != null)
                {
                    for (ModifiableModelAwareRepeaterEntry rule : repeater.getEntries())
                    {
                        String ruleCode = rule.getValue(RULE_CODE);
                        String thematicCode = _thematicsHelper.getThematicCode(ruleCode);
                        Content thematic = thematics.get(thematicCode);
                        
                        // Destination container is compatible with the thematic
                        if (thematicCompatibility.computeIfAbsent(thematicCode, __ -> _thematicsHelper.areCompatible(destContainer, thematic)))
                        {
                            Map<String, Object> repeaterValues = new HashMap<>();
                            repeaterValues.put(RULE_TEXT, rule.getValue(RULE_TEXT));
                            repeaterValues.put(RULE_MOTIVATION, rule.getValue(RULE_MOTIVATION));
                            
                            // It's a derogated rule
                            if (_isDerogatedRule(rule))
                            {
                                entryToRemove.addAll(
                                    _getRuleEntry(destContainer, CONTAINER_RULES, ruleCode)
                                        .map(e -> Set.of(e.getPosition()))
                                        .orElse(Set.of())
                                );
                                
                                repeaterValues.put(RULE_CODE, ruleCode);
                            }
                            // ... else it's an additional rule
                            else
                            {
                                String newRuleCode = thematicCode + "-A" + _getUniqueRuleNumber(destContainer, thematicCode);
                                repeaterValues.put(RULE_CODE, newRuleCode);
                                repeaterValues.put(RULE_THEMATIC_CODE, rule.getValue(RULE_THEMATIC_CODE));
                            }
                            
                            destRepeaterValues.add(repeaterValues);
                        }
                    }
                }
                
                Map<String, Object> values = new HashMap<>();
                // Add derogated and additional rules to values. Here derogated rules are overridden and additional rules are added
                values.put(CONTAINER_RULES, SynchronizableRepeater.appendOrRemove(destRepeaterValues, entryToRemove));
                
                // Then add complementary rules to values. Here complementary rules are overridden
                ContentType containerCType = _cTypeEP.getExtension(ContainerFactory.CONTAINER_CONTENT_TYPE);
                View complementaryRules = containerCType.getView(COMPLEMENTARY_RULE_VIEW);
                values.putAll(container.dataToMap(complementaryRules));
                
                // Then add specific data if exist to values. Here specific data are overridden
                View specificRules = containerCType.getView(SPECIFIC_RULE_VIEW);
                if (specificRules != null)
                {
                    values.putAll(container.dataToMap(specificRules));
                }
                
                try
                {
                    _contentWorkflowHelper.editContent(destContainer, values, 222);
                    updatedContainers.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                }
                catch (Exception e)
                {
                    containersInError.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                    getLogger().error("An error occurred copying rules for content '{}'", destContainerId, e);
                }
            }
            else
            {
                containersInError.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
                if (!hasRight)
                {
                    getLogger().error("Unable to copy rule on container '{}' because user has no sufficient right", destContainerId);
                }
                else
                {
                    getLogger().error("Unable to copy rule on container '{}' because MCC status does not allow to edit rules", destContainerId);
                }
            }
        }
        
        if (!containersInError.isEmpty())
        {
            results.put("size", destContainerIds.size());
            results.put("error-containers", containersInError);
        }
        
        results.put("updated-containers", updatedContainers);
        return results;
    }
    
    /**
     * Copy all rules from view of the source container to the dest containers
     * @param srcContainerId the source container id
     * @param destContainerIds the destination container ids
     * @param viewName the view name
     * @return the copy result
     */
    @Callable(rights = HANDLE_RULES_RIGHT, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> copyRulesFromView(String srcContainerId, List<String> destContainerIds, String viewName)
    {
        Map<String, Object> results = new HashMap<>();
        Container container = _resolver.resolveById(srcContainerId);
        _checkContainer(container);
        
        List<Map<String, String>> containersInError = new ArrayList<>();
        List<Map<String, String>> updatedContainers = new ArrayList<>();
        for (String destContainerId : destContainerIds)
        {
            Container destContainer = _resolver.resolveById(destContainerId);
            boolean isEditionAvailable = isRulesModificationAvailable(destContainer);
            boolean hasRight = hasHandleRulesRight(destContainer);
            if (isEditionAvailable && hasRight)
            {
                _copyContentComponent.editContent(srcContainerId, destContainerId, Map.of(), viewName, null);
            }
            else
            {
                if (!hasRight)
                {
                    getLogger().error("Unable to copy rules from view '{}' on container '{}' because user has no sufficient right", viewName, destContainerId);
                }
                else
                {
                    getLogger().error("Unable to copy rules from view '{}' on container '{}' because MCC status does not allow to edit rules", viewName, destContainerId);
                }
                containersInError.add(Map.of("id", destContainerId, "title", destContainer.getTitle(), "code", destContainer.getCode()));
            }
        }
        
        if (!containersInError.isEmpty())
        {
            results.put("size", destContainerIds.size());
            results.put("error-containers", containersInError);
        }
        
        results.put("updated-containers", updatedContainers);
        return results;
    }
    
    private boolean _isDerogatedRule(ModifiableModelAwareRepeaterEntry entry)
    {
        return !entry.hasValue(RULE_THEMATIC_CODE);
    }

    private Optional< ? extends ModifiableIndexableRepeaterEntry> _getRuleEntry(Content content, String repeaterName, String ruleCode)
    {
        return Optional.ofNullable(content)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(repeaterName))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(e -> ruleCode.equals(e.getValue(RULE_CODE)))
                .findAny();
    }
    
    private Long _getUniqueRuleNumber(WorkflowAwareContent content, String thematicCode)
    {
        Long ruleNumber = content.getInternalDataHolder().getValue(thematicCode + RulesManager.RULES_NB_SUFFIX, 0L);
        ruleNumber++;
        content.getInternalDataHolder().setValue(thematicCode + RulesManager.RULES_NB_SUFFIX, ruleNumber);
        content.saveChanges();
        
        return ruleNumber;
    }
    
    private Object _getRichTextToJSONForEdition(Content content, ModifiableIndexableRepeaterEntry entry, String repeaterName, String attributeName)
    {
        String dataPath = repeaterName + "[" + entry.getPosition() + "]/" + attributeName;
        DataContext context = RepositoryDataContext.newInstance()
            .withObject(content)
            .withDataPath(dataPath);
        ModelItemType type = entry.getType(attributeName);
        return type.valueToJSONForEdition(entry.getValue(attributeName), context);
    }
    
    private RichText _object2RichText(Container container, String path, Object richTextObject)
    {
        if (richTextObject instanceof RichText richText)
        {
            return richText;
        }
        else if (richTextObject instanceof String richTextAsString)
        {
            ElementDefinition definition = (ElementDefinition) container.getDefinition(path);
            ElementType type = definition.getType();
            
            DataContext withModelItem = RepositoryDataContext.newInstance()
                    .withObject(container)
                    .withModelItem(definition);
            
            return (RichText) type.fromJSONForClient(richTextAsString, withModelItem);
        }
        
        return null;
    }
}
