/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Helper for thematics.
 */
public class ThematicsHelper implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = ThematicsHelper.class.getName();
    
    /** The handle thematics right */
    public static final String HANDLE_THEMATICS_RIGHT = "ODF_Rights_Thematics_Handle";
    
    private static final Function<String, Expression> __NOT_EMPTY_ARRAY_EXPRESSION = attributeName -> new Expression()
    {
        public String build()
        {
            return "jcr:like(ametys:" + attributeName + ", '_%')";
        }
    };
    
    private AmetysObjectResolver _resolver;
    private ODFHelper _odfHelper;
    private RightManager _rightManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    /**
     * Determines if current user is allowed to handle thematics
     * @return <code>true</code> if current user is allowed to handle thematics
     */
    public boolean hasHandleThematicRight()
    {
        return _rightManager.currentUserHasRight(HANDLE_THEMATICS_RIGHT, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Determines if user is allowed to handle thematics
     * @param user The user identity
     * @return <code>true</code> if user is allowed to handle thematics
     */
    public boolean hasHandleThematicRight(UserIdentity user)
    {
        return _rightManager.hasRight(user, HANDLE_THEMATICS_RIGHT, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Get all compatible thematics from a catalog and a degree.
     * @param catalog The catalog name
     * @param degreeId The degree identifier
     * @return A list of thematics
     */
    public List<Content> getCompatibleThematics(String catalog, String degreeId)
    {
        Map<String, List<String>> restrictions = Map.of(
            RulesManager.THEMATIC_DEGREE, List.of(degreeId)
        );
        
        return _getCompatibleThematics(catalog, restrictions);
    }
    
    /**
     * Get all compatible thematics for a container
     * @param container The container
     * @return A list of thematics
     */
    public List<Content> getCompatibleThematics(Container container)
    {
        Map<String, List<String>> restrictions = Map.of(
            RulesManager.THEMATIC_DEGREE, _getContainerDegrees(container),
            RulesManager.THEMATIC_REGIME, _getRegimeValueAsList(container),
            RulesManager.THEMATIC_NB_SESSIONS, _getSessionsValueAsList(container)
        );
        
        return _getCompatibleThematics(container.getCatalog(), restrictions);
    }
    
    private List<String> _getSessionsValueAsList(Content content)
    {
        return Optional.of(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS)
                .map(content::<String>getValue)
                .filter(StringUtils::isNotEmpty)
                .map(List::of)
                .orElseGet(List::of);
    }
    
    private List<String> _getRegimeValueAsList(Content content)
    {
        return Optional.of(PilotageHelper.CONTAINER_MCC_REGIME)
                .map(content::<ContentValue>getValue)
                .filter(Objects::nonNull)
                .map(ContentValue::getContentId)
                .map(List::of)
                .orElseGet(List::of);
    }
    
    private List<Content> _getCompatibleThematics(String catalog, Map<String, List<String>> restrictions)
    {
        // Add standard filters
        List<Expression> expressions = _getCommonThematicExpressions(catalog, true);
        
        // Add restrictions filter
        for (String key : restrictions.keySet())
        {
            expressions.add(_getMultipleStringExpressionOrEmpty(key, restrictions.get(key)));
        }
        
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(expressions.toArray(Expression[]::new)));
        return _resolver.<Content>query(query).stream().toList();
    }
    
    /**
     * Get a thematic content from catalog and code
     * @param catalog The catalog
     * @param thematicCode The code of the thematic
     * @return the content corresponding to the catalog and code, <code>null</code> if not found or archived
     */
    public Content getThematic(String catalog, String thematicCode)
    {
        List<Expression> expressions = _getCommonThematicExpressions(catalog, true);
        expressions.add(new StringExpression(RulesManager.THEMATIC_CODE, Operator.EQ, thematicCode));
        
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(expressions.toArray(Expression[]::new)));
        return _resolver.<Content>query(query).stream().findFirst().orElse(null);
    }
    
    /**
     * Get thematic contents from catalog
     * @param catalog The catalog
     * @return the contents corresponding to the catalog
     */
    public Stream<Content> getThematics(String catalog)
    {
        List<Expression> expressions = _getCommonThematicExpressions(catalog, true);
        
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(expressions.toArray(Expression[]::new)));
        return _resolver.<Content>query(query).stream();
    }
    
    private List<Expression> _getCommonThematicExpressions(String catalog, boolean excludeArchived)
    {
        List<Expression> expressions = new ArrayList<>();
        expressions.add(new ContentTypeExpression(Operator.EQ, RulesManager.THEMATIC_CONTENT_TYPE));
        expressions.add(new StringExpression("catalog", Operator.EQ, catalog));
        if (excludeArchived)
        {
            expressions.add(new NotExpression(new BooleanExpression("archived", true)));
        }
        return expressions;
    }
    
    /**
     * Get compatible containers with the given thematic
     * @param thematic The thematic
     * @param additionalPredicates Additional predicates to apply on containers
     * @return {@link Stream} of compatible {@link Container}
     */
    public Stream<Container> getCompatibleContainers(Content thematic, List<Predicate<Container>> additionalPredicates)
    {
        // Nb sessions filter
        List<String> nbSessions = Optional.ofNullable(thematic.<String>getValue(RulesManager.THEMATIC_NB_SESSIONS))
                .filter(StringUtils::isNotEmpty)
                .map(List::of)
                .orElseGet(List::of);
        
        // Regimes filter
        List<String> regimes = Optional.ofNullable(thematic.<ContentValue[]>getValue(RulesManager.THEMATIC_REGIME))
                .map(Stream::of)
                .orElseGet(Stream::of)
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(Content::getId)
                .toList();
        
        // Degree filter
        List<String> degrees = _getThematicDegrees(thematic).toList();
        
        return getCompatibleContainers(thematic.getValue("catalog"), nbSessions, regimes, degrees, additionalPredicates);
    }

    /**
     * Get containers with same restrictions as current container
     * @param container The current container
     * @param additionalPredicates Additional predicates to apply on containers
     * @return {@link Stream} of compatible {@link Container}
     */
    public Stream<Container> getSimilarContainers(Container container, List<Predicate<Container>> additionalPredicates)
    {
        // Nb sessions filter
        List<String> nbSessions = Optional.ofNullable(container.<String>getValue(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS))
                .filter(StringUtils::isNotEmpty)
                .map(List::of)
                .orElseGet(List::of);
        
        // Regimes filter
        List<String> regimes = Optional.ofNullable(container.<ContentValue>getValue(PilotageHelper.CONTAINER_MCC_REGIME))
                .filter(Objects::nonNull)
                .map(ContentValue::getContentId)
                .map(List::of)
                .orElseGet(List::of);

        // Degree filter
        List<String> degrees = _getContainerDegrees(container);
        
        return getCompatibleContainers(container.getValue("catalog"), nbSessions, regimes, degrees, additionalPredicates);
    }
    
    /**
     * Get compatible containers with the given restrictions
     * @param catalog the catalog
     * @param nbSessions the number of sessions
     * @param regimes the regimes
     * @param degrees the degrees
     * @param additionalPredicates Additional predicates to apply on containers
     * @return {@link Stream} of compatible {@link Container}
     */
    public Stream<Container> getCompatibleContainers(String catalog, List<String> nbSessions, List<String> regimes, List<String> degrees, List<Predicate<Container>> additionalPredicates)
    {
        Optional<String> yearId = _odfHelper.getYearId();
        
        // Don't search for compatible containers if year nature is not defined
        if (yearId.isEmpty())
        {
            return Stream.of();
        }
        
        List<Expression> expressions = new ArrayList<>();
        
        // Standard filters
        expressions.add(new ContentTypeExpression(Operator.EQ, ContainerFactory.CONTAINER_CONTENT_TYPE));
        expressions.add(new StringExpression(Container.NATURE, Operator.EQ, yearId.get()));
        expressions.add(new StringExpression("catalog", Operator.EQ, catalog));
        
        // Nb sessions filter
        if (!nbSessions.isEmpty())
        {
            expressions.add(_getSingleStringExpressionOrEmpty(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS, nbSessions));
        }
        
        // Regime filter
        if (!regimes.isEmpty())
        {
            expressions.add(_getSingleStringExpressionOrEmpty(PilotageHelper.CONTAINER_MCC_REGIME, regimes));
        }
        
        // Execute query
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(expressions.toArray(Expression[]::new)));
        Stream<Container> containers = _resolver.<Container>query(query).stream();
        
        // Apply additional predicates
        for (Predicate<Container> additionalPredicate : additionalPredicates)
        {
            containers = containers.filter(additionalPredicate);
        }
        
        // Degree filter
        Predicate<Container> degreeFilter = degrees.isEmpty()
            ? __ -> true
            : c -> _matchDegree(c, degrees);
        
        return containers.filter(degreeFilter);
    }
    
    private boolean _matchDegree(Container container, List<String> degrees)
    {
        List<String> containerDegrees = _getContainerDegrees(container);
        if (!containerDegrees.isEmpty() && !containerDegrees.stream().anyMatch(degrees::contains))
        {
            return false;
        }
        
        return true;
    }
    
    private Expression _getSingleStringExpressionOrEmpty(String attributeName, List<String> values)
    {
        List<Expression> orExpressions = new ArrayList<>();

        if (!values.isEmpty())
        {
            // Attribute match one of the values
            for (String value : values)
            {
                orExpressions.add(new StringExpression(attributeName, Operator.EQ, value));
            }
            
            orExpressions.add(new NotExpression(new MetadataExpression(attributeName)));            // Attribute is not set
            orExpressions.add(new StringExpression(attributeName, Operator.EQ, StringUtils.EMPTY)); // Attribute is empty
        }
        
        return new OrExpression(orExpressions.toArray(Expression[]::new));
    }
    
    private Expression _getMultipleStringExpressionOrEmpty(String attributeName, List<String> values)
    {
        List<Expression> orExpressions = new ArrayList<>();
        
        if (!values.isEmpty())
        {
            // Attribute match one of the values
            for (String value : values)
            {
                orExpressions.add(new StringExpression(attributeName, Operator.EQ, value));
            }
            
            orExpressions.add(new NotExpression(new MetadataExpression(attributeName)));                // Attribute is not set
            orExpressions.add(new NotExpression(__NOT_EMPTY_ARRAY_EXPRESSION.apply(attributeName)));    // Attribute event if multiple is empty, empty array: not(jcr:like(key, '_%'))
        }
        
        return new OrExpression(orExpressions.toArray(Expression[]::new));
    }
    
    /**
     * Get the thematic code from the rule code.
     * @param ruleCode The rule code
     * @return The thematic code
     */
    public String getThematicCode(String ruleCode)
    {
        return StringUtils.substringBefore(ruleCode, "-");
    }
    
    /**
     * Check if the thematic is compatible with the container
     * @param container The container
     * @param thematic The thematic
     * @return <code>true</code> if they are compatible
     */
    public boolean areCompatible(Container container, Content thematic)
    {
        // Check catalog
        if (!Strings.CS.equals(thematic.getValue("catalog"), container.getCatalog()))
        {
            return false;
        }
        
        // Check nbSessions
        if (thematic.hasValue(RulesManager.THEMATIC_NB_SESSIONS) && container.hasValue(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS) && !!Strings.CS.equals(thematic.getValue(RulesManager.THEMATIC_NB_SESSIONS), container.getValue(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS)))
        {
            return false;
        }
        
        // Check regime
        if (thematic.hasValue(RulesManager.THEMATIC_REGIME) && container.hasValue(PilotageHelper.CONTAINER_MCC_REGIME) && !ArrayUtils.contains(thematic.<ContentValue[]>getValue(RulesManager.THEMATIC_REGIME), container.<ContentValue>getValue(PilotageHelper.CONTAINER_MCC_REGIME)))
        {
            return false;
        }
        
        // Check degree
        List<String> thematicDegrees = _getThematicDegrees(thematic).toList();
        if (!thematicDegrees.isEmpty() && !_matchDegree(container, thematicDegrees))
        {
            return false;
        }
        
        return true;
    }
    
    private List<String> _getContainerDegrees(Container container)
    {
        return _odfHelper.getParentPrograms(container)
            .stream()
            .map(Program::getDegree)
            .filter(StringUtils::isNotEmpty)
            .distinct()
            .toList();
    }
    
    private Stream<String> _getThematicDegrees(Content thematic)
    {
        return Optional.ofNullable(thematic.<ContentValue[]>getValue(RulesManager.THEMATIC_DEGREE))
            .map(Stream::of)
            .orElseGet(Stream::of)
            .map(ContentValue::getContentId)
            .filter(StringUtils::isNotEmpty);
    }
}
