/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;

/**
 * Observer to remove derogation and additional rule if the thematic is modified.
 */
public class UpdateThematicRulesStep2Observer extends AbstractThematicRulesObserver
{
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _thematicsHelper = (ThematicsHelper) manager.lookup(ThematicsHelper.ROLE);
    }
    
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFIED;
    }
    
    @Override
    protected Map<Container, Set<Integer>> getRulePositionsToDelete(ModifiableContent content, String catalog, String thematicCode, List< ? extends ModifiableIndexableRepeaterEntry> ruleEntries) throws RepositoryException
    {
        Session session = ((DefaultAmetysObject) content).getNode().getSession();
        
        List<String> ruleCodes = ruleEntries.stream()
                .map(e -> e.<String>getValue(RulesManager.RULE_CODE))
                .toList();
        
        if (content.getValue("archived", false, false))
        {
            // Remove all old and new thematic rules
            List<String> codes = ListUtils.union(ruleCodes, _getOldModifiedRuleCodes(content)).stream().distinct().toList();
            
            // ... the thematic is archived, get all thematic rule positions in containers to remove them
            return _getContainerRulePositions(session, catalog, thematicCode, codes, true);
        }
        else
        {
            Map<Container, Set<Integer>> rulePositionsByContainer = _getRulePositionsToDeleteOnRestrictionsChanged(content, session, catalog, thematicCode, ruleCodes);

            // Get suspended, modified or not derogable rules
            List<String> archivedOrModifiedRuleCodes = ruleEntries.stream()
                    .filter(c -> this._filterRule(content, c))
                    .map(e -> e.<String>getValue(RulesManager.RULE_CODE))
                    .toList();
            
            // ... then get old rule code to delete
            List<String> oldCodeToDeleted = _getOldModifiedRuleCodes(content).stream()
                    .filter(c -> !ruleCodes.contains(c))
                    .toList();
            
            // ... then get rule positions in containers of all this rule to delete it
            _getContainerRulePositions(session, catalog, thematicCode, ListUtils.union(archivedOrModifiedRuleCodes, oldCodeToDeleted), false).entrySet()
                .stream()
                .forEach(e -> {
                    Set<Integer> positions = rulePositionsByContainer.getOrDefault(e.getKey(), new HashSet<>());
                    positions.addAll(e.getValue());
                    rulePositionsByContainer.put(e.getKey(), positions);
                });
            
            return rulePositionsByContainer;
        }
    }

    /**
     * Get the rule positions to delete if restrictions changed
     * @param content the thematic content
     * @param session the session
     * @param catalog the catalog
     * @param thematicCode the thematic code
     * @param ruleCodes the rule codes
     * @return the map of rule position to delete for each container
     * @throws RepositoryException if an error occurred
     */
    private Map<Container, Set<Integer>> _getRulePositionsToDeleteOnRestrictionsChanged(ModifiableContent content, Session session, String catalog, String thematicCode, List<String> ruleCodes) throws RepositoryException
    {
        List<String> degreeIds = Stream.of(content.getValue(RulesManager.THEMATIC_DEGREE, false, new ContentValue[0]))
            .map(ContentValue::getContentId)
            .toList();
        List<String> oldDegreeIds = _getOldDegreeIds(content);
        
        List<String> regimeIds = Stream.of(content.getValue(RulesManager.THEMATIC_REGIME, false, new ContentValue[0]))
                .map(ContentValue::getContentId)
                .toList();
        List<String> oldRegimeIds = _getOldRegimeIds(content);
        
        String nbSessions = content.getValue(RulesManager.THEMATIC_NB_SESSIONS);
        String oldNbSessions = _getOldNumberOfSessions(content);
        
        // Only process if the mccRegime or mccNbSessions or degrees have been modified
        if (!ListUtils.isEqualList(degreeIds, oldDegreeIds) || !ListUtils.isEqualList(regimeIds, oldRegimeIds) || !!Strings.CS.equals(StringUtils.defaultString(nbSessions), StringUtils.defaultString(oldNbSessions)))
        {
            List<String> nbSessionsAsList = StringUtils.isNotBlank(nbSessions) ? List.of(nbSessions) : List.of();
            List<Container> newCompatibleContainers = _thematicsHelper.getCompatibleContainers(catalog, nbSessionsAsList, regimeIds, degreeIds, List.of()).toList();
            List<String> oldNbSessionsAsList = StringUtils.isNotBlank(oldNbSessions) ? List.of(oldNbSessions) : List.of();
            List<Container> incompatibleContainers = _thematicsHelper.getCompatibleContainers(catalog, oldNbSessionsAsList, oldRegimeIds, oldDegreeIds, List.of(Predicate.not(newCompatibleContainers::contains))).toList();
            
            return _getContainerRulePositions(session, catalog, thematicCode, ruleCodes, true).entrySet()
                .stream()
                .filter(e -> incompatibleContainers.contains(e.getKey()))
                .collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue()));
        }
        
        return new HashMap<>();
    }
    
    private boolean _filterRule(Content content, ModifiableIndexableRepeaterEntry entry)
    {
        boolean isDerogable = entry.getValue(RulesManager.RULE_DEROGABLE, false, false);
        if (!isDerogable)
        {
            return true;
        }
        
        String status = entry.getValue(RulesManager.RULE_STATUS);
        switch (status)
        {
            // If suspended, the rule must be deleted in containers
            case "SUSPENDED":
                return true;
            // If modified, the rule must be deleted in containers only if the old status is not modified
            case "MODIFIED":
                List<String> oldModifiedRuleCodes = _getOldModifiedRuleCodes(content);
                return !oldModifiedRuleCodes.contains(entry.getValue(RulesManager.RULE_CODE));
            default:
                return false;
        }
    }
}
