/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.schedulable;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.AddTaskClientSideElement;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This implementation creates an element for adding a new report task
 */
public class AddReportTaskClientSideElement extends AddTaskClientSideElement implements Initializable
{
    /** The report extension point. */
    protected ReportExtensionPoint _reportEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _reportEP = (ReportExtensionPoint) smanager.lookup(ReportExtensionPoint.ROLE);
        super.service(smanager);
    }

    @Override
    public void initialize() throws Exception
    {
        initializeSchedulableParameters(_script, _reportEP);
    }

    @Override
    protected String _configureClass(Configuration configuration) throws ConfigurationException
    {
        return "Ametys.plugins.odfpilotage.controllers.ReportAwareAddTaskButtonController";
    }
    
    /**
     * Method to initialize schedulable parameters. Because it's impossible to have a multiple heritage in Java and the
     * {@link ODFSelectionAwareAddReportTaskClientSideElement} needs intialization, we added a static method.
     * @param script The script to update
     * @param reportEP {@link ReportExtensionPoint} to get some informations on schedulable parameters
     */
    public static void initializeSchedulableParameters(Script script, ReportExtensionPoint reportEP)
    {
        Map<String, Object> parameters = script.getParameters();
        
        String extensionId = Optional.of(parameters)
            .map(p -> p.get("schedulable-parameters"))
            .map(Map.class::cast)
            .map(sp -> sp.get("extensionId"))
            .map(String.class::cast)
            .orElse(null);
        
        if (StringUtils.isNotEmpty(extensionId) && reportEP.hasExtension(extensionId))
        {
            I18nizableText label = reportEP.getExtension(extensionId).getLabel();
            parameters.put("label", label);
            parameters.put("get-parameters-title", label);
            parameters.put("dialog-title", label);
        }
    }

    @Override
    protected Script _configureScript(Configuration configuration) throws ConfigurationException
    {
        Script script = super._configureScript(configuration);
        script.getScriptFiles().add(new ScriptFile("/plugins/odf-pilotage/resources/js/Ametys/plugins/odfpilotage/controllers/ReportAwareAddTaskButtonController.js"));
        return script;
    }
}
