/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tool;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.commons.io.comparator.NameFileComparator;
import org.apache.commons.io.comparator.SizeFileComparator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.ServerCommHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.odfpilotage.manager.PilotageLogFileManager;
import org.ametys.plugins.odfpilotage.report.AbstractPilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;
import org.ametys.plugins.odfpilotage.report.pipeline.PilotageReportZipGenerator;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.ProgramReportSchedulable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * SAX the last 30 log files
 *
 */
public class ListReportsAction extends ServiceableAction
{
    private static final String _CRITERIA_FILENAME = "filename";
    private static final String _CRITERIA_FILE_TYPE = "reportType";
    private static final String _CRITERIA_TARGET = "target";
    private static final String _CRITERIA_CATALOG = "catalog";
    private static final String _CRITERIA_LAST_MODIFIED_AFTER = "lastModifiedAfter";
    private static final String _CRITERIA_LAST_MODIFIED_BEFORE = "lastModifiedBefore";

    private static final String _COLUMN_FILENAME = "reportfile";
    private static final String _COLUMN_LAST_MODIFIED = "lastModified";
    private static final String _COLUMN_LENGTH = "length";
    private static final String _COLUMN_TYPE = "type";
    private static final String _COLUMN_GENERATION_DATE = "generationDate";
    private static final String _COLUMN_OUTPUT_FORMAT = "outputFormat";
    private static final String _COLUMN_CATALOG = "catalog";
    private static final String _COLUMN_LANG = "lang";
    private static final String _COLUMN_TARGET = "target";
    private static final String _COLUMN_CONTEXT = "context";
    private static final String _COLUMN_MANIFEST = "manifest";
    private static final String _COLUMN_LOG_FILE = "logfile";
    
    private static final Map<String, Comparator<File>> _NAME_TO_COMPARATOR = new HashMap<>();
    static
    {
        _NAME_TO_COMPARATOR.put(_COLUMN_FILENAME, NameFileComparator.NAME_INSENSITIVE_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LAST_MODIFIED, LastModifiedFileComparator.LASTMODIFIED_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LENGTH, SizeFileComparator.SIZE_COMPARATOR);
    }
    
    /** ServerComm Helper */
    protected ServerCommHelper _serverCommHelper;
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    /** Pilotage log file manager */
    protected PilotageLogFileManager _pilotageLogFileManager;
    /** Pilotage helper */
    protected ReportHelper _reportHelper;
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    /** The language manager */
    protected LanguagesManager _languageManager;
    /** The catalog manager */
    protected CatalogsManager _catalogManager;
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _serverCommHelper = (ServerCommHelper) smanager.lookup(ServerCommHelper.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _pilotageLogFileManager = (PilotageLogFileManager) smanager.lookup(PilotageLogFileManager.ROLE);
        _reportHelper = (ReportHelper) smanager.lookup(ReportHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _reportEP = (ReportExtensionPoint) smanager.lookup(ReportExtensionPoint.ROLE);
        _languageManager = (LanguagesManager) smanager.lookup(LanguagesManager.ROLE);
        _catalogManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        // Get JS parameters
        Map<String, Object> jsParameters = _serverCommHelper.getJsParameters();
        
        // Search
        Map<String, Object> searchParams = (Map<String, Object>) jsParameters.get("values");
        File[] reportFiles = _getReportFiles(searchParams);
        Integer offset = _getIntValue(jsParameters, "start", 0);
        Integer limit = _getIntValue(jsParameters, "limit", Integer.MAX_VALUE);
        List<Map<String, Object>> results = _getReportResults(reportFiles, offset, limit, _getSortList(jsParameters.get("sort")), searchParams);

        // Construct the JSON object
        Map<String, Object> objectToRead = new HashMap<>();
        objectToRead.put("items", results);
        objectToRead.put("total", reportFiles.length);
        
        // Add JSON to the request to be parsed
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, objectToRead);
        
        return EMPTY_MAP;
    }

    private int _getIntValue(Map<String, Object> values, String key, int defaultValue)
    {
        if (values.containsKey(key))
        {
            return Integer.valueOf(values.get(key).toString()).intValue();
        }
        
        return defaultValue;
    }
    
    private File[] _getReportFiles(Map<String, Object> parameters)
    {
        FileFilter filter = new PilotageFileFilter(
            _getZonedDateTimeFromParameters(parameters, _CRITERIA_LAST_MODIFIED_AFTER),
            _getZonedDateTimeFromParameters(parameters, _CRITERIA_LAST_MODIFIED_BEFORE)
        );
        
        return _reportHelper.getPilotageFolder().listFiles(filter);
    }
    
    private ZonedDateTime _getZonedDateTimeFromParameters(Map<String, Object> parameters, String parameterName)
    {
        return Optional.ofNullable(parameters)
            .map(p -> p.get(parameterName))
            .map(Object::toString)
            .map(DateUtils::parseLocalDate)
            .map(ld -> DateUtils.asZonedDateTime(ld, null))
            .orElse(null);
    }
    
    private List<Map<String, Object>> _getReportResults(File[] reportFiles, Integer offset, Integer limit, List<Object> sort, Map<String, Object> parameters)
    {
        List<Map<String, Object>> reports = new LinkedList<>();
        
        // Get the filter to apply on the results
        Predicate<ZipContent> dataFilter = _createFilterOnZipContent(parameters);
        
        int count = 0;
        for (File reportFile : _sortFiles(reportFiles, sort))
        {
            // Check files until we reach the limit of results (the limit for the page + the offset)
            if (count < offset + limit)
            {
                // Open the zip and retrieve its informations
                ZipContent zipContent = _getZipContent(reportFile);

                // Check if the result validates the filter
                if (dataFilter.test(zipContent))
                {
                    // If the result validates the filter and the result is one to display (greater than the offset, which means on this page), then add it to the files to report
                    if (count >= offset)
                    {
                        _addFileToReport(reports, reportFile, zipContent.manifestData());
                    }
                    
                    // If the result is not to be displayed (it is still part of the previous page / the count is lower than the offset)
                    count++;
                }
            }
            else if (count > offset + limit)
            {
                break;
            }
        }
        
        return reports;
    }
    
    private Predicate<ZipContent> _createFilterOnZipContent(Map<String, Object> parameters)
    {
        // Get the criteria entered
        String reportType = MapUtils.getString(parameters, _CRITERIA_FILE_TYPE);
        String target = MapUtils.getString(parameters, _CRITERIA_TARGET);
        String catalog = MapUtils.getString(parameters, _CRITERIA_CATALOG);
        String filename = MapUtils.getString(parameters, _CRITERIA_FILENAME);

        // No filter defined
        if (StringUtils.isBlank(reportType) && StringUtils.isBlank(target) && StringUtils.isBlank(catalog) && StringUtils.isBlank(filename))
        {
            return __ -> true;
        }
        
        // If at least one criteria is requested, check that the manifest is not null
        Predicate<ZipContent> dataFilter  = z -> z.manifestData() != null;
        
        // If a report type is requested, add it to the filter
        if (StringUtils.isNotBlank(reportType))
        {
            dataFilter = dataFilter.and(z -> reportType.equalsIgnoreCase((String) z.manifestData().get(PilotageReportZipGenerator.MANIFEST_TYPE)));
        }
        
        // If a target is requested, add it to the filter
        if (StringUtils.isNotBlank(target))
        {
            dataFilter = dataFilter.and(z -> target.equalsIgnoreCase((String) z.manifestData().get(PilotageReportZipGenerator.MANIFEST_TARGET)));
        }
        
        // If a catalog is requested, add the caalog check to the filter
        if (StringUtils.isNotBlank(catalog))
        {
            dataFilter = dataFilter.and(z -> _checkCatalog(z, catalog));
        }
        
        // If a filename is requested, add the filenames check to the filter
        if (StringUtils.isNotBlank(filename))
        {
            dataFilter = dataFilter.and(z -> _zipContainsFile(z, filename));
        }
        
        return dataFilter;
    }
    
    private boolean _checkCatalog(ZipContent zipContent, String catalog)
    {
        String zipContentCatalog = (String) zipContent.manifestData().get(OrgUnitReportSchedulable.JOBDATAMAP_CATALOG_KEY);
        
        // First check the catalog from the manifest
        if (StringUtils.isNotBlank(zipContentCatalog))
        {
            return catalog.equalsIgnoreCase(zipContentCatalog);
        }
        // If the target is a Program, retrieve the catalog from the program
        else if (PilotageReportTarget.PROGRAM.name().equalsIgnoreCase((String) zipContent.manifestData().get(PilotageReportZipGenerator.MANIFEST_TARGET)))
        {
            String programId = (String) zipContent.manifestData().get(ProgramReportSchedulable.JOBDATAMAP_PROGRAM_KEY);
            if (programId != null)
            {
                try
                {
                    Program program = _resolver.resolveById(programId);
                    
                    return catalog.equals(program.getCatalog());
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().warn("The content '" + programId + "' has probably been deleted. The catalog filter could not be applied");
                }
            }
        }
        
        return false;
    }
    
    private boolean _zipContainsFile(ZipContent zipContent, String fileNameToFind)
    {
        // Check if at least one of the files in the zip has a name like the one requested
        return zipContent.filenames().stream()
                                     .anyMatch(filename -> Strings.CI.contains(filename, fileNameToFind));
    }
    
    private ZipContent _getZipContent(File reportFile)
    {
        try (ZipFile zipFile = new ZipFile(reportFile);)
        {
            Map<String, Object> manifestData = null;
            ZipEntry zipEntry = zipFile.getEntry(AbstractPilotageReport.MANIFEST_FILENAME);
            if (zipEntry != null)
            {
                try (InputStream is = zipFile.getInputStream(zipEntry);)
                {
                    String manifestContent = IOUtils.toString(is, StandardCharsets.UTF_8);
                    manifestData = _jsonUtils.convertJsonToMap(manifestContent);
                }
            }
            
            List<String> filenames = new ArrayList<>();
            
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements())
            {
                ZipEntry fileEntry = entries.nextElement();
                if (fileEntry != null && !AbstractPilotageReport.MANIFEST_FILENAME.equals(fileEntry.getName()))
                {
                    filenames.add(fileEntry.getName());
                }
            }
            
            return new ZipContent(manifestData, filenames);
        }
        catch (IOException e)
        {
            getLogger().error("An error occured while reading the manifest.json file of " + reportFile.getName(), e);
        }
        
        return null;
    }
    
    private void _addFileToReport(List<Map<String, Object>> reports, File reportFile, Map<String, Object> manifestData)
    {
        String filename = reportFile.getName();
        
        Map<String, Object> report = new HashMap<>();
        report.put(_COLUMN_FILENAME, filename);
        report.put(_COLUMN_LENGTH, String.valueOf(reportFile.length()));
        report.put(_COLUMN_LAST_MODIFIED, DateUtils.epochMilliToString(reportFile.lastModified()));
        
        // Parse manifest.json (if it exists)
        if (manifestData != null)
        {
            report.put(_COLUMN_MANIFEST, manifestData);
            report.putAll(_convertManifestToColumns(manifestData));
            
            // Get log file from manifest data
            File logFile = _getLogFile(manifestData);
            if (logFile != null)
            {
                report.put(_COLUMN_LOG_FILE, logFile.getName());
            }
        }
        
        reports.add(report);
    }
    
    private Map<String, Object> _convertManifestToColumns(Map<String, Object> manifestData)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put(_COLUMN_TYPE, _getReportTypeInfos((String) manifestData.get(PilotageReportZipGenerator.MANIFEST_TYPE)));
        infos.put(_COLUMN_GENERATION_DATE, manifestData.get(PilotageReportZipGenerator.MANIFEST_DATE));
        infos.put(_COLUMN_OUTPUT_FORMAT, manifestData.get(AbstractReportSchedulable.JOBDATAMAP_OUTPUT_FORMAT_KEY));
        String target = manifestData.get(PilotageReportZipGenerator.MANIFEST_TARGET).toString().toUpperCase();
        infos.put(_COLUMN_TARGET, target);
        if (target.equals(PilotageReportTarget.ORGUNIT.name()))
        {
            String contextId = (String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_ORGUNIT_KEY);
            if (StringUtils.isNotEmpty(contextId))
            {
                infos.put(_COLUMN_CONTEXT, _getContentInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_ORGUNIT_KEY)));
            }
            infos.put(_COLUMN_CATALOG, _getCatalogInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_CATALOG_KEY)));
            infos.put(_COLUMN_LANG, _getLanguageInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_LANG_KEY)));
        }
        else if (target.equals(PilotageReportTarget.PROGRAM.name()))
        {
            String programId = manifestData.get(ProgramReportSchedulable.JOBDATAMAP_PROGRAM_KEY).toString();
            infos.put(_COLUMN_CONTEXT, _getContentInfos(programId));
            try
            {
                Program program = _resolver.resolveById(programId);
                infos.put(_COLUMN_CATALOG, _getCatalogInfos(program.getCatalog()));
                infos.put(_COLUMN_LANG, _getLanguageInfos(program.getLanguage()));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The content '" + programId + "' has probably been deleted.");
            }
        }
        return infos;
    }
    
    private Map<String, Object> _getReportTypeInfos(String typeId)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("value", typeId);
        
        PilotageReport report = _reportEP.getExtension(typeId);
        if (report != null)
        {
            infos.put("label", report.getLabel());
        }
        
        return infos;
    }
    
    private Map<String, Object> _getContentInfos(String contentId)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", contentId);
        
        try
        {
            Content content = _resolver.resolveById(contentId);
            infos.put("title", content.getTitle());
            infos.put("isSimple", _contentHelper.isSimple(content));
        }
        catch (UnknownAmetysObjectException e)
        {
            // Nothing
        }
        
        return infos;
    }
    
    private Map<String, Object> _getCatalogInfos(String name)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("value", name);
        
        Catalog catalog = _catalogManager.getCatalog(name);
        if (catalog != null)
        {
            infos.put("label", catalog.getTitle());
        }
        return infos;
        
    }
    
    private Map<String, Object> _getLanguageInfos(String lang)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("code", lang);
        
        if (lang != null)
        {
            Language language = _languageManager.getLanguage(lang);
            if (language != null)
            {
                infos.put("icon", language.getSmallIcon());
                infos.put("label", language.getLabel());
            }
        }
        
        return infos;
    }
    /**
     * Get the log file associated to the report
     * @param manifestData The manifest content
     * @return the log file if exists, or null
     */
    private File _getLogFile(Map<String, Object> manifestData)
    {
        return Optional.ofNullable(manifestData.get(PilotageReportZipGenerator.MANIFEST_TYPE))
            .map(String.class::cast)
            .map(_reportEP::getExtension)
            .map(PilotageReport::getType)
            .map(type -> type + "-" + manifestData.get(PilotageReportZipGenerator.MANIFEST_DATE) + ".log")
            .map(filename -> new File(_pilotageLogFileManager.getLogsDirectory(), filename))
            .filter(File::exists)
            .orElse(null);
    }

    private List<Object> _getSortList(Object sortValues)
    {
        if (sortValues != null)
        {
            return _jsonUtils.convertJsonToList(sortValues.toString());
        }
        
        return null;
    }
    
    @SuppressWarnings("unchecked")
    private File[] _sortFiles(File[] files, List<Object> sortList)
    {
        if (sortList != null)
        {
            for (Object sortValueObj : sortList.reversed())
            {
                Map<String, Object> sortValue = (Map<String, Object>) sortValueObj;
                Comparator<File> comparator = _NAME_TO_COMPARATOR.get(sortValue.get("property"));
                Object direction = sortValue.get("direction");
                if (direction != null && direction.toString().equalsIgnoreCase("DESC"))
                {
                    comparator = Collections.reverseOrder(comparator);
                }
                Arrays.sort(files, comparator);
            }
        }
        
        return files;
    }
    
    private record ZipContent(Map<String, Object> manifestData, List<String> filenames) { /* empty */ }
}
