/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.plugins.contentio.synchronize.impl.AbstractDefaultSynchronizableContentsCollection;
import org.ametys.plugins.odfsync.apogee.ApogeePreviousYearsFieldsDAO;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * SCC for previous years fields.
 */
public abstract class AbstractPreviousYearsSynchronizableContentsCollection extends AbstractDefaultSynchronizableContentsCollection
{
    /** Name of parameter holding the data source id */
    public static final String PARAM_DATASOURCE_ID = "datasourceId";
    /** Name of parameter holding the administrative year */
    public static final String PARAM_CURRENT_YEAR = "currentYear";
    /** Name of parameter holding the last administrative year (N-1) */
    public static final String PARAM_PRECEDING_YEAR = "precedingYear";
    
    /** The DAO for remote DB Apogee */
    protected ApogeePreviousYearsFieldsDAO _apogeePreviousYearsFieldsDAO;
    
    /** The Apogée SCC helper */
    protected ApogeeSynchronizableContentsCollectionHelper _apogeeSCCHelper;

    /** The CDM-fr handler */
    protected CDMFRHandler _cdmfrHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _apogeePreviousYearsFieldsDAO = (ApogeePreviousYearsFieldsDAO) manager.lookup(ApogeePreviousYearsFieldsDAO.ROLE);
        _apogeeSCCHelper = (ApogeeSynchronizableContentsCollectionHelper) manager.lookup(ApogeeSynchronizableContentsCollectionHelper.ROLE);
        _cdmfrHandler = (CDMFRHandler) manager.lookup(CDMFRHandler.ROLE);
    }
    
    /**
     * Get the id of data source
     * @return The id of data source
     */
    protected String getDataSourceId()
    {
        return (String) getParameterValues().get(PARAM_DATASOURCE_ID);
    }
    
    /**
     * Get the current administrative year
     * @return The current administrative year
     */
    protected String getCurrentYear()
    {
        return (String) getParameterValues().get(PARAM_CURRENT_YEAR);
    }
    /**
     * Get the previous administrative year (N-1)
     * @return The previous administrative year (N-1)
     */
    protected String getPrecedingYear()
    {
        return (String) getParameterValues().get(PARAM_PRECEDING_YEAR);
    }
    
    @Override
    public boolean checkCollection()
    {
        // We only synchronize, so we don't care about this parameter
        // If set to true on first launch, no field will be synchronized, so always set to false
        return false;
    }
    
    /**
     * Get the identifier column (can be a concatened column).
     * @return the column id
     */
    protected String getIdColumn()
    {
        return "ID_SYNC";
    }
    
    @Override
    public List<String> getLanguages()
    {
        return List.of(_apogeeSCCHelper.getSynchronizationLang());
    }
    
    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        Set<String> fields = new HashSet<>();
        fields.add(getCurrentYearAttributeName());
        fields.add(getPrecedingYearAttributeName());
        return fields;
    }
    
    @Override
    public void updateSyncInformations(ModifiableContent content, String syncCode, Logger logger) throws Exception
    {
        throw new UnsupportedOperationException("updateSyncInformations() method is not supported for this synchronizable contents collections.");
    }
    
    @Override
    public List<ModifiableContent> populate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        Set<String> contentIds = null;
        
        try
        {
            _startHandleCDMFR();
            List<ModifiableContent> contents = super.populate(logger, progressionTracker);
            contentIds = contents.stream().map(ModifiableContent::getId).collect(Collectors.toSet());
            return contents;
        }
        finally
        {
            _endHandleCDMFR(contentIds);
        }
    }
    
    /**
     * Start handle CDM-fr treatments
     */
    protected void _startHandleCDMFR()
    {
        _cdmfrHandler.suspendCDMFRObserver();
    }
    
    /**
     * End handle CDM-fr treatments
     * @param contentIds the updated contents ids
     */
    protected void _endHandleCDMFR(Set<String> contentIds)
    {
        _cdmfrHandler.unsuspendCDMFRObserver(contentIds);
    }
    
    /**
     * Get the attribute name for current year data.
     * @return the attribute name
     */
    protected abstract String getCurrentYearAttributeName();
    
    /**
     * Get the attribute name for preceding year data.
     * @return the attribute name
     */
    
    protected abstract String getPrecedingYearAttributeName();
    
    /**
     * Retrieve the contents for which to retrieve the values
     * @return The contents
     */
    protected Stream<Content> getContents()
    {
        String query = _getContentPathQuery(_apogeeSCCHelper.getSynchronizationLang(), null, getContentType(), false);
        return _resolver.<Content>query(query).stream();
    }

    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> initialSearchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        // It is a single search if the code is defined in the search, in other cases, it is a global search
        boolean isSingleSearch = initialSearchParameters.containsKey(getIdField());
        
        // Add sync code to search parameters if needed
        Map<String, Object> commonSearchParameters = isSingleSearch
            ? _addSyncCodeToSearchParams(initialSearchParameters)
            : initialSearchParameters;
        
        // Get all results
        List<Map<String, Object>> listOfResults = _search(commonSearchParameters);
        
        // Fill the map with sync codes and codes correspondance
        Map<String, String> syncCode2Code = isSingleSearch
            ? Map.of((String) initialSearchParameters.get(getIdField()), (String) commonSearchParameters.get("syncCode"))
            : _getAllSyncCodes();
        
        // Group results by code
        return _group(listOfResults, syncCode2Code);
    }
    
    private Map<String, Object> _addSyncCodeToSearchParams(Map<String, Object> initialSearchParameters)
    {
        Map<String, Object> commonSearchParameters = new HashMap<>(initialSearchParameters);
        
        String code = (String) commonSearchParameters.remove(getIdField());
        
        String lang = _apogeeSCCHelper.getSynchronizationLang();
        
        // The "code" in parameter is the code of the content not the sync code, retrieve it and add it to search parameters
        String syncCode = Optional.ofNullable(getContent(lang, code, false))
            .map(this::getSyncCode)
            // If sync code does not exists, throw an exception
            .orElseThrow(() -> new IllegalArgumentException("The content with code '" + code + "' with language '" + lang + "' in catalog '" + _apogeeSCCHelper.getSynchronizationCatalog() + "' does not have a synchronization code for SCC '" + getId() + "'"));
        
        // Update search parameters
        commonSearchParameters.put("syncCode", syncCode);
        
        return commonSearchParameters;
    }
    
    private List<Map<String, Object>> _search(Map<String, Object> searchParameters)
    {
        List<Map<String, Object>> results = new ArrayList<>();
        results.addAll(searchByYear(getCurrentYear(), getCurrentYearAttributeName(), searchParameters));
        results.addAll(searchByYear(getPrecedingYear(), getPrecedingYearAttributeName(), searchParameters));
        return results;
    }
    
    private Map<String, Map<String, Object>> _group(List<Map<String, Object>> listOfResults, Map<String, String> syncCode2Code)
    {
        // Compute results
        String idColumn = getIdColumn();
        // Map<code, Map<column, value>>
        Map<String, Map<String, Object>> results = new HashMap<>();
        for (Map<String, Object> resultFromList : listOfResults)
        {
            String syncCode = resultFromList.get(idColumn).toString();
            String code = syncCode2Code.get(syncCode);
            
            // Ignore elements that does not have corresponding code with the sync code, that means the corresponding content has not been imported yet.
            if (StringUtils.isNotBlank(code))
            {
                Map<String, Object> result = results.computeIfAbsent(code, __ -> new HashMap<>());
                result.putAll(resultFromList);
            }
        }
        
        return results;
    }
    
    /**
     * Search the values for a year
     * @param year The year value
     * @param yearAttributeName The year attribute name
     * @param searchParameters The common search parameters
     * @return The results
     */
    protected List<Map<String, Object>> searchByYear(String year, String yearAttributeName, Map<String, Object> searchParameters)
    {
        // If the year is not filled in the configuration, it may not be requested
        if (StringUtils.isBlank(year))
        {
            return List.of();
        }
        
        Map<String, Object> searchParametersForYear = new HashMap<>(searchParameters);
        searchParametersForYear.put("yearValue", year);
        searchParametersForYear.put("attributeName", yearAttributeName);
        
        return executeApogeeRequest(searchParametersForYear);
    }
    
    /**
     * Get the sync code for a content
     * @param content The content
     * @return The sync code
     */
    protected String getSyncCode(Content content)
    {
        return content.getValue(getSyncCodeItemName());
    }
    
    /**
     * Get the item name that contains the synchronization code, it can be an attribute or a property.
     * @return the synchronization code item name
     */
    protected abstract String getSyncCodeItemName();
    
    /**
     * Execute the corresponding apogee request to retrieve the values from Apogee
     * @param parameters The parameters of the request
     * @return The results of the request
     */
    protected abstract List<Map<String, Object>> executeApogeeRequest(Map<String, Object> parameters);
    
    @Override
    protected List<Expression> _getExpressionsList(String lang, String idValue, String contentType, boolean forceStrictCheck)
    {
        List<Expression> expList = super._getExpressionsList(lang, idValue, contentType, forceStrictCheck);
        
        String catalog = _apogeeSCCHelper.getSynchronizationCatalog();
        if (catalog != null)
        {
            expList.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        }
        
        return expList;
    }
    
    @Override
    protected ModifiableContent _importContent(String idValue, Map<String, Object> additionalParameters, String lang, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        throw new UnsupportedOperationException("The method _importContent is not handled by PreviousYearsSCC. The previous years fields can only be synchronized.");
    }
    
    /**
     * Ensure title is present.
     * @implNote This method always forces the title to the current one, and does not warn
     */
    @Override
    protected void ensureTitleIsPresent(Content content, Map<String, List<Object>> remoteValues, Logger logger)
    {
        remoteValues.put(Content.ATTRIBUTE_TITLE, List.of(content.getTitle()));
    }
    
    private Map<String, String> _getAllSyncCodes()
    {
        Map<String, String> syncCode2Code = new HashMap<>();
        
        getContents().forEach(
            content ->
            {
                String syncCode = getSyncCode(content);
                if (StringUtils.isNotEmpty(syncCode))
                {
                    syncCode2Code.put(syncCode, content.getValue(getIdField()));
                }
            }
        );
        
        return syncCode2Code;
    }
}
