/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.components;

import java.io.InputStream;
import java.util.Map;
import java.util.Set;

import org.apache.cocoon.ProcessingException;
import org.slf4j.Logger;
import org.w3c.dom.Element;

import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;

/**
 * Interface for component to import a CDM-fr input stream.
 */
public interface ImportCDMFrComponent
{
    /** Avalon Role */
    public static final String ROLE = ImportCDMFrComponent.class.getName();

    /**
     * Return the attribute identifier for CDM-fr synchronization.
     * @return The attribute name (cdmfrSyncCode)
     */
    public String getIdField();

    /**
     * Get the path of tri-state fields (with local and external values)
     * @param additionalParameters Additional parameters
     * @return the synchronized fields
     */
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters);

    /**
     * Handle the CDM-fr input stream to import all the programs and its dependencies containing into it.
     * @param input The CDM-fr input stream
     * @param parameters The parameters useful for the operation
     * @param scc The {@link SynchronizableContentsCollection} calling this component
     * @param logger The logger
     * @return The list of imported/synchronized programs
     * @throws ProcessingException If an error occurs
     */
    public Map<String, Object> handleInputStream(InputStream input, Map<String, Object> parameters, SynchronizableContentsCollection scc, Logger logger) throws ProcessingException;
    
    /**
     * Import or synchronize the content.
     * @param contentElement Element of the content
     * @param wfDescription The workflow description
     * @param title The title
     * @param syncCode The synchronization code
     * @param context the import context
     * @return The imported or synchronized content
     */
    public ModifiableContent importOrSynchronizeContent(Element contentElement, ContentWorkflowDescription wfDescription, String title, String syncCode, ImportCDMFrContext context);
    
    /**
     * Get the content from the synchronization code, the content type and the context.
     * @param contentType The content type
     * @param syncCode The synchronization code
     * @param context the import context
     * @return the retrieved content
     */
    public ModifiableContent getContent(String contentType, String syncCode, ImportCDMFrContext context);
    
    /**
     * Get the name of catalog to use for import
     * @param contentElement Element of the content
     * @return The catalog to used
     */
    public String getCatalogName(Element contentElement);
    
    /**
     * Get the content ID from the CDM code, if there is no match with the CDM code, then we search with the code.
     * If nothing is found we return null. 
     * @param tableRefId The reference table ID
     * @param cdmCode The CDM code
     * @return A content ID or <code>null</code>
     */
    public String getIdFromCDMThenCode(String tableRefId, String cdmCode);
    
    /**
     * Add specific fields to the content during import or synchronization.
     * @param content Content to update
     * @param additionalParameters Additional parameters
     * @param logger The logger
     * @return The synchronization result of the additional operations
     */
    public ContentSynchronizationResult additionalOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger);
}
