/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.pegase.scc;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.slf4j.Logger;

import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.workflow.AbstractCreateODFContentFunction;
import org.ametys.plugins.contentio.synchronize.AbstractSimpleSynchronizableContentsCollection;
import org.ametys.plugins.odfsync.pegase.ws.PegaseApiManager;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;

import com.opensymphony.workflow.WorkflowException;

import fr.pcscol.pegase.odf.ApiException;
import fr.pcscol.pegase.odf.api.MaquettesExterneApi;
import fr.pcscol.pegase.odf.api.ObjetsMaquetteExterneApi;
import fr.pcscol.pegase.odf.externe.model.DescripteursFormation;
import fr.pcscol.pegase.odf.externe.model.DescripteursGroupement;
import fr.pcscol.pegase.odf.externe.model.DescripteursObjetFormation;
import fr.pcscol.pegase.odf.externe.model.DescripteursObjetMaquette;
import fr.pcscol.pegase.odf.externe.model.DescripteursSise;
import fr.pcscol.pegase.odf.externe.model.DescripteursSyllabus;
import fr.pcscol.pegase.odf.externe.model.EnfantsStructure;
import fr.pcscol.pegase.odf.externe.model.MaquetteStructure;
import fr.pcscol.pegase.odf.externe.model.Nomenclature;
import fr.pcscol.pegase.odf.externe.model.ObjetMaquetteDetail;
import fr.pcscol.pegase.odf.externe.model.ObjetMaquetteStructure;
import fr.pcscol.pegase.odf.externe.model.ObjetMaquetteSummary;
import fr.pcscol.pegase.odf.externe.model.Pageable;
import fr.pcscol.pegase.odf.externe.model.PagedObjetMaquetteSummaries;
import fr.pcscol.pegase.odf.externe.model.PlageDeChoix;
import fr.pcscol.pegase.odf.externe.model.TypeObjetMaquette;

/**
 * SCC for Pegase (COF).
 */
public class PegaseSynchronizableContentsCollection extends AbstractSimpleSynchronizableContentsCollection
{
    private static final String __INSERTED_GROUPEMENT_SUFFIX = "-LIST-";
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The Pégase API manager */
    protected PegaseApiManager _pegaseApiManager;
    
    /** The CDM-fr handler */
    protected CDMFRHandler _cdmfrHandler;
    
    /** The PégaseSCC helper */
    protected PegaseSCCMappingHelper _pegaseSccMappingHelper;
    
    /** The reference table helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    /** Mapping between metadata and columns */
    protected Map<String, Map<String, String>> _mappingByContentType;
    
    /** Search fields to display */
    protected Set<String> _searchFields;

    /** List of imported contents */
    protected Map<String, Integer> _importedContents;
    
    /** List of synchronized contents having differences */
    protected Set<String> _synchronizedContents;
    
    /** List of updated contents by relation */
    protected Set<String> _updatedRelationContents;
    
    /** Map to link contents to its children at the end of the process */
    protected Map<String, Set<String>> _contentsChildren;
    
    /** Default language configured for ODF */
    protected String _odfLang;
    
    /** The structure code for Pégase */
    protected String _structureCode;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _pegaseApiManager = (PegaseApiManager) manager.lookup(PegaseApiManager.ROLE);
        _cdmfrHandler = (CDMFRHandler) manager.lookup(CDMFRHandler.ROLE);
        _pegaseSccMappingHelper = (PegaseSCCMappingHelper) manager.lookup(PegaseSCCMappingHelper.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public String getIdField()
    {
        return "pegaseSyncCode";
    }

    /**
     * Get the identifier JSON field.
     * @return the column id
     */
    protected String getIdColumn()
    {
        return "id";
    }

    @SuppressWarnings("unchecked")
    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        return Optional.ofNullable(additionalParameters)
            .map(params -> params.get("contentTypes"))
            .filter(List.class::isInstance)
            .map(cTypes -> (List<String>) cTypes)
            .filter(Predicate.not(List::isEmpty))
            .map(l -> l.get(0))
            .map(_mappingByContentType::get)
            .map(Map::keySet)
            .orElse(Set.of());
    }

    @Override
    protected Map<String, Object> putIdParameter(String idValue)
    {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(getIdColumn(), List.of(idValue));
        return parameters;
    }
    
    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        _odfLang = Config.getInstance().getValue("odf.programs.lang");
        _searchFields = new HashSet<>();
        _mappingByContentType = new HashMap<>();
        
        if (Config.getInstance().getValue("pegase.activate", true, false))
        {
            _structureCode = Config.getInstance().getValue("pegase.structure.code");
    
            // Champs affichés dans la recherche
            _searchFields.add("libelle");
            _searchFields.add("code");
            _searchFields.add("espace");
            _searchFields.add("id");
            
            // Program
            Map<String, String> contentTypeMapping = new HashMap<>();
            contentTypeMapping.put("title", "libelle");
            contentTypeMapping.put("pegaseCode", "code");
            contentTypeMapping.put("ects", "ects");
            contentTypeMapping.put("educationKind", "codeNatureDiplomeCode");
            contentTypeMapping.put("presentation", "description");
            contentTypeMapping.put("objectives", "objectif");
            contentTypeMapping.put("neededPrerequisite", "prerequis");
            contentTypeMapping.put("mention", "codeMention");
            contentTypeMapping.put("domain", "codeDomaineFormation");
            contentTypeMapping.put("programField", "codeChampFormation");
            contentTypeMapping.put("educationLevel", "codeNiveauFormation");
            contentTypeMapping.put("rncpLevel", "codeNiveauDiplome");
            contentTypeMapping.put("degree", "codeTypeDiplome");
            contentTypeMapping.put("orgUnit", "orgUnit");
            _mappingByContentType.put(ProgramFactory.PROGRAM_CONTENT_TYPE, contentTypeMapping);
            
            // SubProgram
            contentTypeMapping = new HashMap<>();
            contentTypeMapping.put("title", "libelle");
            contentTypeMapping.put("pegaseCode", "code");
            contentTypeMapping.put("ects", "ects");
            contentTypeMapping.put("presentation", "description");
            contentTypeMapping.put("objectives", "objectif");
            contentTypeMapping.put("neededPrerequisite", "prerequis");
            contentTypeMapping.put("orgUnit", "orgUnit");
            _mappingByContentType.put(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, contentTypeMapping);
    
            // Container
            contentTypeMapping = new HashMap<>();
            contentTypeMapping.put("title", "libelle");
            contentTypeMapping.put("pegaseCode", "code");
            contentTypeMapping.put("ects", "ects");
            contentTypeMapping.put("nature", "typeObjetFormation");
            contentTypeMapping.put("period", "typeObjetFormation");
            _mappingByContentType.put(ContainerFactory.CONTAINER_CONTENT_TYPE, contentTypeMapping);
            
            // CourseList
            contentTypeMapping = new HashMap<>();
            contentTypeMapping.put("title", "libelle");
            contentTypeMapping.put("pegaseCode", "code");
            contentTypeMapping.put("plageDeChoix", "plageDeChoix");
            contentTypeMapping.put("obligatoire", "obligatoire");
            contentTypeMapping.put("min", "plageMin");
            contentTypeMapping.put("max", "plageMax");
            _mappingByContentType.put(CourseListFactory.COURSE_LIST_CONTENT_TYPE, contentTypeMapping);
    
            // Course
            contentTypeMapping = new HashMap<>();
            contentTypeMapping.put("title", "libelle");
            contentTypeMapping.put("pegaseCode", "code");
            contentTypeMapping.put("ects", "ects");
            contentTypeMapping.put("description", "description");
            contentTypeMapping.put("courseType", "typeObjetFormation");
            _mappingByContentType.put(CourseFactory.COURSE_CONTENT_TYPE, contentTypeMapping);
        }
    }

    @Override
    protected void configureSearchModel()
    {
        List<String> sortableColumns = List.of("code", "libelle");

        _searchModelConfiguration.addCriterion("libelle", new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_PEGASE_CRITERION_LIBELLE"), "string");
        _searchModelConfiguration.addCriterion("validee", new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_PEGASE_CRITERION_VALIDEE"), "boolean", "edition.boolean-combobox");

        for (String keptField : _searchFields)
        {
            if (sortableColumns.contains(keptField))
            {
                _searchModelConfiguration.addColumn(keptField, new I18nizableText(keptField), true);
            }
            else
            {
                _searchModelConfiguration.addColumn(keptField, new I18nizableText(keptField), false);
            }
        }
    }

    /**
     * Get the catalog for import.
     * @return the catalog
     */
    protected String getCatalog()
    {
        return Optional.of(getParameterValues())
                   .map(params -> params.get("catalog"))
                   .map(String.class::cast)
                   .filter(StringUtils::isNotBlank)
                   .orElseGet(() -> _catalogsManager.getDefaultCatalogName());
    }
    
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> parameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        Map<String, Map<String, Object>> results = new LinkedHashMap<>();
        
        try
        {
            ObjetsMaquetteExterneApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteExterneApi();
            
            // Get details for all requested programs by objetsMaquetteApi
            PagedObjetMaquetteSummaries pagedPrograms = _getPagedProgramsSummaries(parameters, offset, limit, sort, objetsMaquetteApi);
            Long total = pagedPrograms.getTotalElements();
            
            if (total != null)
            {
                // if total is provided, store it so that getTotalCount won't do a useless full search
                parameters.put("totalCount", total.longValue());
            }
            
            List<ObjetMaquetteSummary> programs = pagedPrograms.getItems();
            if (programs != null)
            {
                for (ObjetMaquetteSummary programSummary : programs)
                {
                    String pegaseSyncCode = programSummary.getId().toString();
                    
                    // If not an import, fill the results with Pégase synchronization code
                    Map<String, Object> result = results.computeIfAbsent(pegaseSyncCode, __ -> new HashMap<>());
                    Map<String, String> fields = _getSummaryFields(programSummary);

                    for (String keptField : _searchFields)
                    {
                        String field = fields.get(keptField);
                        result.put(keptField, field);
                    }

                    result.put(SCC_UNIQUE_ID, pegaseSyncCode);
                }
            }
        }
        catch (ApiException | IOException e)
        {
            throw new RuntimeException("Error while getting remote values", e);
        }
        
        return results;
    }
    
    @Override
    public int getTotalCount(Map<String, Object> searchParameters, Logger logger)
    {
        // avoid to relaunch a full search if already done
        Long totalCount = (Long) searchParameters.get("totalCount");
        
        if (totalCount != null)
        {
            return totalCount.intValue();
        }
        
        return super.getTotalCount(searchParameters, logger);
    }
    
    private PagedObjetMaquetteSummaries _getPagedProgramsSummaries(Map<String, Object> parameters, int offset, int limit, List<Object> sort, ObjetsMaquetteExterneApi objetsMaquetteApi) throws ApiException
    {
        Pageable pageable = new Pageable();
        int pageNumber = offset / 50;
        pageable.setPage(pageNumber);
        pageable.setTaille(limit);
        
        if (sort == null)
        {
            pageable.setTri(List.of());
        }
        else
        {
            // Convert the sort parameter from Object to JSON
            String jsonSortParameters = _jsonUtils.convertObjectToJson(sort.get(0));
            // Convert the sort parameter from JSON to Map<String, Object>
            Map<String, Object> sortParameters = _jsonUtils.convertJsonToMap(jsonSortParameters);
            // Create the list containing the sort result; it is going to be of the form : ["field,direction"]
            List<String> sortParametersArray = new ArrayList<>();
            // Create the sort parameter that is going to be sent in the list; it is going to be of the form : "field,direction"
            StringBuilder stringBuilder = new StringBuilder();
            // Get the parameter "property" which is the column on which the sorting is meant to be made
            String property = (String) sortParameters.get("property");
            
            if (!"code".equals(property) && !"libelle".equals(property))
            {
                if ("libelle".equals(property))
                {
                    stringBuilder.append("libelle,").append((String) sortParameters.get("direction"));
                    sortParametersArray.add(stringBuilder.toString());
                }
            }
            else
            {
                stringBuilder.append(property).append(",").append((String) sortParameters.get("direction"));
                sortParametersArray.add(stringBuilder.toString());
            }
            
            pageable.setTri(sortParametersArray);
        }
        
        String searchLabel = (String) parameters.get("libelle");
        List<TypeObjetMaquette> typeObjets = List.of(TypeObjetMaquette.FORMATION);
        Boolean validated = (Boolean) parameters.get("validee");
        return objetsMaquetteApi.rechercherObjetMaquette(_structureCode, pageable, searchLabel, null, typeObjets, null, null, null, null, null, validated, null);
    }

    private Map<String, String> _getSummaryFields(ObjetMaquetteSummary objectSummary)
    {
        return Map.of("id", objectSummary.getId().toString(),
                      "code",  StringUtils.defaultString(objectSummary.getCode()),
                      "libelle", StringUtils.defaultString(objectSummary.getLibelle()),
                      "espace", StringUtils.defaultString(objectSummary.getEspaceLibelle()));
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> parameters, Logger logger)
    {
        Map<String, Map<String, List<Object>>> results = new LinkedHashMap<>();

        List<String> pegaseSyncCodeValues = (List<String>) parameters.getOrDefault(getIdColumn(), new ArrayList<>());
        
        List<String> idValues = new ArrayList<>();
        
        try
        {
            ObjetsMaquetteExterneApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteExterneApi();
            MaquettesExterneApi maquettesApi = _pegaseApiManager.getMaquettesExterneApi();

            if (!pegaseSyncCodeValues.isEmpty())
            {
                // import or synchronize single items
                idValues.addAll(pegaseSyncCodeValues);
            }
            else
            {
                // global synchronization
                PagedObjetMaquetteSummaries pagedPrograms = _getPagedProgramsSummaries(parameters, 0, Integer.MAX_VALUE, null, objetsMaquetteApi);
                for (ObjetMaquetteSummary programDetails : pagedPrograms.getItems())
                {
                    idValues.add(programDetails.getId().toString());
                }
            }
            
            results.putAll(_getObjectDetailsForImport(idValues, objetsMaquetteApi, maquettesApi, logger));
        }
        catch (ApiException | IOException e)
        {
            throw new RuntimeException("Error while getting remote values", e);
        }
        
        return results;
    }

    Map<String, Map<String, List<Object>>> _getObjectDetailsForImport(List<String> idValues, ObjetsMaquetteExterneApi objetsMaquetteApi, MaquettesExterneApi maquettesApi, Logger logger) throws ApiException
    {
        Map<String, Map<String, List<Object>>> results = new LinkedHashMap<>();
        Set<UUID> alreadyHandledObjects = new HashSet<>();

        for (String idValue : idValues)
        {
            // Get the program tree structure from the idValue
            MaquetteStructure maquette = maquettesApi.lireStructureMaquette(_structureCode, UUID.fromString(idValue));
            ObjetMaquetteStructure racine = maquette.getRacine();
            
            results.putAll(_getObjectDetailsForImport(racine, null, alreadyHandledObjects, objetsMaquetteApi, logger));
        }
        
        return results;
    }
    
    private Map<String, Map<String, List<Object>>> _getObjectDetailsForImport(ObjetMaquetteStructure item, EnfantsStructure structure, Set<UUID> alreadyHandledObjects, ObjetsMaquetteExterneApi objetsMaquetteApi, Logger logger) throws ApiException
    {
        Map<String, Map<String, List<Object>>> results = new LinkedHashMap<>();
        
        UUID id = item.getId();
        
        if (alreadyHandledObjects.add(id))
        {
            ObjetMaquetteDetail detail = objetsMaquetteApi.lireObjetMaquette(_structureCode, id);
            
            // get data for this particular Pegase object
            Map<String, List<Object>> objectData = _getObjectFields(detail, structure, item);
            
            // then compute Ametys children from the Pegase structure
            ComputedChildren children = _computeChildren(item);
            
            // link to Ametys children
            objectData.put("children", children.childrenIds());
            
            results.put(id.toString(), objectData);
            
            // add the newly created intermediary objects, if any
            results.putAll(children.newObjects());
            
            // then finally import recursively next objects
            for (EnfantsStructure child : children.nextObjects())
            {
                results.putAll(_getObjectDetailsForImport(child.getObjetMaquette(), child, alreadyHandledObjects, objetsMaquetteApi, logger));
            }
        }
        
        return results;
    }
    
    private ComputedChildren _computeChildren(ObjetMaquetteStructure item)
    {
        List<EnfantsStructure> enfants = item.getEnfants();
        
        List<Object> childrenIds = new ArrayList<>();
        List<EnfantsStructure> nextObjects = new ArrayList<>();
        Map<String, Map<String, List<Object>>> newObjects = new HashMap<>();
        
        if (_isProgramPart(item))
        {
            // item is a programpart, we allow other programparts as direct children
            List<EnfantsStructure> programPartChildren = enfants.stream().filter(e -> _isProgramPart(e.getObjetMaquette())).toList();
            childrenIds.addAll(programPartChildren.stream().map(e -> e.getObjetMaquette().getId().toString()).toList());
            nextObjects.addAll(programPartChildren);
            
            // if there are direct courses, we should inject an intermediary list
            _computeIntermediateLists(item, enfants, childrenIds, nextObjects, newObjects);
            
            // add groups
            List<EnfantsStructure> listChildren = enfants.stream().filter(e -> _isCourseList(e.getObjetMaquette())).toList();
            childrenIds.addAll(listChildren.stream().map(e -> e.getObjetMaquette().getId().toString()).toList());
            nextObjects.addAll(listChildren);
        }
        else if (_isCourseList(item))
        {
            List<EnfantsStructure> courseChildren = enfants.stream().filter(e -> _isCourse(e.getObjetMaquette())).toList();
            childrenIds.addAll(courseChildren.stream().map(e -> e.getObjetMaquette().getId().toString()).toList());
            nextObjects.addAll(courseChildren);
        }
        else if (_isCourse(item))
        {
            // if there are direct courses, we should inject an intermediary list
            _computeIntermediateLists(item, enfants, childrenIds, nextObjects, newObjects);
            
            // add groups
            List<EnfantsStructure> listChildren = enfants.stream().filter(e -> _isCourseList(e.getObjetMaquette())).toList();
            childrenIds.addAll(listChildren.stream().map(e -> e.getObjetMaquette().getId().toString()).toList());
            nextObjects.addAll(listChildren);
        }
        
        return new ComputedChildren(childrenIds, newObjects, nextObjects);
    }
    
    private String _getContentType(ObjetMaquetteStructure item)
    {
        String type = item.getType();
        if (item.getClasse().equals("F"))
        {
            type = "FORMATION";
        }
        else if (item.getClasse().equals("G"))
        {
            type = "GROUPEMENT";
        }
        
        return Optional.ofNullable(type)
                .map(_pegaseSccMappingHelper::getAmetysType)
                .orElse(CourseFactory.COURSE_CONTENT_TYPE);
    }
    
    private boolean _isProgramPart(ObjetMaquetteStructure item)
    {
        String contentType = _getContentType(item);
        
        if (contentType.equals(ProgramFactory.PROGRAM_CONTENT_TYPE) || contentType.equals(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE) || contentType.equals(ContainerFactory.CONTAINER_CONTENT_TYPE))
        {
            return true;
        }
        
        return false;
    }
    
    private boolean _isCourse(ObjetMaquetteStructure item)
    {
        return _getContentType(item).equals(CourseFactory.COURSE_CONTENT_TYPE);
    }
    
    private boolean _isCourseList(ObjetMaquetteStructure item)
    {
        return _getContentType(item).equals(CourseListFactory.COURSE_LIST_CONTENT_TYPE);
    }
    
    private void _computeIntermediateLists(ObjetMaquetteStructure item, List<EnfantsStructure> enfants, List<Object> childrenIds, List<EnfantsStructure> nextObjects, Map<String, Map<String, List<Object>>> newObjects)
    {
        List<EnfantsStructure> courseChildren = enfants.stream().filter(e -> _isCourse(e.getObjetMaquette()) && e.getObligatoire()).toList();
        _computeIntermediateList(item, courseChildren, true, childrenIds, nextObjects, newObjects);
        
        courseChildren = enfants.stream().filter(e -> _isCourse(e.getObjetMaquette()) && !e.getObligatoire()).toList();
        _computeIntermediateList(item, courseChildren, false, childrenIds, nextObjects, newObjects);
    }
    
    private void _computeIntermediateList(ObjetMaquetteStructure item, List<EnfantsStructure> courseChildren, boolean mandatory, List<Object> childrenIds, List<EnfantsStructure> nextObjects, Map<String, Map<String, List<Object>>> newObjects)
    {
        if (!courseChildren.isEmpty())
        {
            String listId = item.getId() + __INSERTED_GROUPEMENT_SUFFIX + (mandatory ? "O" : "F");
            childrenIds.add(listId);
            
            Map<String, List<Object>> listData = new HashMap<>();
            listData.put("workflowDescription", List.of(ContentWorkflowDescription.COURSELIST_WF_DESCRIPTION));
            listData.put(getIdField(), List.of(listId));
            listData.put("title", List.of(item.getLibelle() + " - Liste " + (mandatory ? "O" : "F")));
            listData.put("obligatoire", List.of(String.valueOf(mandatory)));
            listData.put("plageDeChoix", List.of("false"));
            listData.put("children", courseChildren.stream().map(e -> e.getObjetMaquette().getId().toString()).collect(Collectors.toList()));
            
            newObjects.put(listId, listData);
            nextObjects.addAll(courseChildren);
        }
    }
    
    private Map<String, List<Object>> _getObjectFields(ObjetMaquetteDetail objectDetails, EnfantsStructure structure, ObjetMaquetteStructure item)
    {
        Map<String, List<Object>> result = new HashMap<>();
        
        String contentTypeId = _getContentType(item);
        ContentWorkflowDescription wfDescription = ContentWorkflowDescription.getByContentType(contentTypeId);
        
        result.put("workflowDescription", List.of(wfDescription));
        
        Map<String, String> contentTypeMapping = _mappingByContentType.get(contentTypeId);
        
        Map<String, Object> fields = _getFields(objectDetails, structure);
        
        for (String attribute : contentTypeMapping.keySet())
        {
            String jsonKey = contentTypeMapping.get(attribute);
            Object value = fields.get(jsonKey);
            
            result.put(attribute, value != null ? List.of(value) : null); // Add the retrieved metadata values list to the contentResult
        }
        
        // Add the ID field
        result.put(getIdField(), List.of(objectDetails.getId().toString()));
        
        return result;
    }
    
    private Map<String, Object> _getFields(ObjetMaquetteDetail objectDetails, EnfantsStructure structure)
    {
        Map<String, Object> fields = new HashMap<>();
        
        fields.put("id", objectDetails.getId().toString());
        fields.put("code",  StringUtils.trimToNull(objectDetails.getCode()));
        
        DescripteursObjetMaquette descripteursObjetMaquette = objectDetails.getDescripteursObjetMaquette();
        String libelle = StringUtils.trimToNull(descripteursObjetMaquette.getLibelle());
        fields.put("libelle", libelle);
        fields.put("libelleLong", StringUtils.trimToNull(descripteursObjetMaquette.getLibelleLong()));
        
        if (descripteursObjetMaquette instanceof DescripteursFormation descripteursFormation)
        {
            fields.put("ects", descripteursFormation.getEcts());
            fields.put("orgUnit", StringUtils.trimToNull(descripteursFormation.getStructurePrincipale()));
        }
        else if (descripteursObjetMaquette instanceof DescripteursGroupement descripteursGroupement)
        {
            fields.put("obligatoire", String.valueOf(structure.getObligatoire()));
            
            PlageDeChoix plageDeChoix = descripteursGroupement.getPlageDeChoix();
            if (plageDeChoix != null)
            {
                fields.put("plageDeChoix", "true");
                fields.put("plageMin", descripteursGroupement.getPlageDeChoix().getMin());
                fields.put("plageMax", descripteursGroupement.getPlageDeChoix().getMax());
            }
        }
        else if (descripteursObjetMaquette instanceof DescripteursObjetFormation descripteursObjetFormation)
        {
            fields.put("ects", descripteursObjetFormation.getEcts());
            fields.put("orgUnit", StringUtils.trimToNull(descripteursObjetFormation.getStructurePrincipale()));
            
            String type = _getNomenclature(descripteursObjetFormation.getType());
            fields.put("typeObjetFormation", type);
        }
        
        DescripteursSyllabus syllabus = objectDetails.getDescripteursSyllabus();
        if (syllabus != null)
        {
            fields.put("description", StringUtils.trimToNull(syllabus.getDescription()));
            fields.put("langueEnseignement", StringUtils.trimToNull(syllabus.getLangueEnseignement()));
            fields.put("objectif", StringUtils.trimToNull(syllabus.getObjectif()));
            fields.put("prerequis", StringUtils.trimToNull(syllabus.getPrerequisPedagogique()));
        }
        
        DescripteursSise sise = objectDetails.getDescripteursEnquete().getDescripteursSise();
        if (sise != null)
        {
            fields.put("codeTypeDiplome", _getNomenclature(sise.getTypeDiplome()));
            fields.put("codeMention", _getNomenclature(sise.getMention()));
            fields.put("codeNiveauDiplome", _getNomenclature(sise.getNiveauDiplome()));
            fields.put("codeChampFormation", _getNomenclature(sise.getChampFormation()));
            fields.put("codeDomaineFormation", _getNomenclature(sise.getDomaineFormation()));
        }
        
        return fields;
    }
    
    private String _getNomenclature(Nomenclature nomenclature)
    {
        return nomenclature != null ? StringUtils.trimToNull(nomenclature.getCode()) : null;
    }
    
    @Override
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> importParams, Logger logger) throws Exception
    {
        Map<String, Object> parameters = putIdParameter(idValue);
        return _importOrSynchronizeContents(parameters, true, logger);
    }
    
    @Override
    public void synchronizeContent(ModifiableContent content, Logger logger) throws Exception
    {
        String idValue = content.getValue(getIdField());
        Map<String, Object> parameters = putIdParameter(idValue);
        _importOrSynchronizeContents(parameters, true, logger);
    }
    
    @Override
    protected List<ModifiableContent> _importOrSynchronizeContents(Map<String, Object> searchParams, boolean forceImport, Logger logger, ContainerProgressionTracker progressionTracker)
    {
        _importedContents = new HashMap<>();
        _synchronizedContents = new HashSet<>();
        _updatedRelationContents = new HashSet<>();
        _contentsChildren = new HashMap<>();
        
        try
        {
            _cdmfrHandler.suspendCDMFRObserver();
            
            List<ModifiableContent> contents =  super._importOrSynchronizeContents(searchParams, forceImport, logger, progressionTracker);
            _updateRelations(logger);
            _updateWorkflowStatus(logger);
            return contents;
        }
        finally
        {
            _cdmfrHandler.unsuspendCDMFRObserver(_synchronizedContents);
            
            _importedContents = null;
            _synchronizedContents = null;
            _updatedRelationContents = null;
            _contentsChildren = null;
        }
    }
    
    /**
     * Get or create the content defined by the given parameters.
     * @param lang The content language
     * @param idValue The synchronization code
     * @param remoteValues The remote values
     * @param forceImport <code>true</code> to force import (only on single import or unlimited global synchronization)
     * @param logger The logger
     * @return the content
     * @throws Exception if an error occurs
     */
    protected ModifiableContent _getOrCreateContent(String lang, String idValue, Map<String, List<Object>> remoteValues, boolean forceImport, Logger logger) throws Exception
    {
        ContentWorkflowDescription wfDescription = (ContentWorkflowDescription) remoteValues.get("workflowDescription").get(0);
        ModifiableContent content = _getContent(lang, idValue, wfDescription.getContentType());
        if (content == null && (forceImport || !synchronizeExistingContentsOnly()))
        {
            // Calculate contentTitle
            String contentTitle = Optional.of(remoteValues)
                .map(v -> v.get("title"))
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .filter(StringUtils::isNotEmpty)
                .findFirst()
                .orElse(idValue);
            
            String contentName = NameHelper.filterName(_contentPrefix + "-" + contentTitle + "-" + lang);
            
            Map<String, Object> inputs = new HashMap<>();
            String catalog = getCatalog();
            if (catalog != null)
            {
                inputs.put(AbstractCreateODFContentFunction.CONTENT_CATALOG_KEY, catalog);
            }
            
            Map<String, Object> resultMap = _contentWorkflowHelper.createContent(
                    wfDescription.getWorkflowName(),
                    wfDescription.getInitialActionId(),
                    contentName,
                    contentTitle,
                    new String[] {wfDescription.getContentType()},
                    null,
                    lang,
                    inputs);
            
            if ((boolean) resultMap.getOrDefault("error", false))
            {
                _nbError++;
            }
            
            content = (ModifiableContent) resultMap.get(Content.class.getName());

            if (content != null)
            {
                _sccHelper.updateSCCProperty(content, getId());
                
                // Set sync code
                content.setValue(getIdField(), idValue);
                
                content.saveChanges();
                _importedContents.put(content.getId(), wfDescription.getValidationActionId());
                _nbCreatedContents++;
            }
        }
        return content;
    }
    
    /**
     * Get the content from the synchronization code, the lang, the catalog and the content type.
     * @param lang The lang
     * @param syncCode The synchronization code
     * @param contentType The content type
     * @return the retrieved content
     */
    protected ModifiableContent _getContent(String lang, String syncCode, String contentType)
    {
        String xPathQuery = _getContentPathQuery(lang, syncCode, contentType, false);
        AmetysObjectIterable<ModifiableContent> contents = _resolver.query(xPathQuery);

        if (contents.getSize() > 0)
        {
            return contents.iterator().next();
        }
        
        return null;
    }
    
    @Override
    protected Optional<ModifiableContent> _importOrSynchronizeContent(String idValue, String lang, Map<String, List<Object>> remoteValues, boolean forceImport, Logger logger)
    {
        try
        {
            ModifiableContent content = _getOrCreateContent(lang, idValue, remoteValues, forceImport, logger);
            if (content != null)
            {
                return Optional.of(_synchronizeContent(content, remoteValues, logger));
            }
        }
        catch (Exception e)
        {
            _nbError++;
            logger.error("An error occurred while importing or synchronizing content", e);
        }
        
        return Optional.empty();
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected ModifiableContent _synchronizeContent(ModifiableContent content, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        super._synchronizeContent(content, remoteValues, logger);
        // Add children to the list to handle later to add relations
        if (remoteValues.containsKey("children"))
        {
            Set<String> children = _contentsChildren.computeIfAbsent(content.getId(), __ -> new LinkedHashSet<>());
            children.addAll((List<String>) (Object) remoteValues.get("children"));
        }
        return content;
    }
    
    @Override
    protected boolean _fillContent(Map<String, List<Object>> remoteValues, ModifiableContent content, Map<String, Object> additionalParameters, boolean create, Logger logger) throws Exception
    {
        _synchronizedContents.add(content.getId());
        return super._fillContent(remoteValues, content, additionalParameters, create, logger);
    }
    
    @Override
    public List<String> getLanguages()
    {
        return List.of(_odfLang);
    }

    @Override
    protected List<Expression> _getExpressionsList(String lang, String idValue, String contentType, boolean forceStrictCheck)
    {
        List<Expression> expList = super._getExpressionsList(lang, idValue, contentType, forceStrictCheck);
        String catalog = getCatalog();
        if (catalog != null)
        {
            expList.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        }
        return expList;
    }
    
    @Override
    protected Map<String, Object> _transformRemoteValuesCardinality(Map<String, List<Object>> remoteValues, String obsoleteContentTypeId)
    {
        String realContentTypeId = Optional.of(remoteValues)
                .map(v -> v.get("workflowDescription"))
                .map(l -> l.get(0))
                .map(ContentWorkflowDescription.class::cast)
                .map(ContentWorkflowDescription::getContentType)
                .orElse(null);
        return super._transformRemoteValuesCardinality(remoteValues, realContentTypeId);
    }
    
    private void _updateRelations(Logger logger)
    {
        for (String contentId : _contentsChildren.keySet())
        {
            WorkflowAwareContent content = _resolver.resolveById(contentId);
            Set<String> childrenCodes = _contentsChildren.get(contentId);
            String contentLanguage = content.getLanguage();
            String contentCatalog = content.getValue("catalog");
            Map<String, Set<String>> childrenByAttributeName = new HashMap<>();
            
            for (String childCode : childrenCodes)
            {
                Expression expression = new AndExpression(
                    _sccHelper.getCollectionExpression(getId()),
                    new StringExpression(getIdField(), Operator.EQ, childCode),
                    new StringExpression("catalog", Operator.EQ, contentCatalog),
                    new LanguageExpression(Operator.EQ, contentLanguage)
                );
                
                ModifiableContent childContent = _resolver.<ModifiableContent>query(ContentQueryHelper.getContentXPathQuery(expression))
                    .stream()
                    .findFirst()
                    .orElse(null);
                
                if (childContent == null)
                {
                    logger.warn("Content with code '{}' in {} on catalog '{}' was not found in the repository to update relations with content [{}] '{}' ({}).", childCode, contentLanguage, contentCatalog, content.getValue(getIdField()), content.getTitle(), contentCatalog);
                }
                else
                {
                    String attributesName = _getChildAttributeName(content, childContent);
                    if (attributesName != null)
                    {
                        Set<String> children = childrenByAttributeName.computeIfAbsent(attributesName, __ -> new LinkedHashSet<>());
                        children.add(childContent.getId());
                    }
                    else
                    {
                        logger.warn("The child content [{}] '{}' of type '{}' is not compatible with parent content [{}] '{}' of type '{}'.", childCode, childContent.getTitle(), childContent.getTypes()[0], content.getValue(getIdField()), content.getTitle(), content.getTypes()[0]);
                    }
                }
            }
            _updateRelations(content, childrenByAttributeName, logger);
        }
    }
    
    private String _getChildAttributeName(Content parentContent, Content childContent)
    {
        if (childContent instanceof Course && parentContent instanceof CourseList)
        {
            return CourseList.CHILD_COURSES;
        }
        
        if (parentContent instanceof Course && childContent instanceof CourseList)
        {
            return Course.CHILD_COURSE_LISTS;
        }
        
        if (parentContent instanceof TraversableProgramPart && childContent instanceof ProgramPart)
        {
            return TraversableProgramPart.CHILD_PROGRAM_PARTS;
        }
        
        return null;
    }
    
    private void _updateRelations(WorkflowAwareContent content, Map<String, Set<String>> contentRelationsByAttribute, Logger logger)
    {
        // Compute the view
        View view = View.of(content.getModel(), contentRelationsByAttribute.keySet().toArray(new String[contentRelationsByAttribute.size()]));
        
        // Compute values
        Map<String, Object> values = new HashMap<>();
        for (String attributeName : contentRelationsByAttribute.keySet())
        {
            // Add the content relations to the existing ones
            List<String> attributeValue = _getContentAttributeValue(content, attributeName);
            contentRelationsByAttribute.get(attributeName)
                .stream()
                .filter(id -> !attributeValue.contains(id))
                .forEach(attributeValue::add);
            
            values.put(attributeName, attributeValue.toArray(new String[attributeValue.size()]));
        }
        
        // Compute not synchronized contents
        Set<String> notSynchronizedContentIds = contentRelationsByAttribute.values()
                .stream()
                .flatMap(Collection::stream)
                .filter(id -> !_synchronizedContents.contains(id))
                .collect(Collectors.toSet());
        
        try
        {
            _editContent(content, Optional.of(view), values, Map.of(), false, notSynchronizedContentIds, logger);
        }
        catch (WorkflowException e)
        {
            _nbError++;
            logger.error("The content '{}' cannot be links edited (workflow action)", content, e);
        }
    }
    
    private List<String> _getContentAttributeValue(Content content, String attributeName)
    {
        return Optional.of(attributeName)
            .map(content::<ContentValue[]>getValue)
            .map(Stream::of)
            .orElseGet(Stream::empty)
            .map(ContentValue::getContentId)
            .collect(Collectors.toList());
    }
    
    private void _updateWorkflowStatus(Logger logger)
    {
        // Validate contents -> only on newly imported contents
        if (validateAfterImport())
        {
            for (String contentId : _importedContents.keySet())
            {
                WorkflowAwareContent content = _resolver.resolveById(contentId);
                Integer validationActionId = _importedContents.get(contentId);
                if (validationActionId > 0)
                {
                    validateContent(content, validationActionId, logger);
                }
            }
        }
    }
    
    @Override
    public boolean handleRightAssignmentContext()
    {
        // Rights on ODF contents are handled by ODFRightAssignmentContext
        return false;
    }
    
    @Override
    public ContentSynchronizationResult additionalCommonOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        _setPeriod(content);
        return super.additionalCommonOperations(content, additionalParameters, logger);
    }
    
    private void _setPeriod(ModifiableContent content)
    {
        if (content instanceof Container container && "semestre".equals(_resolver.<Content>resolveById(container.getNature()).getValue("code")))
        {
            String period = _getPeriodFromTitle(content);
            if (period != null)
            {
                OdfReferenceTableEntry entry = _refTableHelper.getItemFromCode(OdfReferenceTableHelper.PERIOD, period);
                if (entry != null)
                {
                    content.setExternalValue(Container.PERIOD, entry.getId());
                }
            }
        }
    }
    
    private String _getPeriodFromTitle(Content content)
    {
        return switch (content.getTitle())
        {
            case String title when Strings.CI.contains(title, "Semestre 1") || Strings.CI.contains(title, "S1") -> "s1";
            case String title when Strings.CI.contains(title, "Semestre 2") || Strings.CI.contains(title, "S2") -> "s2";
            case String title when Strings.CI.contains(title, "Semestre 3") || Strings.CI.contains(title, "S3") -> "s3";
            case String title when Strings.CI.contains(title, "Semestre 4") || Strings.CI.contains(title, "S4") -> "s4";
            case String title when Strings.CI.contains(title, "Semestre 5") || Strings.CI.contains(title, "S5") -> "s5";
            case String title when Strings.CI.contains(title, "Semestre 6") || Strings.CI.contains(title, "S6") -> "s6";
            default -> null;
        };
    }
    
    private record ComputedChildren(List<Object> childrenIds, Map<String, Map<String, List<Object>>> newObjects, List<EnfantsStructure> nextObjects) { /* empty*/ }
}
