/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.pegase.scc;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfsync.scc.operator.AbstractODFSynchronizingContentOperator;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Pegase synchronizing content operator extending {@link AbstractODFSynchronizingContentOperator} because we keep some mapping mechanisms.
 */
public class PegaseSynchronizingContentOperator extends AbstractODFSynchronizingContentOperator
{
    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    protected String getHelperRole()
    {
        return "org.ametys.plugins.odfsync.pegase.scc.operator.PegaseSynchronizingContentOperatorHelper";
    }
    
    @Override
    public Map<String, List<Object>> transform(ContentType obsoleteCType, Map<String, List<Object>> remoteValues, Logger logger)
    {
        // Get the right content type from the category field
        ContentType contentType = Optional.of(remoteValues)
            .map(v -> v.get("workflowDescription"))
            .map(l -> l.get(0))
            .map(ContentWorkflowDescription.class::cast)
            .map(ContentWorkflowDescription::getContentType)
            .map(_contentTypeEP::getExtension)
            .orElse(null);
        
        if (contentType == null)
        {
            return remoteValues;
        }
        
        _transformSpecificValues(contentType.getId(), remoteValues);
        
        return super.transform(contentType, remoteValues, logger);
    }

    private void _transformSpecificValues(String contentTypeId, Map<String, List<Object>> remoteValues)
    {
        switch (contentTypeId)
        {
            case ProgramFactory.PROGRAM_CONTENT_TYPE:
            case SubProgramFactory.SUBPROGRAM_CONTENT_TYPE:
                _transformSpecificAbstractProgramValues(remoteValues);
                break;
            default:
                // Nothing to do
        }
        
        // Always transform courseList values in case of we have courses not under course list
        _transformSpecificCourseListValues(remoteValues);
    }
    
    private void _transformSpecificAbstractProgramValues(Map<String, List<Object>> remoteValues)
    {
        // Transform ects for AbstractProgram (ECTS value is a double, transform it to String with a Long appearance, 5.0 -> "5")
        List<Object> ects = Optional.ofNullable(remoteValues.get("ects"))
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(Objects::nonNull)
                .map(Object::toString)
                .filter(StringUtils::isNotEmpty)
                .map(Double::valueOf)
                .map(Double::longValue)
                .map(String::valueOf)
                .collect(Collectors.toList());
        
        if (ects.isEmpty())
        {
            remoteValues.remove("ects");
        }
        else
        {
            remoteValues.put("ects", ects);
        }
    }
    
    private void _transformSpecificCourseListValues(Map<String, List<Object>> remoteValues)
    {
        // Set the choiceType for CourseList
        String choiceType =
            Optional.ofNullable(_getFirstValueAsString(remoteValues.remove("plageDeChoix")))
                .map(Boolean::parseBoolean)
                .orElse(false)
            ? ChoiceType.CHOICE.toString()
            : (
                Optional.ofNullable(_getFirstValueAsString(remoteValues.remove("obligatoire")))
                    .map(Boolean::parseBoolean)
                    .orElse(false)
                ? ChoiceType.MANDATORY.toString()
                : ChoiceType.OPTIONAL.toString()
            );
        
        remoteValues.put("choiceType", List.of(choiceType));
    }
    
    @Override
    protected List<Object> _transformContentAttributeValues(ContentAttributeDefinition definition, List<Object> values, Logger logger)
    {
        if (definition.getName().equals("orgUnit"))
        {
            List<Object> result = new ArrayList<>();
            
            for (Object value : values)
            {
                if (value != null)
                {
                    ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
                    StringExpression valueExpr = new StringExpression("pegaseCode", Operator.EQ, (String) value);
                    
                    String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, valueExpr));
    
                    String orgUnitId = _resolver.<Content>query(xpathQuery).stream()
                                                .findFirst()
                                                .map(Content::getId)
                                                .orElse(null);
                    
                    if (orgUnitId != null)
                    {
                        result.add(orgUnitId);
                    }
                }
            }

            return result;
        }
        
        return super._transformContentAttributeValues(definition, values, logger);
    }
}
