/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Common class for ODF virtual page representing a {@link ProgramItem}
 * @param <T> The type of factory to use
 */
public abstract class AbstractProgramItemPage<T extends AbstractOdfPageFactory> extends AbstractOdfPage<T>
{
    /**
     * The constructor of abstract program item pages
     * @param root the odf root page.
     * @param configuration The abstract virtual page configuration
     * @param scheme The scheme
     * @param factory The factory
     */
    public AbstractProgramItemPage(Page root, VirtualPageConfiguration configuration, String scheme, T factory)
    {
        super(root, configuration, scheme, factory);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }
    
    @Override
    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePages) throws AmetysRepositoryException
    {
        return getChildrenPages();
    }
    
    @Override
    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        if (index < 0)
        {
            throw new AmetysRepositoryException("Child page index cannot be negative");
        }
        
        AmetysObjectIterator<? extends Page> childPages = getChildrenPages().iterator();
        
        try
        {
            childPages.skip(index);
            return childPages.next();
        }
        catch (NoSuchElementException e)
        {
            throw new UnknownAmetysObjectException("There's no child page at index " + index + " for page " + this.getId());
        }
    }
    
    /**
     * Get the program item wrapped in this page
     * @return the program item
     */
    protected abstract ProgramItem getProgramItem();
    
    /**
     * Get the ODF path corresponding to this proram item from the root {@link Program}
     * The ODF path is composed with the ids of program, subprogram and course parents of each child pages until this page
     * @return the ODF path from the root {@link Program}
     */
    public List<ProgramItem> getOdfPath()
    {
        List<ProgramItem> path = new ArrayList<>();
        path.add(getProgramItem());
        
        Page parentPage = getParent();
     
        while (parentPage instanceof AbstractProgramItemPage programItemPage)
        {
            path.add(programItemPage.getProgramItem());
            parentPage = parentPage.getParent();
        }
        
        Collections.reverse(path);
        return path;
    }
    
    /**
     * Set the current educational paths of the wrapped program item 
     * @param programItem the wrapped program item
     */
    protected void setCurrentEducationalPaths(ProgramItem programItem) 
    {
        // Ignore education path 
        List<EducationalPath> educationalPaths = _factory._odfHelper.getEducationalPaths(programItem, true, true);
        
        List<EducationalPath> currentEducationalPaths = educationalPaths.stream()
                .filter(this::_isPartOfCurrentPath)
                .collect(Collectors.toList());
        
        if (!currentEducationalPaths.isEmpty())
        {
            if (programItem instanceof Course course)
            {
                course.setCurrentEducationalPaths(currentEducationalPaths);
            }
            else if (programItem instanceof AbstractProgram<?> abstractProgram)
            {
                abstractProgram.setCurrentEducationalPaths(currentEducationalPaths);
            }
        }
        
    }
    
    private boolean _isPartOfCurrentPath(EducationalPath path)
    {
        List<ProgramItem> currentOdfPathInSitemap = getOdfPath();
        
        // Filter on program items that is part of ODF pages
        List<ProgramItem> filteredProgramItemIds = path.getProgramItems(_factory.getResolver())
            .stream()
            .filter(pi -> pi instanceof AbstractProgram || pi instanceof Course)
            .toList();
        
        return currentOdfPathInSitemap.equals(filteredProgramItemIds);
    }

}
