/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfweb.repository;

import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.tree.OdfClassificationHandler.LevelValue;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} handling {@link SecondLevelPage}.
 */
public class SecondLevelPageFactory extends AbstractOdfPageFactory implements AmetysObjectFactory<SecondLevelPage>
{
    /**
     * Create a second level page.
     * @param root the odf root page.
     * @param level1Key the parent first level code (name)
     * @param name the page's name.
     * @param title the page's title.
     * @param firstLevelPage The parent first level page.
     * @return The <code>SecondLevelPage</code> created
     */
    public SecondLevelPage createSecondLevelPage(Page root, String level1Key, String name, String title, FirstLevelPage firstLevelPage) 
    {
        return new SecondLevelPage(root, getConfiguration(), this, level1Key, name, title, firstLevelPage);
    }
    
    @Override
    public SecondLevelPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        String[] levels = _extractLevels(id);
        String level1Code = levels[0];
        String level2Code = levels[1];
        Page rootPage = _extractRoot(id);
        
        return _findSecondLevelValueEntry(rootPage, level2Code)
                .map(entry -> _toSecondLevelPage(rootPage, level1Code, entry))
                .orElseThrow(() -> 
                    new UnknownAmetysObjectException("There's no virtual child page named " + level2Code + " for parent " + rootPage));
        
    }
    
    private Page _extractRoot(String id)
    {
        String rootId = StringUtils.substringAfter(id, "?rootId=");
        return _resolver.resolveById(rootId);
    }
    
    private String[] _extractLevels(String id)
    {
        String levelParts = StringUtils.substringBetween(id, "://", "?");
        
        return Arrays.stream(StringUtils.split(levelParts, '/'))
            .map(_odfPageHandler::decodeLevelValue)
            .toArray(String[]::new);
    }
    
    private Optional<Entry<String, LevelValue>> _findSecondLevelValueEntry(Page rootPage, String levelCode)
    {
        Map<String, LevelValue> secondLevelValues = _odfPageHandler.getLevel2Values(rootPage);
        return _findSecondLevelValueEntry(secondLevelValues, levelCode);
    }
    
    private Optional<Entry<String, LevelValue>> _findSecondLevelValueEntry(Map<String, LevelValue> secondLevelValues, String levelCode)
    {
        return secondLevelValues.entrySet().stream()
             // entry = (code, title)
            .filter(entry -> entry.getKey().equals(levelCode))
            .findFirst();
    }
    
    private SecondLevelPage _toSecondLevelPage(Page rootPage, String level1Code, Map.Entry<String, LevelValue> secondLevelValueEntry)
    {
        String level2Code = secondLevelValueEntry.getKey();
        String title = secondLevelValueEntry.getValue().getValue();
        
        return createSecondLevelPage(rootPage, level1Code, level2Code, title, null);
    }
    
    @Override
    public String getScheme()
    {
        return "odfLevel2";
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        String[] levels = _extractLevels(id);
        Page root = _extractRoot(id);
        
        return _hasSecondLevelValue(root, levels[1]);
    }
    
    private boolean _hasSecondLevelValue(Page rootPage, String level2Code) throws AmetysRepositoryException
    {
        // FIXME should use page cache, because can return true when page does
        // not exist. Maybe done like this for performance reasons?
        Map<String, LevelValue> secondLevelValues = _odfPageHandler.getLevel2Values(rootPage);
        return secondLevelValues.containsKey(level2Code);
    }
}
