/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.catalog;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.SiteManager;

/**
 * Generates ODF catalog for ODF-Web
 * See {@link org.ametys.odf.export.ExportCatalogByLevelsGenerator}
 *
 */
public class ExportCatalogByLevelsGenerator extends org.ametys.odf.export.ExportCatalogByLevelsGenerator
{
    /** The ODF Page handler */
    protected OdfPageHandler _odfPageHandler;

    /** The ODF Page resolver */
    protected OdfPageResolver _odfPageResolver;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfPageHandler = (OdfPageHandler) smanager.lookup(OdfPageHandler.ROLE);
        _odfPageResolver = (OdfPageResolver) smanager.lookup(OdfPageResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
    }
    
    @Override
    protected boolean getParameters(Map<String, String> wsParameters) throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String site = getParameter(request, "site");
        wsParameters.put("site", site);
        
        String catalog = getParameter(request, "catalog");
        wsParameters.put("catalog", catalog);
        
        String lang = getParameter(request, "lang");
        wsParameters.put("lang", lang);

        String metadataSet = request.getParameter("metadataSet");
        if (StringUtils.isBlank(metadataSet))
        {
            metadataSet = "main";
        }
        wsParameters.put("metadataSet", metadataSet);
        
        boolean isValidParameters = site != null && catalog != null && lang != null;

        if (isValidParameters)
        {
            Page odfRootPage = _odfPageHandler.getOdfRootPage(site, lang, catalog);
            if (odfRootPage == null)
            {
                XMLUtils.createElement(contentHandler, "error", "No ODF root page on website '" + site + "' for sitemap '" + lang + "' and catalog '" + catalog + "'");
                isValidParameters = false;
            }
            else
            {
                String level1 = request.getParameter("level1");
                String level2 = request.getParameter("level2");
                
                if (level1 == null || level2 == null)
                {
                    if (level1 == null)
                    {
                        level1 = _odfPageHandler.getLevel1Metadata(odfRootPage);
                    }
                    
                    if (level2 == null)
                    {
                        level2 = _odfPageHandler.getLevel2Metadata(odfRootPage);
                    }
                }
                
                if (!_odfClassificationHandler.isEligibleMetadataForLevel(level1, true))
                {
                    XMLUtils.createElement(contentHandler, "error", "The metadata " + level1 + " is not an eligible metadata for the export");
                    isValidParameters = false;
                }
                
                if (!_odfClassificationHandler.isEligibleMetadataForLevel(level2, true))
                {
                    XMLUtils.createElement(contentHandler, "error", "The metadata " + level2 + " is not an eligible metadata for the export");
                    isValidParameters = false;
                }
        
                wsParameters.put("level1", level1);
                wsParameters.put("level2", level2);
            }
        }
        
        return isValidParameters;
    }
    
    @Override
    protected AmetysObjectIterable<Program> getConcernedPrograms(Map<String, String> wsParameters)
    {
        Page odfRootPage = _odfPageHandler.getOdfRootPage(wsParameters.get("site"), wsParameters.get("lang"), wsParameters.get("catalog"));
        return _odfPageHandler.getProgramsWithRestrictions(odfRootPage, wsParameters.get("level1"), null, wsParameters.get("level2"), null, null, null);
    }

    @Override
    protected AttributesImpl getContentAttributes(ProgramItem programItem, Program parentProgram, Map<String, String> wsParameters)
    {
        String siteName = wsParameters.get("site");
        String url = _siteManager.getSite(siteName).getUrl();
        
        AttributesImpl attrs = super.getContentAttributes(programItem, parentProgram, wsParameters);
        
        String path = _odfPageResolver.getProgramItemPagePath(siteName, wsParameters.get("lang"), programItem, parentProgram);
        attrs.addCDATAAttribute("path", url + "/" + path);
        
        return attrs;
    }
}
