/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.odfweb.repository.FirstLevelPageFactory;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.CacheInvalidationPolicy;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;

/**
 * {@link Observer} for observing skills modification to invalidate cache on front-office.
 */
public class InvalidateCacheOnSkillsModifiedObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Cache invalidation policy */
    protected CacheInvalidationPolicy _cachePolicy;

    /** The zone items cache */
    protected PageElementCache _zoneItemCache;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cachePolicy = (CacheInvalidationPolicy) manager.lookup(CacheInvalidationPolicy.ROLE);
        _zoneItemCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/zoneItem");
    }
    
    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    @Override
    public boolean supports(Event event)
    {
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return _contentTypesHelper.isInstanceOf(content, SkillEditionFunction.MACRO_SKILL_TYPE);
        }
        return false;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content macroSkill = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        String catalog = macroSkill.getValue("catalog");
        
        // We do not want to computed all programs concerned by the modified skill, so just invalidate all ODF root pages and their sites
        _getODFRootPages()
            .stream()
            .filter(odfRootPage -> catalog.equals(odfRootPage.getValue(OdfPageHandler.CATALOG_DATA_NAME))) // Filter only ODF root pages using the modified macro skill catalog
            .forEach(odfRootPage -> {
                try
                {
                    _cachePolicy.invalidateCacheOnPageModification(odfRootPage); // Invalidate the cache of the concerned ODF root pages
                    _zoneItemCache.clear(null, odfRootPage.getSiteName(), "CONTENT"); // Invalidate the zone items cache of the concerned ODF root pages' site
                }
                catch (Exception e)
                {
                    getLogger().error("Error while invalidating cache for ODF root page '" + odfRootPage.getId() + "' after skill modification.", e);
                }
            });
    }
    
    /**
     * Get the ODF root pages
     * @return the ODF root pages
     */
    protected AmetysObjectIterable<Page> _getODFRootPages()
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
}
