/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Page representing a second-level page.
 */
@SuppressWarnings("unchecked")
public class SecondLevelPage extends AbstractOdfPage<SecondLevelPageFactory>
{
    private String _name;
    private String _title;
    private String _level1;
    private FirstLevelPage _level1Page;
    
    /**
     * Constructor.
     * @param factory the factory
     * @param root the odf root page.
     * @param level1Key the parent first level code (name)
     * @param name the page's name.
     * @param title the page's title.
     * @param firstLevelPage The parent first level page.
     * @param configuration The second level virtual page's configuration
     */
    public SecondLevelPage(Page root, VirtualPageConfiguration configuration, SecondLevelPageFactory factory, String level1Key, String name, String title, FirstLevelPage firstLevelPage)
    {
        super(root, configuration, factory.getScheme(), factory);
        
        _name = name;
        _title = title;
        _level1 = level1Key;
        _level1Page = firstLevelPage;
    }
    
    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + 2;
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return getParent().getTitle() + " - "  + _title;
    }

    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<ProgramPage> pages = _getChildrenFromCache()
            .orElseGet(() -> _factory.getODFPageHandler().getProgramsWithRestrictions(_root, _level1, _name, null, null))
            .stream()
            .map(this::_createProgramPage)
            .collect(Collectors.toList());
        
        return new CollectionIterable<>(pages);
    }
    
    private Optional<AmetysObjectIterable<Program>> _getChildrenFromCache()
    {
        return Optional.ofNullable(_factory.getODFPageCache().getProgramCache(_root, false))
            .map(firstLevelCache -> firstLevelCache.get(_level1))
            .map(secondLevelCache -> secondLevelCache.get(_name))
            .map(CollectionIterable<Program>::new);
    }
    
    private ProgramPage _createProgramPage(Program program)
    {
        return _factory.getProgramPageFactory().createProgramPage(_root, program, null, null, this);
    }
    
    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        return _root.getPathInSitemap() + "/" + getParent().getName() + "/" + getName();
    }

    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return (A) _factory.getODFPageCache().getChildProgramPage(_root, this, _level1, _name, path);
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return _factory.getODFPageCache().getProgramFromPageName(_root, _level1, _name, name).isPresent();
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: odfLevel2://XA/ALL?rootId=...
        return "odfLevel2://" + _factory.getODFPageHandler().encodeLevelValue(_level1) + "/" + _factory.getODFPageHandler().encodeLevelValue(_name) + "?rootId=" + _root.getId();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        // E.g: arts-lettres-langues-ALL
        return NameHelper.filterName(_title) + "-" + _factory.getODFPageHandler().encodeLevelValue(_name);
    }
    
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        if (_level1Page == null)
        {
            _level1Page = _factory.getODFPageHandler().findFirstLevelPage(_root, _level1);
        }
        
        return _level1Page; 
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return _root.getPath() + "/" + getParent().getName();
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }
}
