/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions.rules;

import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * This implementation of {@link OdfRestrictionRule} restricts ODF to a orgunit and their child orgunits
 */
public class OdfOrgunitRestrictionRule implements OdfRestrictionRule
{
    private final RootOrgUnitProvider _ouProvider;
    private final String _rootOrgUnitId;
    
    /**
     * Constructor for ODF rule restriction on a orgunit
     * @param ouProvider The orgunits provider
     * @param orgUnitId The id of orgunit
     */
    public OdfOrgunitRestrictionRule(RootOrgUnitProvider ouProvider, String orgUnitId)
    {
        _ouProvider = ouProvider;
        _rootOrgUnitId = orgUnitId;
        
        if (StringUtils.isEmpty(_rootOrgUnitId))
        {
            throw new IllegalArgumentException("At least one non-null rule is required.");
        }
    }
    
    @Override
    public Expression getExpression()
    {
        Set<String> validOrgUnits = _ouProvider.getChildOrgUnitIds(_rootOrgUnitId, true);
        validOrgUnits.add(_rootOrgUnitId);
        
        Expression[] orgUnitExprs = validOrgUnits.stream()
            .map(ou -> new StringExpression(ProgramItem.ORG_UNITS_REFERENCES, Operator.EQ, ou))
            .toArray(Expression[]::new);
            
        return new OrExpression(orgUnitExprs);
    }
    
    @Override
    public boolean contains(Program program)
    {
        Set<String> validOrgUnits = _ouProvider.getChildOrgUnitIds(_rootOrgUnitId, true);
        validOrgUnits.add(_rootOrgUnitId);
        
        return program.getOrgUnits().stream()
            .filter(StringUtils::isNotEmpty)
            .anyMatch(orgUnitId -> validOrgUnits.contains(orgUnitId));
    }
    
    @Override
    public boolean hasOrgunitRestrictions()
    {
        return true;
    }
}
