/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.advanced.TreeLeaf;
import org.ametys.cms.search.query.BooleanQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.DocumentTypeQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.solr.SearcherFactory.Searcher;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.web.frontoffice.search.fast.AbstractAutocompletionSearchServiceAction;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.model.RightCheckingMode;
import org.ametys.web.frontoffice.search.instance.model.SearchServiceCriterion;
import org.ametys.web.frontoffice.search.instance.model.SearchServiceCriterionMode;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * Get the proposed program's pages and skills for auto-completion while beginning a search
 */
public class ProgramItemAutocompletionSearchServiceAction extends AbstractAutocompletionSearchServiceAction
{
    /** The ODF page resolver */
    protected OdfPageResolver _odfPageResolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _odfPageResolver = (OdfPageResolver) serviceManager.lookup(OdfPageResolver.ROLE);
    }
    
    @Override
    protected Map<String, Object> _searchAmetysObject(Site site, String zoneItemId, String lang, String escapedQuery, int limit, RightCheckingMode rightCheckingMode) throws Exception
    {
        Map<String, Object> results = super._searchAmetysObject(site, zoneItemId, lang, escapedQuery, limit, rightCheckingMode);
        
        if (ODFSkillsHelper.isSkillsEnabled())
        {
            List<String> catalogNames = _getCatalogNames(zoneItemId);
            
            // Search on skill's title
            AmetysObjectIterable<Content> skillResults = _getSkills(escapedQuery, catalogNames, lang, limit, rightCheckingMode);
            List<Map<String, Object>> skills = skillResults.stream()
                    .map(this::_getContentHit)
                    .collect(Collectors.toList());
            
            results.put("skills", skills);
        }
        
        return results;
    }
    
    @Override
    protected void setHitsInResults(Map<String, Object> results, AmetysObjectIterable<AmetysObject> searchHits, Site site, String zoneItemId, String lang)
    {
        List<String> catalogNames = _getCatalogNames(zoneItemId);
        
        // Compute common ODF root page if search is configured for one catalog
        Page odfRootPage = catalogNames.size() == 1 ? _odfPageResolver.getOdfRootPage(site.getName(), lang, catalogNames.get(0)) : null;
        
        List<Map<String, Object>> pages = searchHits.stream()
                .filter(Content.class::isInstance)
                .map(Content.class::cast)
                .map(c -> _getPageHit(c, site.getName(), odfRootPage))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        results.put("pages", pages);
    }
    
    /**
     * Get the configured catalog in criterion definitions
     * @param zoneItemId The id of zone item
     * @return the catalog's name
     */
    protected List<String> _getCatalogNames(String zoneItemId)
    {
        if (StringUtils.isNotEmpty(zoneItemId))
        {
            SearchServiceInstance serviceInstance = _searchServiceInstanceManager.get(zoneItemId);
            
            List<String> catalogValues = serviceInstance.getCriterionTree()
                                                        .map(AbstractTreeNode::getFlatLeaves)
                                                        .orElseGet(Collections::emptyList)
                                                        .stream()
                                                        .map(TreeLeaf::getValue)
                                                        .filter(c -> c.getMode() == SearchServiceCriterionMode.STATIC)
                                                        .filter(c -> Strings.CS.endsWith(c.getCriterionDefinition().getName(), "$org.ametys.plugins.odf.Content.programItem$catalog"))
                                                        .map(SearchServiceCriterion::getStaticValue)
                                                        .flatMap(Optional::stream)
                                                        .map(List.class::cast)
                                                        .flatMap(List::stream)
                                                        .map(String.class::cast)
                                                        .toList();
            
            return catalogValues;
        }
        else
        {
            return List.of();
        }
        
    }
    
    /**
     * Get the skills contents matching the query
     * @param escapedQuery the query
     * @param catalogNames The catalog names
     * @param lang the language
     * @param limit the max number of results
     * @param rightCheckingMode the mode for checking rights
     * @return the matching contents
     * @throws Exception if an error occurred during search
     */
    protected AmetysObjectIterable<Content> _getSkills(String escapedQuery, List<String> catalogNames, String lang, int limit, RightCheckingMode rightCheckingMode) throws Exception
    {
        // Search for skills in the given catalog(s)
        Query catalogQuery = catalogNames.stream()
                .map(c -> new StringQuery("catalog", c))
                .collect(OrQuery.collector());
        
        // Search for skills that are not orphans (either transversal or linked to a program)
        BooleanQuery transversal = new BooleanQuery("transversal", true);
        StringQuery programQuery = new StringQuery("parentProgram");
        OrQuery orphanSkill = new OrQuery(transversal, programQuery);
        
        Searcher searcher = _searcherFactory.create()
                .withQuery(new StringQuery(SolrFieldNames.TITLE, Operator.SEARCH, escapedQuery, lang, true))
                .addFilterQuery(new DocumentTypeQuery(SolrFieldNames.TYPE_CONTENT))
                .addFilterQuery(new ContentTypeQuery(SkillEditionFunction.MACRO_SKILL_TYPE))
                .addFilterQuery(catalogQuery)
                .addFilterQuery(orphanSkill)
                .withLimits(0, limit);
        
        _setRightCheckingMode(searcher, rightCheckingMode);

        return searcher.search();
    }
    
    /**
     * Get the JSON representation of a page hit
     * @param content the content
     * @param siteName the current site name
     * @param odfRootPage the ODF root page. Can be null if search is configured for multiple catalogs
     * @return the page as json
     */
    protected Map<String, Object> _getPageHit(Content content, String siteName, Page odfRootPage)
    {
        Page page = null;
        if (content instanceof Program)
        {
            page = odfRootPage != null ? _odfPageResolver.getProgramPage(odfRootPage, (Program) content) : _odfPageResolver.getProgramPage((Program) content, siteName);
        }
        else if (content instanceof SubProgram)
        {
            page = odfRootPage != null ? _odfPageResolver.getSubProgramPage(odfRootPage, (SubProgram) content, null) : _odfPageResolver.getSubProgramPage((SubProgram) content, null, siteName);
        }
        else if (content instanceof Course)
        {
            page = odfRootPage != null ? _odfPageResolver.getCoursePage(odfRootPage, (Course) content, (AbstractProgram) null) : _odfPageResolver.getCoursePage((Course) content, (AbstractProgram) null, siteName);
        }
        
        if (page != null)
        {
            Map<String, Object> result = new HashMap<>();
            result.put("title", page.getTitle());
            
            RenderingContext context = _renderingContextHandler.getRenderingContext();
            if (!(context == RenderingContext.BACK))
            {
                StringBuilder uri = new StringBuilder();
                uri.append(_prefixHandler.getUriPrefix(siteName));
                uri.append("/");
                uri.append(page.getSitemapName() + "/" + page.getPathInSitemap() + ".html");
                
                result.put("url", URIUtils.encodePath(uri.toString()));
            }
            else // back
            {
                result.put("url", "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-page', {id:'" + page.getId() + "'});})()");
            }
            
            return result;
        }
        
        return null;
    }
    
    /**
     * Get the JSON representation of a content hit
     * @param content the content
     * @return the content as json
     */
    protected Map<String, Object> _getContentHit(Content content)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("title", content.getTitle());
        result.put("id", content.getId());
        
        return result;
    }

}
