<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2020 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0" 
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exslt="http://exslt.org/common"
                xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                exclude-result-prefixes="i18n exslt">
    
    <!-- +
         | Template to display the skills distribution table over years and/or semesters and courses
         + -->
    <xsl:template name="skills-table">
        <xsl:param name="skills"/>
        <xsl:param name="years"/>
        <xsl:param name="semesters" select="$years/container[attributes/nature/@code='semestre' and not(@excludedFromSkills)]"/>
        <xsl:param name="courses" select="$semesters/courselist/course"/>
        <xsl:param name="tableCls">ametys-skills-table</xsl:param>
        <xsl:param name="additionalCls"/>
        
        <xsl:variable name="higherLevelNature">
            <xsl:choose>
                <xsl:when test="$years">annee</xsl:when>
                <xsl:when test="$semesters">semestre</xsl:when>
            </xsl:choose>
        </xsl:variable>
        
        <table class="{$tableCls} {$additionalCls}">
            <thead>
                <xsl:if test="$years">
                    <tr>
                        <th class="{$tableCls}__year-header" colspan="2"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HEADER_YEAR" i18n:catalogue="plugin.odf"/></th>
                        <xsl:apply-templates select="$years" mode="header-year"/>
                    </tr>
                </xsl:if>
                <xsl:choose>
                    <xsl:when test="$semesters">
                        <tr>
	                        <th class="{$tableCls}__semester-header" colspan="2"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HEADER_SEMESTER" i18n:catalogue="plugin.odf"/></th>
	                        <xsl:apply-templates select="$semesters" mode="header-semester">
                                <xsl:with-param name="tableCls" select="$tableCls"/>
                                <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
                            </xsl:apply-templates>
	                    </tr>
                        <tr>
		                    <th class="{$tableCls}__course-header" colspan="2"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HEADER_COURSE" i18n:catalogue="plugin.odf"/></th>
		                    <xsl:apply-templates select="$courses" mode="header-course">
                                <xsl:with-param name="tableCls" select="$tableCls"/>
                                <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
                            </xsl:apply-templates>
		                </tr>
                    </xsl:when>
	                <xsl:otherwise>
                        <tr class="{$tableCls}__course-header">
                            <th colspan="2"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HEADER_COURSE" i18n:catalogue="plugin.odf"/></th>
                            <xsl:apply-templates select="$courses" mode="header-course">
                                <xsl:with-param name="tableCls" select="$tableCls"/>
                                <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
                            </xsl:apply-templates>
                        </tr>
                    </xsl:otherwise>
                </xsl:choose>
            </thead>
            
            <tbody>
                <xsl:apply-templates select="$skills/skillSet" mode="body-row">
                    <xsl:with-param name="tableCls" select="$tableCls"/>
                    <xsl:with-param name="courses" select="$courses"/>
                    <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
                </xsl:apply-templates>
            </tbody>
        </table>  
        
        <xsl:choose>
            <xsl:when test="$years">
                <xsl:call-template name="years-hidden-columns"/>
            </xsl:when>
            <xsl:when test="$semesters">
                <xsl:call-template name="semesters-hidden-columns"/>
            </xsl:when>
        </xsl:choose>
        
        <xsl:call-template name="hidden-columns-js"/>
         
    </xsl:template>
    
    <xsl:template match="container" mode="header-year">
        <xsl:param name="tableCls">ametys-skills-table</xsl:param>
        
        <xsl:variable name="nbCourses" select="count(container[not(@excludedFromSkills)]/courselist/course|courselist/course)"/>
        
        <xsl:if test="$nbCourses > 0">
            <th class="{$tableCls}__year-header" colspan="{$nbCourses}" data-apc-collapse-col="{@code}">
                <span class="{$tableCls}__collapse-control" data-apc-collapse-control="true" data-apc-collapse-target="{@code}" data-apc-collapse-order="{position()}" title="plugin.odf:PLUGINS_ODF_SKILLS_TABLE_COLLAPSE" i18n:attr="title"></span>
                <xsl:value-of select="@title"/>
            </th>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="container" mode="header-semester">
        <xsl:param name="tableCls">ametys-skills-table</xsl:param>
        <xsl:param name="higherLevelNature"/>
        
        <xsl:variable name="nbCourses" select="count(courselist/course)"/>
        
        <xsl:if test="$nbCourses > 0">
            <th class="{$tableCls}__semester-header" colspan="{$nbCourses}">
                <xsl:choose>
                    <xsl:when test="$higherLevelNature = 'semestre'">
                        <!-- semester is collapsible -->
                        <xsl:attribute name="data-apc-collapse-col"><xsl:value-of select="@code"/></xsl:attribute>
                        <span class="{$tableCls}__collapse-control apc-collapse-control semester" data-apc-collapse-control="true" data-apc-collapse-target="{@code}" data-apc-collapse-order="{position()}" title="plugin.odf:PLUGINS_ODF_SKILLS_TABLE_COLLAPSE" i18n:attr="title"></span>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- year is collapsible -->
                        <xsl:attribute name="data-apc-collapse-col"><xsl:value-of select="ancestor::container[attributes/nature/@code=$higherLevelNature]/@code"/></xsl:attribute>
                    </xsl:otherwise>
                </xsl:choose>
                <div><xsl:value-of select="@title"/></div>
            </th>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="course" mode="header-course">
        <xsl:param name="tableCls">ametys-skills-table</xsl:param>
        <xsl:param name="higherLevelNature"/>
        
        <th class="{$tableCls}__course-header">
            <xsl:if test="$higherLevelNature">
                <xsl:attribute name="data-apc-collapse-col"><xsl:value-of select="ancestor::container[attributes/nature/@code=$higherLevelNature]/@code"/></xsl:attribute>
            </xsl:if>
            <div><xsl:value-of select="@title"/></div>
        </th>
    </xsl:template>
    
    <xsl:template match="skillSet" mode="body-row">
        <xsl:param name="tableCls"/>
        <xsl:param name="courses"/>
        <xsl:param name="higherLevelNature"/>
        
        <xsl:apply-templates select="skill[1]" mode="body-row-first">
            <xsl:with-param name="tableCls" select="$tableCls"/>
            <xsl:with-param name="courses" select="$courses"/>
            <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="skill[position() &gt; 1]" mode="body-row">
            <xsl:with-param name="tableCls" select="$tableCls"/>
            <xsl:with-param name="courses" select="$courses"/>
            <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="skill" mode="body-row-first">
        <xsl:param name="tableCls"/>
        <xsl:param name="courses"/>
        <xsl:param name="higherLevelNature"/>
        
        <tr>
            <th class="{$tableCls}__skillset-header" rowspan="{count(../skill)}"><xsl:value-of select="../@title"/></th>
            <th class="{$tableCls}__skill-header" ><xsl:value-of select="@title"/></th>
            
            <xsl:apply-templates select="$courses" mode="table-cell">
                <xsl:with-param name="skill" select="."/>
                <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
            </xsl:apply-templates>
        </tr>
    </xsl:template>
    
    <xsl:template match="skill" mode="body-row">
        <xsl:param name="tableCls"/>
        <xsl:param name="courses"/>
        <xsl:param name="higherLevelNature"/>
        
        <tr>
            <th class="{$tableCls}__skill-header"><xsl:value-of select="@title"/></th>
            
            <xsl:apply-templates select="$courses" mode="table-cell">
                <xsl:with-param name="skill" select="."/>
                <xsl:with-param name="higherLevelNature" select="$higherLevelNature"/>
            </xsl:apply-templates>
        </tr>
    </xsl:template>
    
    <xsl:template match="course" mode="table-cell">
        <xsl:param name="skill"/>
        <xsl:param name="higherLevelNature"/>
        
        <td>
            <xsl:if test="$higherLevelNature">
                <xsl:attribute name="data-apc-collapse-col"><xsl:value-of select="ancestor::container[attributes/nature/@code=$higherLevelNature]/@code"/></xsl:attribute>
            </xsl:if>
            <xsl:if test="$skill/course[@id = current()/@id]">
                <xsl:attribute name="class">no-level</xsl:attribute>
                <span><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_NO_LEVEL" i18n:catalogue="plugin.odf"/></span>
            </xsl:if>
        </td>
    </xsl:template>
    
    <xsl:template name="years-hidden-columns">
        <div class="hiddenColumns">
            <div class="hiddenColumns__title"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HIDDEN_YEARS" i18n:catalogue="plugin.odf"/></div>
            <ul class="hiddenColumns__list"></ul>
        </div>
    </xsl:template>
    
    <xsl:template name="semesters-hidden-columns">
        <div class="hiddenColumns">
            <div class="hiddenColumns__title"><i18n:text i18n:key="PLUGINS_ODF_SKILLS_TABLE_HIDDEN_SEMESTERS" i18n:catalogue="plugin.odf"/></div>
            <ul class="hiddenColumns__list semester"></ul>
        </div>
    </xsl:template>
    
    <xsl:template name="hidden-columns-js">
        <script type="text/javascript">
            function apcShowColumn(target, e) {
                $j(`[data-apc-collapse-col="${target}"]`).show();
                $j(e).remove();
                apcCountHiddensColumns();
            }
            
            function apcHideColumn(target, e, position)
            {
                $j(`[data-apc-collapse-col="${target}"]`).hide();
                
                // Add as masked colmun
                const nameColumn = $j(e).parent().text();
                $j('.hiddenColumns__list').append(`&lt;li data-col-order="${position}" onclick="apcShowColumn('${target}', this)"&gt;${nameColumn}&lt;/li&gt;`);
                
                // reorder li
                $(".hiddenColumns__list > li").sort(function(a, b) {
                    
                    var a1 = $j(a).data('col-order'),
                        b1 = $j(b).data('col-order');
             
                    if(a1 > b1) {
                        return 1;
                    }
                    if(a1 &lt; b1) {
                        return -1;
                    }
                    return 0;
                }).prependTo("ul.hiddenColumns__list");
                
                apcCountHiddensColumns();
            }
            
            function apcCountHiddensColumns() {
                const countHiddenColumns = $j('.hiddenColumns').find('li').length;
                if(countHiddenColumns == 0) {
                    $j('.hiddenColumns').removeClass('fadeIn');
                } else {
                    $j('.hiddenColumns').addClass('fadeIn');
                }
            }
            
            $j('[data-apc-collapse-control]').click(function () {
              var target = $j(this).data('apc-collapse-target');
              var position = $j(this).data('apc-collapse-order');
              apcHideColumn(target, this, position);
            })
        </script>
    </xsl:template>
    
</xsl:stylesheet>