/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.content.references.OutgoingReferences;
import org.ametys.cms.content.references.OutgoingReferencesExtractor;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.odf.content.code.UniqueCodeGenerator;
import org.ametys.odf.content.code.UniqueCodeGeneratorExtensionPoint;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseContainer;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.data.type.EducationalPathRepositoryElementType;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.AbstractTraversableProgramPart;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectExistsException;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Helper for ODF contents
 *
 */
public class ODFHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable, Initializable
{
    /** The component role. */
    public static final String ROLE = ODFHelper.class.getName();
    
    /** Request attribute to get the "Live" version of contents */
    public static final String REQUEST_ATTRIBUTE_VALID_LABEL = "live-version";
    
    /** The default id of initial workflow action */
    protected static final int __INITIAL_WORKFLOW_ACTION_ID = 0;
    
    private static final String __ANCESTORS_CACHE = ODFHelper.class.getName() + "$ancestors";
    
    private static final String __PARENT_PROGRAM_ITEMS_CACHE = ODFHelper.class.getName() + "$parentProgramItems";
    
    private static final String __ODF_PLUGIN_NAME = "odf";
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The content types manager */
    protected ContentTypeExtensionPoint _cTypeEP;
    /** Root orgunit */
    protected RootOrgUnitProvider _ouRootProvider;
    /** Helper for shareable course */
    protected ShareableCourseHelper _shareableCourseHelper;
    /** The Avalon context */
    protected Context _context;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The content helper */
    protected ContentHelper _contentHelper;
    /** The outgoing references extractor */
    protected OutgoingReferencesExtractor _outgoingReferencesExtractor;
    /** The content DAO **/
    protected ContentDAO _contentDAO;
    /** The helper for reference tables from ODF */
    protected OdfReferenceTableHelper _refTableHelper;
    /** The right manager */
    protected RightManager _rightManager;
    
    private UniqueCodeGeneratorExtensionPoint _codeGeneratorEP;
    private Optional<String> _yearId = Optional.empty();

    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _ouRootProvider = (RootOrgUnitProvider) manager.lookup(RootOrgUnitProvider.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _outgoingReferencesExtractor = (OutgoingReferencesExtractor) manager.lookup(OutgoingReferencesExtractor.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _codeGeneratorEP = (UniqueCodeGeneratorExtensionPoint) manager.lookup(UniqueCodeGeneratorExtensionPoint.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__ANCESTORS_CACHE,
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ODF_ANCESTORS_LABEL"),
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ODF_ANCESTORS_DESCRIPTION"),
                false);
        
        _cacheManager.createRequestCache(__PARENT_PROGRAM_ITEMS_CACHE,
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ODF_PARENT_PROGRAM_ITEMS_LABEL"),
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ODF_PARENT_PROGRAM_ITEMS_DESCRIPTION"),
                false);
    }
    
    /**
     * Determines if legacy sax of program structure should be used
     * @param programItem the program item beeing rendering
     * @return true if legacy sax of program structure should be used
     */
    public boolean useLegacyProgramStructure(ProgramItem programItem)
    {
        return true;
    }
    
    /**
     * Generate an unique code for a given ODF content type
     * @param cTypeId The id of ODF content type
     * @return The unique code
     */
    public synchronized String generateUniqueCode(String cTypeId)
    {
        String codeGenId = Config.getInstance().getValue("odf.code.generator", true, "odf-code-incremental");
        UniqueCodeGenerator codeGen = _codeGeneratorEP.getExtension(codeGenId);
        
        return codeGen.generateUniqueCode(cTypeId);
    }
    
    /**
     * Gets the root for ODF contents
     * @return the root for ODF contents
     */
    public AmetysObjectCollection getRootContent()
    {
        return getRootContent(false);
    }
    
    /**
     * Gets the root for ODF contents
     * @param create <code>true</code> to create automatically the root when missing.
     * @return the root for ODF contents
     */
    public AmetysObjectCollection getRootContent(boolean create)
    {
        ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins/");
        
        boolean needSave = false;
        if (!pluginsNode.hasChild(__ODF_PLUGIN_NAME))
        {
            if (create)
            {
                pluginsNode.createChild(__ODF_PLUGIN_NAME, "ametys:unstructured");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/" + __ODF_PLUGIN_NAME + "' is missing");
            }
        }
        
        ModifiableTraversableAmetysObject pluginNode = pluginsNode.getChild(__ODF_PLUGIN_NAME);
        if (!pluginNode.hasChild(RepositoryConstants.NAMESPACE_PREFIX + ":contents"))
        {
            if (create)
            {
                pluginNode.createChild(RepositoryConstants.NAMESPACE_PREFIX + ":contents", "ametys:collection");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/" + __ODF_PLUGIN_NAME + "/ametys:contents' is missing");
            }
        }
        
        if (needSave)
        {
            pluginsNode.saveChanges();
        }
        
        return pluginNode.getChild(RepositoryConstants.NAMESPACE_PREFIX + ":contents");
    }
    
    /**
     * Get the {@link ProgramItem}s matching the given arguments
     * @param cTypeId The id of content type. Can be null to get program's items whatever their content type.
     * @param code The code. Can be null to get program's items regardless of their code
     * @param catalogName The search catalog. Can be null to get program's items regardless the catalog they belong to.
     * @param lang The search language. Can be null to get program's items regardless of their language
     * @param <C> The content return type
     * @return The matching program items
     */
    public <C extends Content> AmetysObjectIterable<C> getProgramItems(String cTypeId, String code, String catalogName, String lang)
    {
        return getProgramItems(cTypeId, code, catalogName, lang, null, null);
    }
    
    /**
     * Get the {@link ProgramItem}s matching the given arguments
     * @param cTypeIds The id of content types. Can be empty to get program's items whatever their content type.
     * @param code The code. Can be null to get program's items regardless of their code
     * @param catalogName The search catalog. Can be null to get program's items regardless the catalog they belong to.
     * @param lang The search language. Can be null to get program's items regardless of their language
     * @param <C> The content return type
     * @return The matching program items
     */
    public <C extends Content> AmetysObjectIterable<C> getProgramItems(Collection<String> cTypeIds, String code, String catalogName, String lang)
    {
        return getProgramItems(cTypeIds, code, catalogName, lang, null, null);
    }
    
    /**
     * Get the {@link ProgramItem}s matching the given arguments
     * @param cTypeId The id of content type. Can be null to get program's items whatever their content type.
     * @param code The code. Can be null to get program's items regardless of their code
     * @param catalogName The search catalog. Can be null to get program's items regardless the catalog they belong to.
     * @param lang The search language. Can be null to get program's items regardless of their language
     * @param additionnalExpr An additional expression for filtering result. Can be null
     * @param sortCriteria criteria for sorting results
     * @param <C> The content return type
     * @return The matching program items
     */
    public <C extends Content> AmetysObjectIterable<C> getProgramItems(String cTypeId, String code, String catalogName, String lang, Expression additionnalExpr, SortCriteria sortCriteria)
    {
        return getProgramItems(cTypeId != null ? Collections.singletonList(cTypeId) : Collections.EMPTY_LIST, code, catalogName, lang, additionnalExpr, sortCriteria);
    }
    
    /**
     * Get the {@link ProgramItem}s matching the given arguments
     * @param cTypeIds The id of content types. Can be empty to get program's items whatever their content type.
     * @param code The code. Can be null to get program's items regardless of their code
     * @param catalogName The search catalog. Can be null to get program's items regardless the catalog they belong to.
     * @param lang The search language. Can be null to get program's items regardless of their language
     * @param additionnalExpr An additional expression for filtering result. Can be null
     * @param sortCriteria criteria for sorting results
     * @param <C> The content return type
     * @return The matching program items
     */
    public <C extends Content> AmetysObjectIterable<C> getProgramItems(Collection<String> cTypeIds, String code, String catalogName, String lang, Expression additionnalExpr, SortCriteria sortCriteria)
    {
        List<Expression> exprs = new ArrayList<>();
        
        if (!cTypeIds.isEmpty())
        {
            exprs.add(new ContentTypeExpression(Operator.EQ, cTypeIds.toArray(new String[cTypeIds.size()])));
        }
        if (StringUtils.isNotEmpty(code))
        {
            exprs.add(new StringExpression(ProgramItem.CODE, Operator.EQ, code));
        }
        if (StringUtils.isNotEmpty(catalogName))
        {
            exprs.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalogName));
        }
        if (StringUtils.isNotEmpty(lang))
        {
            exprs.add(new LanguageExpression(Operator.EQ, lang));
        }
        if (additionnalExpr != null)
        {
            exprs.add(additionnalExpr);
        }
        
        Expression expr = new AndExpression(exprs.toArray(new Expression[exprs.size()]));
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr, sortCriteria);
        return _resolver.query(xpathQuery);
    }
    
    /**
     * Get the equivalent {@link CoursePart} of the source {@link CoursePart} in given catalog and language
     * @param srcCoursePart The source course part
     * @param catalogName The name of catalog to search into
     * @param lang The search language
     * @return The equivalent program item or <code>null</code> if not exists
     */
    public CoursePart getCoursePart(CoursePart srcCoursePart, String catalogName, String lang)
    {
        return getODFContent(CoursePartFactory.COURSE_PART_CONTENT_TYPE, srcCoursePart.getCode(), catalogName, lang);
    }
    
    /**
     * Get the equivalent {@link ProgramItem} of the source {@link ProgramItem} in given catalog and language
     * @param <T> The type of returned object, it have to be a subclass of {@link ProgramItem}
     * @param srcProgramItem The source program item
     * @param catalogName The name of catalog to search into
     * @param lang The search language
     * @return The equivalent program item or <code>null</code> if not exists
     */
    public <T extends ProgramItem> T getProgramItem(T srcProgramItem, String catalogName, String lang)
    {
        return getODFContent(((Content) srcProgramItem).getTypes()[0], srcProgramItem.getCode(), catalogName, lang);
    }
    
    /**
     * Get the equivalent {@link Content} having the same code in given catalog and language
     * @param <T> The type of returned object, it have to be a subclass of {@link AmetysObject}
     * @param contentType The content type to search for
     * @param odfContentCode The code of the ODF content
     * @param catalogName The name of catalog to search into
     * @param lang The search language
     * @return The equivalent content or <code>null</code> if not exists
     */
    public <T extends AmetysObject> T getODFContent(String contentType, String odfContentCode, String catalogName, String lang)
    {
        Expression contentTypeExpr = new ContentTypeExpression(Operator.EQ, contentType);
        Expression langExpr = new LanguageExpression(Operator.EQ, lang);
        Expression catalogExpr = new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalogName);
        Expression codeExpr = new StringExpression(ProgramItem.CODE, Operator.EQ, odfContentCode);
        
        Expression expr = new AndExpression(contentTypeExpr, langExpr, catalogExpr, codeExpr);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        AmetysObjectIterable<T> contents = _resolver.query(xpathQuery);
        AmetysObjectIterator<T> contentsIt = contents.iterator();
        if (contentsIt.hasNext())
        {
            return contentsIt.next();
        }
        
        return null;
    }
    
    /**
     * Get the child program items of a {@link ProgramItem}
     * @param programItem The program item
     * @return The child program items
     */
    public List<ProgramItem> getChildProgramItems(ProgramItem programItem)
    {
        List<ProgramItem> children = new ArrayList<>();
        
        if (programItem instanceof TraversableProgramPart programPart)
        {
            children.addAll(programPart.getProgramPartChildren());
        }
        
        if (programItem instanceof CourseContainer courseContainer)
        {
            children.addAll(courseContainer.getCourses());
        }
        
        if (programItem instanceof Course course)
        {
            children.addAll(course.getCourseLists());
        }
        
        return children;
    }

    /**
     * Get the child subprograms of a {@link ProgramPart}
     * @param programPart The program part
     * @return The child subprograms
     */
    public Set<SubProgram> getChildSubPrograms(ProgramPart programPart)
    {
        Set<SubProgram> subPrograms = new HashSet<>();
        
        if (programPart instanceof TraversableProgramPart traversableProgram)
        {
            if (programPart instanceof SubProgram subProgram)
            {
                subPrograms.add(subProgram);
            }
            traversableProgram.getProgramPartChildren().forEach(child -> subPrograms.addAll(getChildSubPrograms(child)));
        }

        return subPrograms;
    }
    
    /**
     * Gets (recursively) parent containers of this program item.
     * @param programItem The program item
     * @return parent containers of this program item.
     */
    public Set<Container> getParentContainers(ProgramItem programItem)
    {
        return getParentContainers(programItem, false);
    }
    
    /**
     * Gets (recursively) parent containers of this program item.
     * @param programItem The program item
     * @param continueIfFound If <code>true</code> continue searching corresponding parents in the parent structure, otherwise only closest items are returned.
     * @return parent containers of this program item.
     */
    public Set<Container> getParentContainers(ProgramItem programItem, boolean continueIfFound)
    {
        return _getParentsOfType(programItem, Container.class, continueIfFound);
    }

    /**
     * Gets (recursively) parent programs of this course part.
     * @param coursePart The course part
     * @return parent programs of this course part.
     */
    public Set<Program> getParentPrograms(CoursePart coursePart)
    {
        Set<Program> programs = new HashSet<>();
        for (Course course : coursePart.getCourses())
        {
            programs.addAll(getParentPrograms(course));
        }
        return programs;
    }
    
    /**
     * Gets (recursively) parent programs of this program item.
     * @param programItem The program item
     * @return parent programs of this program item.
     */
    public Set<Program> getParentPrograms(ProgramItem programItem)
    {
        return getParentPrograms(programItem, false);
    }
    
    /**
     * Gets (recursively) parent programs of this program item.
     * @param programItem The program item
     * @param includingItself <code>true</code> to return the program item if it's a {@link Program}.
     * @return parent programs of this program item.
     */
    public Set<Program> getParentPrograms(ProgramItem programItem, boolean includingItself)
    {
        if (includingItself && programItem instanceof Program program)
        {
            return Set.of(program);
        }
        
        return _getParentsOfType(programItem, Program.class, false);
    }

    /**
     * Gets (recursively) parent subprograms of this course part.
     * @param coursePart The course part
     * @return parent subprograms of this course part.
     */
    public Set<SubProgram> getParentSubPrograms(CoursePart coursePart)
    {
        return getParentSubPrograms(coursePart, false);
    }

    /**
     * Gets (recursively) parent subprograms of this course part.
     * @param coursePart The course part
     * @param continueIfFound If <code>true</code> continue searching corresponding parents in the parent structure, otherwise only closest items are returned.
     * @return parent subprograms of this course part.
     */
    public Set<SubProgram> getParentSubPrograms(CoursePart coursePart, boolean continueIfFound)
    {
        Set<SubProgram> abstractPrograms = new HashSet<>();
        for (Course course : coursePart.getCourses())
        {
            abstractPrograms.addAll(getParentSubPrograms(course, continueIfFound));
        }
        return abstractPrograms;
    }
    
    /**
     * Gets (recursively) parent subprograms of this program item.
     * @param programItem The program item
     * @return parent subprograms of this program item.
     */
    public Set<SubProgram> getParentSubPrograms(ProgramItem programItem)
    {
        return getParentSubPrograms(programItem, false);
    }
    
    /**
     * Gets (recursively) parent subprograms of this program item.
     * @param programItem The program item
     * @param continueIfFound If <code>true</code> continue searching corresponding parents in the parent structure, otherwise only closest items are returned.
     * @return parent subprograms of this program item.
     */
    public Set<SubProgram> getParentSubPrograms(ProgramItem programItem, boolean continueIfFound)
    {
        return _getParentsOfType(programItem, SubProgram.class, continueIfFound);
    }

    /**
     * Gets (recursively) parent abstract programs of this course part.
     * @param coursePart The course part
     * @return parent abstract programs of this course part.
     */
    public Set<AbstractProgram> getParentAbstractPrograms(CoursePart coursePart)
    {
        return getParentAbstractPrograms(coursePart, false);
    }

    /**
     * Gets (recursively) parent abstract programs of this course part.
     * @param coursePart The course part
     * @param continueIfFound If <code>true</code> continue searching corresponding parents in the parent structure, otherwise only closest items are returned.
     * @return parent abstract programs of this course part.
     */
    public Set<AbstractProgram> getParentAbstractPrograms(CoursePart coursePart, boolean continueIfFound)
    {
        Set<AbstractProgram> abstractPrograms = new HashSet<>();
        for (Course course : coursePart.getCourses())
        {
            abstractPrograms.addAll(getParentAbstractPrograms(course, continueIfFound));
        }
        return abstractPrograms;
    }
    
    /**
     * Gets (recursively) parent abstract programs of this program item.
     * @param programItem The program item
     * @return parent abstract programs of this program item.
     */
    public Set<AbstractProgram> getParentAbstractPrograms(ProgramItem programItem)
    {
        return getParentAbstractPrograms(programItem, false);
    }
    
    /**
     * Gets (recursively) parent abstract programs of this program item.
     * @param programItem The program item
     * @param continueIfFound If <code>true</code> continue searching corresponding parents in the parent structure, otherwise only closest items are returned.
     * @return parent abstract programs of this program item.
     */
    public Set<AbstractProgram> getParentAbstractPrograms(ProgramItem programItem, boolean continueIfFound)
    {
        return _getParentsOfType(programItem, AbstractProgram.class, continueIfFound);
    }

    private <T> Set<T> _getParentsOfType(ProgramItem programItem, Class<T> classToTest, boolean continueIfFound)
    {
        Set<ProgramItem> visitedProgramItems = new HashSet<>();
        visitedProgramItems.add(programItem);
        return _getParentsOfType(programItem, visitedProgramItems, classToTest, continueIfFound);
    }
    
    @SuppressWarnings("unchecked")
    private <T> Set<T> _getParentsOfType(ProgramItem programItem, Set<ProgramItem> visitedProgramItems, Class<T> classToTest, boolean continueIfFound)
    {
        Set<T> parentsOfType = new HashSet<>();
        List<ProgramItem> parents = getParentProgramItems(programItem);
        
        for (ProgramItem parent : parents)
        {
            // Only parents not already visited
            if (visitedProgramItems.add(parent))
            {
                // Cast to Content if instance of Content instead of another type (for structures containing both Container and SubProgram)
                boolean found = false;
                if (classToTest.isInstance(parent))
                {
                    parentsOfType.add((T) parent);
                    found = true;
                }
                
                if (!found || continueIfFound)
                {
                    parentsOfType.addAll(_getParentsOfType(parent, visitedProgramItems, classToTest, continueIfFound));
                }
            }
        }
        
        return parentsOfType;
    }
    
    /**
     * Get the programs that belong to a given {@link OrgUnit} or a child orgunit
     * @param orgUnit The orgUnit. Can be null
     * @param catalog The catalog. Can be null.
     * @param lang The lang. Can be null.
     * @return The child programs as unmodifiable list
     */
    public List<Program> getProgramsFromOrgUnit(OrgUnit orgUnit, String catalog, String lang)
    {
        return getProgramsFromOrgUnit(orgUnit, catalog, lang, true);
    }
    
    /**
     * Get the child programs of an {@link OrgUnit}
     * @param orgUnit The orgUnit. Can be null
     * @param catalog The catalog. Can be null.
     * @param lang The lang. Can be null.
     * @param browseChildOrgunits Set to true to get programs among child orgunits recursively, false to get only programs directly linked to the given orgunit.
     * @return The child programs as unmodifiable list
     */
    public List<Program> getProgramsFromOrgUnit(OrgUnit orgUnit, String catalog, String lang, boolean browseChildOrgunits)
    {
        Expression ouExpr = null;
        if (orgUnit != null)
        {
            if (!browseChildOrgunits)
            {
                ouExpr = new StringExpression(ProgramItem.ORG_UNITS_REFERENCES, Operator.EQ, orgUnit.getId());
            }
            else
            {
                ouExpr = new OrExpression();
                for (String orgUnitId : getSubOrgUnitIds(orgUnit))
                {
                    ((OrExpression) ouExpr).add(new StringExpression(ProgramItem.ORG_UNITS_REFERENCES, Operator.EQ, orgUnitId));
                }
            }
        }
        
        AmetysObjectIterable<Program> programs = getProgramItems(ProgramFactory.PROGRAM_CONTENT_TYPE, null, catalog, lang, ouExpr, null);
        return programs.stream().toList();
    }
    
    /**
     * Get the current orgunit and its suborgunits recursively identifiers.
     * @param orgUnit The orgunit at the top
     * @return A {@link List} of {@link OrgUnit} ids
     */
    public List<String> getSubOrgUnitIds(OrgUnit orgUnit)
    {
        List<String> orgUnitIds = new ArrayList<>();
        orgUnitIds.add(orgUnit.getId());
        for (String id : orgUnit.getSubOrgUnits())
        {
            OrgUnit childOrgUnit = _resolver.resolveById(id);
            orgUnitIds.addAll(getSubOrgUnitIds(childOrgUnit));
        }
        
        return orgUnitIds;
    }
    
    /**
     * Get all program item linked to the given orgUnit
     * @param orgUnit the given orgUnit
     * @return the set of program item
     */
    public Set<Program> getProgramsReferencingOrgunit(OrgUnit orgUnit)
    {
        StringExpression stringExpression = new StringExpression(ProgramItem.ORG_UNITS_REFERENCES, Operator.EQ, orgUnit.getId());
        String contentXPathQuery = ContentQueryHelper.getContentXPathQuery(stringExpression);
        
        return _resolver.<Content>query(contentXPathQuery)
            .stream()
            .filter(ProgramItem.class::isInstance)
            .map(ProgramItem.class::cast)
            .map(p -> getParentPrograms(p, true))
            .flatMap(Set::stream)
            .collect(Collectors.toSet());
    }
    
    /**
     * Get the linked program items to the given person
     * @param person the person
     * @return the set of program items
     */
    public Set<Program> getProgramsReferencingPerson(Person person)
    {
        return _contentHelper.getReferencingContents(person)
            .stream()
            .map(Pair::getValue)
            .map(this::_getProgramsReferencingContent)
            .flatMap(Set::stream)
            .collect(Collectors.toSet());
    }
    
    private Set<Program> _getProgramsReferencingContent(Content content)
    {
        if (content instanceof ProgramItem programItem)
        {
            return getParentPrograms(programItem, true);
        }
        else if (content instanceof OrgUnit orgUnit)
        {
            return getProgramsReferencingOrgunit(orgUnit);
        }
        
        return Set.of();
    }
    
    /**
     * Determines if the {@link ProgramItem} has parent program items
     * @param programItem The program item
     * @return true if has parent program items
     */
    public boolean hasParentProgramItems(ProgramItem programItem)
    {
        boolean hasParent = false;
        
        if (programItem instanceof ProgramPart programPart)
        {
            hasParent = !programPart.getProgramPartParents().isEmpty() || hasParent;
        }
        
        if (programItem instanceof CourseList courseList)
        {
            hasParent = !courseList.getParentCourses().isEmpty() || hasParent;
        }
        
        if (programItem instanceof Course course)
        {
            hasParent = !course.getParentCourseLists().isEmpty() || hasParent;
        }
        
        return hasParent;
    }
    
    /**
     * Get the parent program items of a {@link ProgramItem}
     * @param programItem The program item
     * @return The parent program items
     */
    public List<ProgramItem> getParentProgramItems(ProgramItem programItem)
    {
        return getParentProgramItems(programItem, null);
    }
    
    /**
     * Get the program item parents into the given ancestor {@link ProgramPart}
     * @param programItem The program item
     * @param parentProgramPart The parent program, subprogram or container. If null, all parents program items will be returned.
     * @return The parent program items which have given parent program part has an ancestor
     */
    public List<ProgramItem> getParentProgramItems(ProgramItem programItem, ProgramPart parentProgramPart)
    {
        Cache<ParentProgramItemsCacheKey, List<ProgramItem>> cache = _cacheManager.get(__PARENT_PROGRAM_ITEMS_CACHE);
        
        return cache.get(ParentProgramItemsCacheKey.of(programItem.getId(), parentProgramPart != null ? parentProgramPart.getId() : "__NOPARENT"), item -> {
            List<ProgramItem> parents = new ArrayList<>();
            
            if (programItem instanceof Program)
            {
                return parents;
            }
            
            if (programItem instanceof ProgramPart programPart)
            {
                List<ProgramPart> allParents = programPart.getProgramPartParents();
                
                for (ProgramPart parent : allParents)
                {
                    if (parentProgramPart == null || parent.equals(parentProgramPart))
                    {
                        parents.add(parent);
                    }
                    else if (!getParentProgramItems(parent, parentProgramPart).isEmpty())
                    {
                        parents.add(parent);
                    }
                }
            }
            
            if (programItem instanceof CourseList courseList)
            {
                for (Course parentCourse : courseList.getParentCourses())
                {
                    if (!getParentProgramItems(parentCourse, parentProgramPart).isEmpty())
                    {
                        parents.add(parentCourse);
                    }
                }
            }
            
            if (programItem instanceof Course course)
            {
                for (CourseList cl : course.getParentCourseLists())
                {
                    if (!getParentProgramItems(cl, parentProgramPart).isEmpty())
                    {
                        parents.add(cl);
                    }
                    
                }
            }
            
            return parents;
        });
    }
    
    private static class ParentProgramItemsCacheKey extends AbstractCacheKey
    {
        public ParentProgramItemsCacheKey(String programItemId, String parentProgramPartId)
        {
            super(programItemId, parentProgramPartId);
        }
        
        public static ParentProgramItemsCacheKey of(String programItemId, String parentProgramPartId)
        {
            return new ParentProgramItemsCacheKey(programItemId, parentProgramPartId);
        }
    }
    
    /**
     * Get the first nearest program item parent into the given parent {@link AbstractProgram}
     * @param programItem The program item
     * @param parentProgram The parent program or subprogram. If null, the nearest abstract program will be returned.
     * @return The parent program item or null if not found.
     */
    public ProgramItem getParentProgramItem(ProgramItem programItem, AbstractProgram parentProgram)
    {
        List<ProgramItem> parentProgramItems = getParentProgramItems(programItem, parentProgram);
        return parentProgramItems.isEmpty() ? null : parentProgramItems.get(0);
    }
    
    /**
     * Get information of the program item
     * @param programItemId the program item id
     * @param programItemPathIds the list of program item ids containing in the path of the program item ... starting with itself. Can be null or empty
     * @return a map of information
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getProgramItemInfo(String programItemId, List<String> programItemPathIds)
    {
        Map<String, Object> results = new HashMap<>();
        ProgramItem programItem = _resolver.resolveById(programItemId);

        // Get catalog
        String catalog = programItem.getCatalog();
        if (StringUtils.isNotBlank(catalog))
        {
            results.put("catalog", catalog);
        }
        
        // Get the orgunits
        List<String> orgUnits = programItem.getOrgUnits();
        if (programItemPathIds == null || programItemPathIds.isEmpty())
        {
            // The programItemPathIds is null or empty because we do not know the program item context.
            // so get the information in the parent structure if unique.
            while (programItem != null && orgUnits.isEmpty())
            {
                orgUnits = programItem.getOrgUnits();
                List<ProgramItem> parentProgramItems = getParentProgramItems(programItem);
                programItem = parentProgramItems.size() == 1 ? parentProgramItems.get(0) : null;
            }
        }
        else // We have the program item context: parent structure is known ...
        {
            // ... the first element of the programItemPathIds is the programItem itself, so begin to index 1
            int position = 1;
            int size = programItemPathIds.size();
            while (position < size && orgUnits.isEmpty())
            {
                programItem = _resolver.resolveById(programItemPathIds.get(position));
                orgUnits = programItem.getOrgUnits();
                position++;
            }
        }
        results.put("orgUnits", orgUnits);
        
        return results;
    }
    
    /**
     * Get information of the program item structure (type, if program has children) or orgunit (no structure for now)
     * @param contentId the content id
     * @return a map of information
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getStructureInfo(String contentId)
    {
        if (StringUtils.isNotBlank(contentId))
        {
            Content content = _resolver.resolveById(contentId);
            return getStructureInfo(content);
        }
        
        return Map.of();
    }
    
    /**
     * Get information of the program item structure (type, if program has children) or orgunit (no structure for now)
     * @param content the content
     * @return a map of information
     */
    public Map<String, Object> getStructureInfo(Content content)
    {
        Map<String, Object> results = new HashMap<>();
        
        if (content instanceof ProgramItem programItem)
        {
            results.put("id", content.getId());
            results.put("title", content.getTitle());
            results.put("code", programItem.getDisplayCode());
            
            List<ProgramItem> childProgramItems = getChildProgramItems(programItem);
            results.put("hasChildren", !childProgramItems.isEmpty());
            
            List<ProgramItem> parentProgramItems = getParentProgramItems(programItem);
            results.put("hasParent", !parentProgramItems.isEmpty());
            
            results.put("paths", getPaths(programItem, " > "));
        }
        else if (content instanceof OrgUnit orgunit)
        {
            results.put("id", content.getId());
            results.put("title", content.getTitle());
            results.put("code", orgunit.getUAICode());
            
            // Always to false, we don't manage complete copy with children
            results.put("hasChildren", false);
            
            results.put("hasParent", orgunit.getParentOrgUnit() != null);
            
            results.put("paths", List.of(getOrgUnitPath(orgunit, " > ")));
        }
        
        return results;
    }
    
    /**
     * Get information of the program item structure (type, if program has children)
     * @param programItemIds the list of program item id
     * @return a map of information
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Map<String, Object>> getStructureInfo(List<String> programItemIds)
    {
        Map<String,  Map<String, Object>> results = new HashMap<>();
        
        for (String programItemId : programItemIds)
        {
            Content content = _resolver.resolveById(programItemId);
            if (_rightManager.currentUserHasReadAccess(content))
            {
                results.put(programItemId, getStructureInfo(content));
            }
        }
        
        return results;
    }
    
    /**
     * Get all the path of the orgunit.<br>
     * The path is built with the contents' title and code
     * @param orgunit The orgunit
     * @param separator The path separator
     * @return the path in parent orgunit
     */
    public String getOrgUnitPath(OrgUnit orgunit, String separator)
    {
        String path = orgunit.getTitle() + " (" + orgunit.getUAICode() + ")";
        OrgUnit parent = orgunit.getParentOrgUnit();
        if (parent != null)
        {
            path = getOrgUnitPath(parent, separator) + separator + path;
        }
        return path;
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem} as readable values
     * The path is built with the contents' title and code
     * @param item The program item
     * @param separator The path separator
     * @return the paths in parent program items
     */
    public List<String> getPaths(ProgramItem item, String separator)
    {
        Function<ProgramItem, String> mapper = c -> ((Content) c).getTitle() + " (" + c.getDisplayCode() + ")";
        return getPaths(item, separator, mapper, true);
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem} as readable values
     * The path is built with the mapper function.
     * @param item The program item
     * @param separator The path separator
     * @param mapper the function to apply to each program item to build the path
     * @param includeItself set to false to not include final item in path
     * @return the paths in parent program items
     */
    public List<String> getPaths(ProgramItem item, String separator, Function<ProgramItem, String> mapper, boolean includeItself)
    {
        return getPaths(item, separator, mapper, x -> true, includeItself, false);
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem} as readable values
     * The path is built with the mapper function.
     * @param item The program item
     * @param separator The path separator
     * @param mapper the function to apply to each program item to build the path
     * @param filterPathSegment predicate to exclude some program item of path
     * @param includeItself set to false to not include final item in path
     * @param ignoreOrphanPath set to true to ignore paths that is not part of a Program
     * @return the paths in parent program items
     */
    public List<String> getPaths(ProgramItem item, String separator, Function<ProgramItem, String> mapper, Predicate<ProgramItem> filterPathSegment, boolean includeItself, boolean ignoreOrphanPath)
    {
        List<EducationalPath> educationalPaths = getEducationalPaths(item, includeItself, ignoreOrphanPath);
        
        return educationalPaths.stream()
            .map(p -> getEducationalPathAsString(p, mapper, separator, filterPathSegment))
            .collect(Collectors.toList());
    }
    
    /**
     * Get the value of an attribute for a given educational path. The attribute is necessarily in a repeater composed of at least one educational path attribute and the attribute to be retrieved.
     * @param <T> The type of the value returned by the path
     * @param programItem The program item
     * @param path Full or partial educational path. Cannot be null. In case of partial educational path (ie., no root program) the full possible paths will be computed.
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @return the value for this educational path
     */
    public <T> Optional<T> getValueForPath(ProgramItem programItem, String dataPath, EducationalPath path)
    {
        return getValueForPath(programItem, dataPath, List.of(path));
    }
    
    
    /**
     * Get the value of an attribute for a given educational path. The attribute is necessarily in a repeater composed of at least one educational path attribute and the attribute to be retrieved.
     * @param <T> The type of the value returned by the path
     * @param programItem The program item
     * @param paths Full educational paths (from root program). Cannot be null nor empty.
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @return the value for this educational path
     */
    public <T> Optional<T> getValueForPath(ProgramItem programItem, String dataPath, List<EducationalPath> paths)
    {
        String repeaterPath = StringUtils.substringBeforeLast(dataPath, "/");
        String attributeName = StringUtils.substringAfterLast(dataPath, "/");
        
        Content content = (Content) programItem;
        if (!content.hasValue(repeaterPath))
        {
            // Repeater is not defined
            getLogger().warn("There is no repeater '{}' defined for content '{}'", repeaterPath, programItem.getId());
            return Optional.empty();
        }
        
        // Get the repeater
        ModelAwareRepeater repeater = content.getRepeater(repeaterPath);
        
        // Get a value for each value in paths because if it is multiple, we want to check all paths, keep null values
        List<T> values = paths.stream()
             .map(path ->
                 getRepeaterEntriesByPath(repeater, List.of(path))
                     .findFirst()
                     .map(e -> e.<T>getValue(attributeName))
                     .orElse(null)
             )
             .toList();
        
        boolean isSameValues = values.stream().distinct().count() == 1;
        if (!isSameValues)
        {
            // No same values for each path
            getLogger().warn("Unable to determine value for '{}' attribute of content '{}'. Multiple educational paths are available for requested context with no same values", dataPath, programItem.getId());
            return Optional.empty();
        }

        // Same value for each available paths => return this common value
        return values.stream().filter(Objects::nonNull).findFirst();
    }
    
    /**
     * Get the value of an attribute for each available educational paths
     * @param <T> The type of the values returned by the path
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param defaultValue The default value to use if the repeater contains no entry for this educational path. Can be null.
     * @return the values for each educational paths
     */
    public <T> Map<EducationalPath, T> getValuesForPaths(ProgramItem programItem, String dataPath, T defaultValue)
    {
        return getValuesForPaths(programItem, dataPath, getEducationalPaths(programItem), defaultValue);
    }
    
    /**
     * Get the value of an attribute for each given educational paths
     * @param <T> The type of the value returned by the path
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param paths The full educational paths (from root programs)
     * @param defaultValue The default value to use if the repeater contains no entry for this educational path. Can be null.
     * @return the values for each educational paths
     */
    @SuppressWarnings("unchecked")
    public <T> Map<EducationalPath, T> getValuesForPaths(ProgramItem programItem, String dataPath, List<EducationalPath> paths, T defaultValue)
    {
        Map<EducationalPath, T> valuesByPath = new HashMap<>();
        
        paths.stream().forEach(path -> {
            valuesByPath.put(path, (T) getValueForPath(programItem, dataPath, path).orElse(defaultValue));
        });
        
        return valuesByPath;
    }
    
    /**
     * Determines if the values of an attribute depending of a educational path is the same for all available educational paths
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @return true if the value is the same for all available educational paths
     */
    public boolean isSameValueForAllPaths(ProgramItem programItem, String dataPath)
    {
        return isSameValueForPaths(programItem, dataPath, getEducationalPaths(programItem));
    }
    
    /**
     * Determines if the values of an attribute depending of a educational path is the same for all available educational paths
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param paths The full educational paths (from root programs)
     * @return true if the value is the same for all available educational paths
     */
    public boolean isSameValueForPaths(ProgramItem programItem, String dataPath, List<EducationalPath> paths)
    {
        String repeaterPath = StringUtils.substringBeforeLast(dataPath, "/");
        if (!((Content) programItem).hasValue(repeaterPath))
        {
            // Repeater is empty, the value is the default value for all educational paths
            return true;
        }
        
        return getValuesForPaths(programItem, dataPath, paths, null).values().stream().distinct().count() == 1;
    }
    
    /**
     * Get a position of repeater entry that match the given educational path
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param path The full educational path (from root program). Cannot be null.
     * @return the index position of the entry matching the educational path with a non-empty value for requested attribute, -1 otherwise
     */
    public int getRepeaterEntryPositionForPath(ProgramItem programItem, String dataPath, EducationalPath path)
    {
        return getRepeaterEntryPositionForPath(programItem, dataPath, List.of(path));
    }
    
    /**
     * Get a position of repeater entry that match the given educational path
     * @param programItem The program item
     * @param dataPath The full path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param paths The full educational paths (from root program). Cannot be null.
     * @return the index position of the entry matching the educational path with a non-empty value for requested attribute, -1 otherwise
     */
    public int getRepeaterEntryPositionForPath(ProgramItem programItem, String dataPath, List<EducationalPath> paths)
    {
        String repeaterPath = StringUtils.substringBeforeLast(dataPath, "/");
        String attributeName = StringUtils.substringAfterLast(dataPath, "/");
        
        Content content = (Content) programItem;
        if (!content.hasValue(repeaterPath))
        {
            // Repeater is not defined
            getLogger().debug("There is no repeater '{}' defined for content '{}'", repeaterPath, programItem.getId());
            return -1;
        }
        
        // Get the repeater
        ModelAwareRepeater repeater = content.getRepeater(repeaterPath);
        
        // Get a value for each value in paths because if it is multiple, we want to check all paths
        List<Pair<Integer, Object>> values = paths.stream()
             .map(path ->
                 getRepeaterEntriesByPath(repeater, List.of(path))
                     .findFirst()
                     .map(e -> Pair.of(e.getPosition(), e.getValue(attributeName)))
                     .orElseGet(() -> Pair.of(-1, null))
             )
             .toList();
        
        boolean isSameValues = values.stream().map(Pair::getRight).distinct().count() == 1;
        if (!isSameValues)
        {
            // No same values for each path
            getLogger().warn("Unable to determine repeater entry for '{}' attribute of content '{}'. Multiple educational paths are available for requested context with no same values", dataPath, programItem.getId());
            return -1;
        }
        
        // Same value for each available paths => return position of any entry
        return values.stream().findFirst().map(Pair::getLeft).orElse(-1);
    }
    
    /**
     * Get the full educations paths (from a root {@link Program}) from a full or partial path
     * @param path A full or partial path composed by program item ancestors
     * @return the full educational paths
     */
    public List<EducationalPath> getEducationPathFromPath(EducationalPath path)
    {
        return getEducationPathFromPath(path.getProgramItems(_resolver));
    }
    
    /**
     * Get the full educations paths (from a root {@link Program}) from a full or partial path
     * @param path A full or partial path composed by program item ancestors
     * @return the full educational paths
     */
    public List<EducationalPath> getEducationPathFromPath(List<ProgramItem> path)
    {
        return getEducationPathFromPaths(List.of(path));
    }
    
    /**
     * Get the full educations paths (from a root {@link Program}) from full or partial paths
     * @param paths full or partial paths composed by program item ancestors
     * @return the full educational paths
     */
    public List<EducationalPath> getEducationPathFromPaths(List<List<ProgramItem>> paths)
    {
        return getEducationPathFromPaths(paths, null);
    }
    
    /**
     * Get the full educations paths (from a root {@link Program})from full or partial paths
     * @param paths full or partial paths composed by program item ancestors
     * @param withAncestor filter the educational paths that contains this ancestor. Can be null.
     * @return the full educational paths
     */
    public List<EducationalPath> getEducationPathFromPaths(List<List<ProgramItem>> paths, ProgramItem withAncestor)
    {
        List<EducationalPath> fullPaths = new ArrayList<>();
        
        for (List<ProgramItem> partialPath : paths)
        {
            ProgramItem firstProgramItem = partialPath.get(0);
            if (!(firstProgramItem instanceof Program))
            {
                // First program item of path is not a root program => computed the available full paths from this first item ancestors
                List<EducationalPath> parentEducationalPaths = getEducationalPaths(firstProgramItem, false);
                
                fullPaths.addAll(parentEducationalPaths.stream()
                    .filter(p -> withAncestor == null || p.getProgramItems(_resolver).contains(withAncestor)) // filter educational paths that is not composed by the required ancestor
                    .map(p -> EducationalPath.of(p, partialPath.toArray(ProgramItem[]::new))) // concat path
                    .toList());
            }
            else if (withAncestor == null || partialPath.contains(withAncestor))
            {
                // The path is already a full path
                fullPaths.add(EducationalPath.of(partialPath.toArray(ProgramItem[]::new)));
            }
        }
        
        return fullPaths;
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem}
     * The path is built with the mapper function.
     * @param programItem The program item
     * @return the paths in parent program items
     */
    public List<EducationalPath> getEducationalPaths(ProgramItem programItem)
    {
        return getEducationalPaths(programItem, true);
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem}
     * The path is built with the mapper function.
     * @param programItem The program item
     * @param includeItself set to false to not include final item in path
     * @return the paths in parent program items
     */
    public List<EducationalPath> getEducationalPaths(ProgramItem programItem, boolean includeItself)
    {
        return getEducationalPaths(programItem, includeItself, false);
    }
    
    /**
     * Get all {@link EducationalPath} of a {@link ProgramItem}
     * The path is built with the mapper function.
     * @param programItem The program item
     * @param includeItself set to false to not include final item in path
     * @param ignoreOrphanPath set to true to ignore paths that is not part of a Program
     * @return the paths in parent program items
     */
    public List<EducationalPath> getEducationalPaths(ProgramItem programItem, boolean includeItself, boolean ignoreOrphanPath)
    {
        List<EducationalPath> paths = new ArrayList<>();
        
        List<List<ProgramItem>> ancestorPaths = getPathOfAncestors(programItem);
        for (List<ProgramItem> ancestorPath : ancestorPaths)
        {
            if (!ignoreOrphanPath || ancestorPath.get(0) instanceof Program) // ignore paths that is not part of a Program if ignoreOrphanPath is true
            {
                // Don't modify directly the returned value, it may be immutable and cause side effects
                List<ProgramItem> ancestorPathCopy = new ArrayList<>(ancestorPath);
                if (!includeItself)
                {
                    ancestorPathCopy.remove(programItem);
                }
                
                if (!ancestorPathCopy.isEmpty())
                {
                    paths.add(EducationalPath.of(ancestorPathCopy.toArray(ProgramItem[]::new)));
                }
            }
        }
        
        return paths;
    }
    
    /**
     * Get a readable value of a {@link EducationalPath}
     * @param path the educational path
     * @return a String representing the path with program item's title separated by '>'
     */
    public String getEducationalPathAsString(EducationalPath path)
    {
        return getEducationalPathAsString(path, pi -> ((Content) pi).getTitle(), " > ");
    }
    
    /**
     * Get a readable value of a {@link EducationalPath}
     * @param path the educational path
     * @param mapper the function to use for the readable value of a program item
     * @param separator the separator to use
     * @return a String representing the path with program item's readable value given by mapper function and separated by given separator
     */
    public String getEducationalPathAsString(EducationalPath path, Function<ProgramItem, String> mapper, CharSequence separator)
    {
        return getEducationalPathAsString(path, mapper, separator, x -> true);
    }
    
    /**
     * Get a readable value of a {@link EducationalPath}
     * @param path the educational path
     * @param mapper the function to use for the readable value of a program item
     * @param separator the separator to use
     * @param filterPathSegment predicate to exclude some program item of path
     * @return a String representing the path with program item's readable value given by mapper function and separated by given separator
     */
    public String getEducationalPathAsString(EducationalPath path, Function<ProgramItem, String> mapper, CharSequence separator, Predicate<ProgramItem> filterPathSegment)
    {
        return path.resolveProgramItems(_resolver)
                .filter(filterPathSegment)
                .map(mapper)
                .collect(Collectors.joining(separator));
    }
    
    /**
     * Determines if a educational path is valid ODF path
     * @param path the educational path
     * @return <code>true</code> if path represents a valid ODF path
     */
    public boolean isValid(EducationalPath path)
    {
        // Get leaf program item of this educational path
        String leafProgramItemId = path.getProgramItemIds().getLast();
        if (_resolver.hasAmetysObjectForId(leafProgramItemId))
        {
            ProgramItem leafProgramItem = _resolver.resolveById(leafProgramItemId);
            return isValid(path, leafProgramItem, false);
        }
        return false;
    }
    
    
    /**
     * Determines if the given educational path is a valid path for the program item
     * @param path the educational path. The path must include the program item itself
     * @param programItem the program item
     * @param ignoreOrphanPath set to true to ignore paths that is not part of a Program
     * @return <code>true</code> if path is valid for the given program item
     */
    public boolean isValid(EducationalPath path, ProgramItem programItem, boolean ignoreOrphanPath)
    {
        return getEducationalPaths(programItem, true, ignoreOrphanPath).contains(path);
    }
    
    /**
     * Get the full path to program item for highest ancestors. The path includes this final item.
     * @param programItem the program item
     * @return a list for each highest ancestors found. Each item of the list contains the program items to the path to this program item.
     */
    protected List<List<ProgramItem>> getPathOfAncestors(ProgramItem programItem)
    {
        Cache<ProgramItem, List<List<ProgramItem>>> cache = _cacheManager.get(__ANCESTORS_CACHE);
        
        return cache.get(programItem, item -> {
            List<ProgramItem> parentProgramItems = getParentProgramItems(item);
            
            // There is no more parents, the only path is the item itself
            if (parentProgramItems.isEmpty())
            {
                return List.of(List.of(item));
            }
            
            List<List<ProgramItem>> ancestors = new ArrayList<>();
            
            // Compute the path for each parent
            for (ProgramItem parentProgramItem : parentProgramItems)
            {
                for (List<ProgramItem> ancestorPaths : getPathOfAncestors(parentProgramItem))
                {
                    List<ProgramItem> ancestorPathsCopy = new ArrayList<>(ancestorPaths);
                    ancestorPathsCopy.add(item);
                    // Add an immutable list to avoid unvoluntary modifications in cache
                    ancestors.add(Collections.unmodifiableList(ancestorPathsCopy));
                }
            }

            // Add an immutable list to avoid unvoluntary modifications in cache
            return Collections.unmodifiableList(ancestors);
        });
    }
    
    /**
     * Get the enumeration of educational paths for a program item for highest ancestors. The paths does not includes this final item.
     * @param programItemId the id of program item
     * @return a list of educational paths with paths' label (composed by ancestors' title separated by '>') and paths' id (composed by ancestors' id seperated by coma)
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public List<Map<String, String>> getEducationalPathsEnumeration(String programItemId)
    {
        List<Map<String, String>> paths = new ArrayList<>();
        
        ProgramItem programItem = _resolver.resolveById(programItemId);
        
        getEducationalPaths(programItem)
            .stream()
            .forEach(p -> paths.add(Map.of("id", p.toString(), "title", getEducationalPathAsString(p, c -> ((Content) c).getTitle(), " > "))));
        
        return paths;
    }
    
    
    /**
     * Get the path of a {@link ProgramItem} into a {@link Program}<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param programItemId The id of the program item
     * @param programId The id of program. Can not be null.
     * @return the path into the parent program or null if the item is not part of this program.
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public String getPathInProgram (String programItemId, String programId)
    {
        ProgramItem item = _resolver.resolveById(programItemId);
        Program program = _resolver.resolveById(programId);
        
        return getPathInProgram(item, program);
    }
    
    /**
     * Get the path of a ODF content into a {@link Program}.<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param item The program item
     * @param parentProgram The parent root (sub)program. Can not be null.
     * @return the path from the parent program
     */
    public String getPathInProgram (ProgramItem item, Program parentProgram)
    {
        if (item instanceof Program)
        {
            // The program item is already the program it self or another program
            return item.equals(parentProgram) ? "" : null;
        }
        
        List<EducationalPath> paths = getEducationalPaths(item, true, true);
        
        for (EducationalPath path : paths)
        {
            if (path.getProgramItemIds().contains(parentProgram.getId()))
            {
                // Find a path that match the given parent program
                Stream<ProgramItem> resolvedPath = path.resolveProgramItems(_resolver);
                return resolvedPath.map(ProgramItem::getName).collect(Collectors.joining("/"));
            }
        }
        
        return null;
    }
    
    /**
     * Get the path of a {@link Course} or a {@link CourseList} into a {@link Course}<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param contentId The id of the content
     * @param parentCourseId The id of parent course. Can not be null.
     * @return the path into the parent course or null if the item is not part of this course.
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public String getPathInCourse (String contentId, String parentCourseId)
    {
        Content content = _resolver.resolveById(contentId);
        Course parentCourse = _resolver.resolveById(parentCourseId);
        
        return getPathInCourse(content, parentCourse);
    }
    
    /**
     * Get the path of a {@link Course}  or a {@link CourseList} into a {@link Course}<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param courseOrList The course or the course list
     * @param parentCourse The parent course. Can not be null.
     * @return the path into the parent course or null if the item is not part of this course.
     */
    public String getPathInCourse(Content courseOrList, Course parentCourse)
    {
        if (courseOrList.equals(parentCourse))
        {
            return "";
        }
        
        String path = _getPathInCourse(courseOrList, parentCourse);
        
        return path;
    }
    
    private String _getPathInCourse(Content content, Content parentContent)
    {
        if (content.equals(parentContent))
        {
            return content.getName();
        }

        List<? extends Content> parents;
        
        if (content instanceof Course course)
        {
            parents = course.getParentCourseLists();
        }
        else if (content instanceof CourseList courseList)
        {
            parents = courseList.getParentCourses();
        }
        else
        {
            throw new IllegalStateException();
        }
        
        for (Content parent : parents)
        {
            String path = _getPathInCourse(parent, parentContent);
            if (path != null)
            {
                return path + '/' + content.getName();
            }
        }
        return null;
    }
    
    /**
     * Get the hierarchical path of a {@link OrgUnit} from the root orgunit id.<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param orgUnitId The id of the orgunit
     * @param rootOrgUnitId The root orgunit id
     * @return the path into the parent program or null if the item is not part of this program.
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public String getOrgUnitPath(String orgUnitId, String rootOrgUnitId)
    {
        OrgUnit rootOU = null;
        if (StringUtils.isNotBlank(rootOrgUnitId))
        {
            rootOU = _resolver.resolveById(rootOrgUnitId);
        }
        else
        {
            rootOU = _ouRootProvider.getRoot();
        }
        
        if (orgUnitId.equals(rootOU.getId()))
        {
            // The orgunit is already the root orgunit
            return rootOU.getName();
        }
        
        OrgUnit ou = _resolver.resolveById(orgUnitId);
        
        List<String> paths = new ArrayList<>();
        paths.add(ou.getName());
        
        OrgUnit parent = ou.getParentOrgUnit();
        while (parent != null && !parent.getId().equals(rootOU.getId()))
        {
            paths.add(parent.getName());
            parent = parent.getParentOrgUnit();
        }
        
        if (parent != null)
        {
            paths.add(rootOU.getName());
            Collections.reverse(paths);
            return StringUtils.join(paths, "/");
        }
        
        return null;
    }
    
    /**
     * Get the hierarchical path of a {@link OrgUnit} from the root orgunit.<br>
     * The path is construct with the contents' names and the used separator is '/'.
     * @param orgUnitId The id of the orgunit
     * @return the path into the parent program or null if the item is not part of this program.
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public String getOrgUnitPath(String orgUnitId)
    {
        return getOrgUnitPath(orgUnitId, null);
    }
    
    /**
     * Return true if the given {@link ProgramPart} has in its hierarchy a parent of given id
     * @param part The program part
     * @param parentId The ancestor id
     * @return true if the given {@link ProgramPart} has in its hierarchy a parent of given id
     */
    public boolean hasAncestor (ProgramPart part, String parentId)
    {
        List<ProgramPart> parents = part.getProgramPartParents();
        
        for (ProgramPart parent : parents)
        {
            if (parent.getId().equals(parentId))
            {
                return true;
            }
            else if (hasAncestor(parent, parentId))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Determines if a program item is shared
     * @param programItem the program item
     * @return true if the program item is shared
     */
    public boolean isShared(ProgramItem programItem)
    {
        List<ProgramItem> parents = getParentProgramItems(programItem);
        if (parents.size() > 1)
        {
            return true;
        }
        else
        {
            return parents.isEmpty() ? false : isShared(parents.get(0));
        }
    }
    
    /**
     * Check if a relation can be establish between two ODF contents
     * @param srcContent The source content (copied or moved)
     * @param targetContent The target content
     * @param errors The list of error messages
     * @param contextualParameters the contextual parameters
     * @return true if the relation is valid, false otherwise
     */
    public boolean isRelationCompatible(Content srcContent, Content targetContent, List<I18nizableText> errors, Map<String, Object> contextualParameters)
    {
        boolean isCompatible = true;
        
        if (targetContent instanceof ProgramItem || targetContent instanceof OrgUnit)
        {
            if (!_isContentTypeCompatible(srcContent, targetContent))
            {
                // Invalid relations between content types
                errors.add(new I18nizableText("plugin.odf", "PLUGINS_ODF_RELATIONS_SETCONTENTATTRIBUTE_REFERENCE_ERROR_CONTENT_TYPES", _getContentParameters(srcContent, targetContent)));
                isCompatible = false;
            }
            else if (!_isCatalogCompatible(srcContent, targetContent))
            {
                // Catalog is invalid
                errors.add(new I18nizableText("plugin.odf", "PLUGINS_ODF_RELATIONS_SETCONTENTATTRIBUTE_REFERENCE_ERROR_CATALOG", _getContentParameters(srcContent, targetContent)));
                isCompatible = false;
            }
            else if (!_isLanguageCompatible(srcContent, targetContent))
            {
                // Language is invalid
                errors.add(new I18nizableText("plugin.odf", "PLUGINS_ODF_RELATIONS_SETCONTENTATTRIBUTE_REFERENCE_ERROR_LANGUAGE", _getContentParameters(srcContent, targetContent)));
                isCompatible = false;
            }
            else if (!_areShareableFieldsCompatibles(srcContent, targetContent, contextualParameters))
            {
                // Shareable fields don't match
                errors.add(new I18nizableText("plugin.odf", "PLUGINS_ODF_RELATIONS_SETCONTENTATTRIBUTE_REFERENCE_ERROR_SHAREABLE_COURSE", _getContentParameters(srcContent, targetContent)));
                isCompatible = false;
            }
        }
        else if (srcContent instanceof ProgramItem || srcContent instanceof OrgUnit)
        {
            // If the target isn't ODF related but the source is, the relation is not compatible.
            errors.add(new I18nizableText("plugin.odf", "PLUGINS_ODF_RELATIONS_SETCONTENTATTRIBUTE_REFERENCE_ERROR_NO_PROGRAM_ITEM", _getContentParameters(srcContent, targetContent)));
            isCompatible = false;
        }
        
        return isCompatible;
    }
    
    /**
     * Get the name of attribute holding the relation between a parent content and its children
     * @param parentProgramItem the parent content
     * @param childProgramItem the child content
     * @return the name of attribute the child relation
     */
    public String getDescendantRelationAttributeName(ProgramItem parentProgramItem, ProgramItem childProgramItem)
    {
        if (parentProgramItem instanceof CourseList && childProgramItem instanceof Course)
        {
            return CourseList.CHILD_COURSES;
        }
        else if (parentProgramItem instanceof Course && childProgramItem instanceof CourseList)
        {
            return Course.CHILD_COURSE_LISTS;
        }
        else if (parentProgramItem instanceof Course && childProgramItem instanceof CoursePart)
        {
            return Course.CHILD_COURSE_PARTS;
        }
        else if (parentProgramItem instanceof TraversableProgramPart && childProgramItem instanceof ProgramPart)
        {
            return TraversableProgramPart.CHILD_PROGRAM_PARTS;
        }
        
        return null;
    }
    
    private boolean _isCourseAlreadyBelongToCourseList(Course course, CourseList courseList)
    {
        return courseList.getCourses().contains(course);
    }
    
    private boolean _isContentTypeCompatible(Content srcContent, Content targetContent)
    {
        if (srcContent instanceof Container || srcContent instanceof SubProgram)
        {
            return targetContent instanceof AbstractTraversableProgramPart;
        }
        else if (srcContent instanceof CourseList)
        {
            return targetContent instanceof CourseListContainer;
        }
        else if (srcContent instanceof Course)
        {
            return targetContent instanceof CourseList;
        }
        else if (srcContent instanceof OrgUnit)
        {
            return targetContent instanceof OrgUnit;
        }
        
        return false;
    }
    
    private boolean _isCatalogCompatible(Content srcContent, Content targetContent)
    {
        if (srcContent instanceof ProgramItem srcProgramItem && targetContent instanceof ProgramItem targetProgramItem)
        {
            return srcProgramItem.getCatalog().equals(targetProgramItem.getCatalog());
        }
        return true;
    }
    
    private boolean _isLanguageCompatible(Content srcContent, Content targetContent)
    {
        return srcContent.getLanguage().equals(targetContent.getLanguage());
    }
    
    private boolean _areShareableFieldsCompatibles(Content srcContent, Content targetContent, Map<String, Object> contextualParameters)
    {
        // We check shareable fields only if the course content is not created (or created by copy) and not moved
        if (srcContent instanceof Course srcCourse
                && targetContent instanceof CourseList targetCourseList
                && _shareableCourseHelper.handleShareableCourse()
                && !"create".equals(contextualParameters.get("mode"))
                && !"copy".equals(contextualParameters.get("mode"))
                && !"move".equals(contextualParameters.get("mode"))
                // In this case, it means that we try to change the position of the course in the courseList, so don't check shareable fields
                && !_isCourseAlreadyBelongToCourseList(srcCourse, targetCourseList))
        {
            return _shareableCourseHelper.isShareableFieldsMatch(srcCourse, targetCourseList);
        }
        
        return true;
    }
    
    private List<String> _getContentParameters(Content srcContent, Content targetContent)
    {
        List<String> parameters = new ArrayList<>();
        parameters.add(srcContent.getTitle());
        parameters.add(srcContent.getId());
        parameters.add(targetContent.getTitle());
        parameters.add(targetContent.getId());
        return parameters;
    }
    /**
     * Copy a {@link ProgramItem}
     * @param srcContent The program item to copy
     * @param targetCatalog The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param copiedContents the initial contents with their copied content
     * @return The created content
     * @param <C> The modifiable content return type
     * @throws AmetysRepositoryException If an error occurred during copy
     * @throws WorkflowException If an error occurred during copy
     */
    public <C extends ModifiableContent> C copyProgramItem(ProgramItem srcContent, String targetCatalog, boolean fullCopy, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        return copyProgramItem(srcContent, null, null, __INITIAL_WORKFLOW_ACTION_ID, targetCatalog, fullCopy, copiedContents);
    }
    
    /**
     * Copy a {@link ProgramItem}
     * @param srcContent The program item to copy
     * @param targetContentName The name of content to created. Can be null. If null, the new name will be get from the source object.
     * @param targetContentLanguage The name of content to created. Can be null. If null, the language of target content will be the same as source object.
     * @param targetCatalog The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param copiedContents the initial contents with their copied content
     * @param <C> The modifiable content return type
     * @return The created content
     * @throws AmetysObjectExistsException If a program item with same code, catalog and language already exists
     * @throws AmetysRepositoryException If an error occurred
     * @throws WorkflowException If an error occurred
     */
    public <C extends ModifiableContent> C copyProgramItem(ProgramItem srcContent, String targetContentName, String targetContentLanguage, String targetCatalog, boolean fullCopy, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        return copyProgramItem(srcContent, targetContentName, targetContentLanguage, __INITIAL_WORKFLOW_ACTION_ID, targetCatalog, fullCopy, copiedContents);
    }
    
    /**
     * Copy a {@link CoursePart}
     * @param srcContent The course part to copy
     * @param targetContentName The name of content to created. Can be null. If null, the new name will be get from the source object.
     * @param targetContentLanguage The name of content to created. Can be null. If null, the language of target content will be the same as source object.
     * @param initWorkflowActionId The initial workflow action id
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param targetCatalog The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param copiedContents the initial contents with their copied content
     * @param <C> The modifiable content return type
     * @return The created content
     * @throws AmetysObjectExistsException If a program item with same code, catalog and language already exists
     * @throws AmetysRepositoryException If an error occurred
     * @throws WorkflowException If an error occurred
     */
    public <C extends ModifiableContent> C copyCoursePart(CoursePart srcContent, String targetContentName, String targetContentLanguage, int initWorkflowActionId, String targetCatalog, boolean fullCopy, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        return _copyODFContent((Content) srcContent, srcContent.getCatalog(), srcContent.getCode(), targetContentName, targetContentLanguage, initWorkflowActionId, targetCatalog, fullCopy, copiedContents);
    }
    
    /**
     * Copy a {@link ProgramItem}
     * @param srcContent The program item to copy
     * @param targetContentName The name of content to created. Can be null. If null, the new name will be get from the source object.
     * @param targetContentLanguage The name of content to created. Can be null. If null, the language of target content will be the same as source object.
     * @param initWorkflowActionId The initial workflow action id
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param targetCatalog The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param copiedContents the initial contents with their copied content
     * @param <C> The modifiable content return type
     * @return The created content
     * @throws AmetysObjectExistsException If a program item with same code, catalog and language already exists
     * @throws AmetysRepositoryException If an error occurred
     * @throws WorkflowException If an error occurred
     */
    public <C extends ModifiableContent> C copyProgramItem(ProgramItem srcContent, String targetContentName, String targetContentLanguage, int initWorkflowActionId, String targetCatalog, boolean fullCopy, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        return _copyODFContent((Content) srcContent, srcContent.getCatalog(), srcContent.getCode(), targetContentName, targetContentLanguage, initWorkflowActionId, targetCatalog, fullCopy, copiedContents);
    }
    
    /**
     * Copy a {@link ProgramItem}. Also copy the synchronization metadata (status and alternative value)
     * @param srcContent The program item to copy
     * @param catalog The catalog
     * @param code The odf content code
     * @param targetContentName The name of content to created. Can be null. If null, the new name will be get from the source object.
     * @param targetContentLanguage The name of content to created. Can be null. If null, the language of target content will be the same as source object.
     * @param initWorkflowActionId The initial workflow action id
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param targetCatalog The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param copiedContents the initial contents with their copied content
     * @param <C> The modifiable content return type
     * @return The created content
     * @throws AmetysObjectExistsException If a program item with same code, catalog and language already exists
     * @throws AmetysRepositoryException If an error occurred
     * @throws WorkflowException If an error occurred
     */
    @SuppressWarnings("unchecked")
    private <C extends ModifiableContent> C _copyODFContent(Content srcContent, String catalog, String code, String targetContentName, String targetContentLanguage, int initWorkflowActionId, String targetCatalog, boolean fullCopy, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        String computedTargetLanguage = targetContentLanguage;
        if (computedTargetLanguage == null)
        {
            computedTargetLanguage = srcContent.getLanguage();
        }
        
        String computeTargetName = targetContentName;
        if (computeTargetName == null)
        {
            // Compute content name from source content and requested language
            computeTargetName = srcContent.getName() + (targetContentLanguage != null && !targetContentLanguage.equals(srcContent.getName()) ? "-" + targetContentLanguage : "");
        }
        
        String computeTargetCatalog = targetCatalog;
        if (computeTargetCatalog == null)
        {
            computeTargetCatalog = catalog;
        }
        
        String principalContentType = srcContent.getTypes()[0];
        ModifiableContent createdContent = getODFContent(principalContentType, code, computeTargetCatalog, computedTargetLanguage);
        if (createdContent != null)
        {
            getLogger().info("A program item already exists with the same type, code, catalog and language [{}, {}, {}, {}]", principalContentType, code, computeTargetCatalog, targetContentLanguage);
        }
        else
        {
            // Copy content without notifying observers (done later) and copying ACL
            DataContext context = RepositoryDataContext.newInstance()
                                                       .withExternalMetadataInCopy(true);
            createdContent = ((DefaultContent) srcContent).copyTo(getRootContent(true), computeTargetName, targetContentLanguage, initWorkflowActionId, false, true, false, true, context);
            
            if (fullCopy)
            {
                _cleanContentMetadata(createdContent);
                
                if (targetCatalog != null)
                {
                    if (createdContent instanceof ProgramItem programItem)
                    {
                        programItem.setCatalog(targetCatalog);
                    }
                    else if (createdContent instanceof CoursePart coursePart)
                    {
                        coursePart.setCatalog(targetCatalog);
                    }
                    
                }
                
                if (srcContent instanceof ProgramItem programItem)
                {
                    copyProgramItemStructure(programItem, createdContent, computedTargetLanguage, initWorkflowActionId, computeTargetCatalog, copiedContents);
                }
                
                _extractOutgoingReferences(createdContent);
                
                createdContent.saveChanges();
            }
            
            // Notify observers after all structure has been copied
            _contentDAO.notifyContentCopied(createdContent, false);
            
            copiedContents.put(srcContent, createdContent);
        }
        
        return (C) createdContent;
    }
    
    /**
     * Copy the structure of a {@link ProgramItem}
     * @param srcContent the content to copy
     * @param targetContent the target content
     * @param targetContentLanguage The name of content to created. Can be null. If null, the language of target content will be the same as source object.
     * @param initWorkflowActionId The initial workflow action id
     * @param targetCatalogName The target catalog. Can be null. The target catalog will be the catalog of the source object.
     * @param copiedContents the initial contents with their copied content
     * @throws AmetysRepositoryException If an error occurred during copy
     * @throws WorkflowException If an error occurred during copy
     */
    protected void copyProgramItemStructure(ProgramItem srcContent, ModifiableContent targetContent, String targetContentLanguage, int initWorkflowActionId, String targetCatalogName, Map<Content, Content> copiedContents) throws AmetysRepositoryException, WorkflowException
    {
        List<ProgramItem> srcChildContents = new ArrayList<>();
        Map<Pair<String, String>, List<String>> values = new HashMap<>();
        
        String childMetadataPath = null;
        String parentMetadataPath = null;
        
        if (srcContent instanceof TraversableProgramPart programPart)
        {
            childMetadataPath = TraversableProgramPart.CHILD_PROGRAM_PARTS;
            parentMetadataPath = ProgramPart.PARENT_PROGRAM_PARTS;
            srcChildContents.addAll(programPart.getProgramPartChildren());
        }
        else if (srcContent instanceof CourseList courseList)
        {
            childMetadataPath = CourseList.CHILD_COURSES;
            parentMetadataPath = Course.PARENT_COURSE_LISTS;
            srcChildContents.addAll(courseList.getCourses());
        }
        else if (srcContent instanceof Course course)
        {
            childMetadataPath = Course.CHILD_COURSE_LISTS;
            parentMetadataPath = CourseList.PARENT_COURSES;
            srcChildContents.addAll(course.getCourseLists());

            List<String> refCoursePartIds = new ArrayList<>();
            for (CoursePart srcChildContent : course.getCourseParts())
            {
                CoursePart targetChildContent = copyCoursePart(srcChildContent, null, targetContentLanguage, initWorkflowActionId, targetCatalogName, true, copiedContents);
                refCoursePartIds.add(targetChildContent.getId());
            }
            _addFormValues(values, Course.CHILD_COURSE_PARTS, CoursePart.PARENT_COURSES, refCoursePartIds);
        }

        List<String> refChildIds = new ArrayList<>();
        for (ProgramItem srcChildContent : srcChildContents)
        {
            ProgramItem targetChildContent = copyProgramItem(srcChildContent, null, targetContentLanguage, initWorkflowActionId, targetCatalogName, true, copiedContents);
            refChildIds.add(targetChildContent.getId());
        }

        _addFormValues(values, childMetadataPath, parentMetadataPath, refChildIds);

        _editChildRelation((ModifiableWorkflowAwareContent) targetContent, values);
    }
    
    private void _addFormValues(Map<Pair<String, String>, List<String>> values, String childMetadataPath, String parentMetadataPath, List<String> refChildIds)
    {
        if (!refChildIds.isEmpty())
        {
            values.put(Pair.of(childMetadataPath, parentMetadataPath), refChildIds);
        }
    }
    
    private void _editChildRelation(ModifiableWorkflowAwareContent parentContent, Map<Pair<String, String>, List<String>> values) throws AmetysRepositoryException
    {
        if (!values.isEmpty())
        {
            for (Map.Entry<Pair<String, String>, List<String>> entry : values.entrySet())
            {
                String childMetadataName = entry.getKey().getLeft();
                String parentMetadataName = entry.getKey().getRight();
                List<String> childContents = entry.getValue();
                
                parentContent.setValue(childMetadataName, childContents.toArray(new String[childContents.size()]));
                
                for (String childContentId : childContents)
                {
                    ModifiableContent content = _resolver.resolveById(childContentId);
                    String[] parentContentIds = ContentDataHelper.getContentIdsArrayFromMultipleContentData(content, parentMetadataName);
                    content.setValue(parentMetadataName, ArrayUtils.add(parentContentIds, parentContent.getId()));
                    content.saveChanges();
                }
            }
        }
    }
    
    /**
     * Clean the CONTENT metadata created after a copy but whose values reference the initial content' structure
     * @param createdContent The created content to clean
     */
    protected void _cleanContentMetadata(ModifiableContent createdContent)
    {
        if (createdContent instanceof ProgramPart)
        {
            _removeFullValue(createdContent, ProgramPart.PARENT_PROGRAM_PARTS);
        }
        
        if (createdContent instanceof TraversableProgramPart)
        {
            _removeFullValue(createdContent, TraversableProgramPart.CHILD_PROGRAM_PARTS);
        }
        
        if (createdContent instanceof CourseList)
        {
            _removeFullValue(createdContent, CourseList.CHILD_COURSES);
            _removeFullValue(createdContent, CourseList.PARENT_COURSES);
        }
        
        if (createdContent instanceof Course)
        {
            _removeFullValue(createdContent, Course.CHILD_COURSE_LISTS);
            _removeFullValue(createdContent, Course.PARENT_COURSE_LISTS);
            _removeFullValue(createdContent, Course.CHILD_COURSE_PARTS);
        }
        
        if (createdContent instanceof CoursePart)
        {
            _removeFullValue(createdContent, CoursePart.PARENT_COURSES);
        }
    }
    
    private void _removeFullValue(ModifiableContent content, String attributeName)
    {
        content.removeValue(attributeName);
        content.removeExternalizableMetadataIfExists(attributeName);
    }
    
    private void _extractOutgoingReferences(ModifiableContent content)
    {
        Map<String, OutgoingReferences> outgoingReferencesByPath = _outgoingReferencesExtractor.getOutgoingReferences(content);
        content.setOutgoingReferences(outgoingReferencesByPath);
    }
    
    /**
     * Switch the ametys object to Live version if it has one
     * @param ao the Ametys object
     * @throws NoLiveVersionException if the content has no live version
     */
    public void switchToLiveVersion(DefaultAmetysObject ao) throws NoLiveVersionException
    {
        // Switch to the Live label if exists
        String[] allLabels = ao.getAllLabels();
        String[] currentLabels = ao.getLabels();
        
        boolean hasLiveVersion = Arrays.asList(allLabels).contains(CmsConstants.LIVE_LABEL);
        boolean currentVersionIsLive = Arrays.asList(currentLabels).contains(CmsConstants.LIVE_LABEL);
        
        if (hasLiveVersion && !currentVersionIsLive)
        {
            ao.switchToLabel(CmsConstants.LIVE_LABEL);
        }
        else if (!hasLiveVersion)
        {
            throw new NoLiveVersionException("The ametys object '" + ao.getId() + "' has no live version");
        }
    }
    
    /**
     * Switch to Live version if is required
     * @param ao the Ametys object
     * @throws NoLiveVersionException if the Live version is required but not exist
     */
    public void switchToLiveVersionIfNeeded(DefaultAmetysObject ao) throws NoLiveVersionException
    {
        Request request = _getRequest();
        if (request != null && request.getAttribute(REQUEST_ATTRIBUTE_VALID_LABEL) != null)
        {
            switchToLiveVersion(ao);
        }
    }
    
    /**
     * Count the hours accumulation in the {@link ProgramItem}
     * @param programItem The program item on which we compute the total number of hours
     * @return The hours accumulation
     */
    public Double getCumulatedHours(ProgramItem programItem)
    {
        // Ignore optional course list and avoid useless expensive calls
        if (programItem instanceof CourseList courseList && ChoiceType.OPTIONAL.equals(courseList.getType()))
        {
            return 0.0;
        }

        List<ProgramItem> children = getChildProgramItems(programItem);

        Double coef = 1.0;
        Double countNbHours = 0.0;

        // If the program item is a course list, compute the coef (mandatory: 1, optional: 0, optional: min / total)
        if (programItem instanceof CourseList courseList)
        {
            // If there is no children, compute the coef is useless
            // Also choice list can throw an exception while dividing by zero
            if (children.isEmpty())
            {
                return 0.0;
            }
            
            switch (courseList.getType())
            {
                case CHOICE:
                    // Apply the average of number of EC from children multiply by the minimum ELP to select
                    coef = ((double) courseList.getMinNumberOfCourses()) / children.size();
                    break;
                case MANDATORY:
                default:
                    // Add all ECTS from children
                    break;
            }
        }

        // If it's a course and we have a value for the number of hours
        // Then get the value
        if (programItem instanceof Course course && course.hasValue(Course.NUMBER_OF_HOURS))
        {
            countNbHours += course.<Double>getValue(Course.NUMBER_OF_HOURS);
        }
        // Else if there are program item children on the item
        // Then compute on children
        else if (children.size() > 0)
        {
            for (ProgramItem child : children)
            {
                countNbHours += getCumulatedHours(child);
            }
        }
        // Else, it's a course but there is no value for the number of hours and we don't have program item children
        // Then compute on course parts
        else if (programItem instanceof Course course)
        {
            countNbHours += course.getCourseParts()
                .stream()
                .mapToDouble(CoursePart::getNumberOfHours)
                .sum();
        }
        
        return coef * countNbHours;
    }
    
    /**
     * Get the request
     * @return the request
     */
    protected Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
    
    /**
     * Get the first orgunit matching the given UAI code
     * @param uaiCode the UAI code
     * @return the orgunit or null if not found
     */
    public OrgUnit getOrgUnitByUAICode(String uaiCode)
    {
        Expression expr = new AndExpression(
                new ContentTypeExpression(Operator.EQ, OrgUnitFactory.ORGUNIT_CONTENT_TYPE),
                new StringExpression(OrgUnit.CODE_UAI, Operator.EQ, uaiCode)
        );
        
        String xPathQuery = QueryHelper.getXPathQuery(null, OrgUnitFactory.ORGUNIT_NODETYPE, expr);
        AmetysObjectIterable<OrgUnit> orgUnits = _resolver.query(xPathQuery);
        
        return orgUnits.stream()
            .findFirst()
            .orElse(null);
    }
    
    /**
     * Get the repeater entries filtered by path, it takes care of "common" attribute if exists and find itself the educational-path attribute.
     * @param repeater The repeater to filter
     * @param educationalPaths List of full educational paths
     * @return a {@link Stream} of filtered repeater entries
     */
    public Stream<? extends ModelAwareRepeaterEntry> getRepeaterEntriesByPath(ModelAwareRepeater repeater, List<EducationalPath> educationalPaths)
    {
        // Build the filter to apply on repeater entries
        Predicate<ModelAwareRepeaterEntry> filterRepeaterEntries =  _buildRepeaterEntryByPathPredicate(repeater, educationalPaths);
        
        // If predicate is null, an error has been logged
        if (filterRepeaterEntries == null)
        {
            return Stream.empty();
        }
        
        // For each entry, check if the entry is common (if attribute exists) or the path correspond to one of the retrieved full educational paths
        return repeater.getEntries()
            .stream()
            .filter(filterRepeaterEntries);
    }
    
    private Predicate<ModelAwareRepeaterEntry> _buildRepeaterEntryByPathPredicate(ModelAwareRepeater repeater, List<EducationalPath> educationalPaths)
    {
        RepeaterDefinition repeaterModel = repeater.getModel();
        
        Predicate<ModelItem> keepSinglePaths = modelItem -> {
            // Retrieve the path of the parent to check if it is a multiple attribute (remove the repeater path at the beginning of the path, because it would always be multiple otherwise)
            // The parent cannot be null because we know that the repeater is the parent
            String path = StringUtils.substringAfter(modelItem.getParent().getPath(), repeaterModel.getPath() + ModelItem.ITEM_PATH_SEPARATOR);
            
            // If the path is empty, that means the attribute is at the first level of the repeater, so it can be used
            return StringUtils.isEmpty(path) || !DataHolderHelper.isMultiple(repeaterModel, path);
        };
        
        List<ModelItem> educationalPathModelItem = ModelHelper.findModelItemsByType(repeaterModel, EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID)
                               .stream()
                               // If the item containing the path is multiple, we cannot use the path to filter the repeater entries, so exclude them
                               .filter(keepSinglePaths)
                               .toList();
        
        if (educationalPathModelItem.size() != 1)
        {
            getLogger().error("Unable to determine repeater entry matching an education path. No attribute or several attributes of type '{}' found.", EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID, repeaterModel.getPath());
            return null;
        }
        
        // Prepare the predicate for educational path attribute
        String pathAttributeName = educationalPathModelItem.get(0).getName();
        Predicate<ModelAwareRepeaterEntry> repeaterEntriesFilter = e -> educationalPaths.contains(e.getValue(pathAttributeName, false, null));
        
        // Complete predicate with common attribute if exists
        if (repeater.getModel().hasModelItem("common"))
        {
            return ((Predicate<ModelAwareRepeaterEntry>) e -> e.getValue("common", true, true)).or(repeaterEntriesFilter);
        }
        
        return repeaterEntriesFilter;
    }
    
    /**
     * Determine if the content is a container of nature equals to "annee"
     * @param content The content
     * @return <code>true</code> if the current content item is a container of nature equals to "annee"
     */
    public boolean isContainerOfTypeYear(Content content)
    {
        return content instanceof Container container && isContainerOfTypeYear(container);
    }
    
    /**
     * Determine if the container nature equals to "annee"
     * @param containerId The container id
     * @return <code>true</code> if the current container nature equals to "annee"
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public boolean isContainerOfTypeYear(String containerId)
    {
        Container container = _resolver.resolveById(containerId);
        return isContainerOfTypeYear(container);
    }
    
    /**
     * Determine if the container nature equals to "annee"
     * @param container The container
     * @return <code>true</code> if the current container nature equals to "annee"
     */
    public boolean isContainerOfTypeYear(Container container)
    {
        return getYearId()
                .map(id -> Strings.CS.equals(id, container.getNature()))
                .orElse(false);
    }
    
    /**
     * Get the year container nature identifier.
     * @return an {@link Optional} of the year identifier
     */
    public synchronized Optional<String> getYearId()
    {
        if (_yearId.isEmpty())
        {
            _yearId = Optional.of(_refTableHelper)
                .map(rth -> rth.getItemFromCode(OdfReferenceTableHelper.CONTAINER_NATURE, "annee"))
                .map(OdfReferenceTableEntry::getId)
                .filter(StringUtils::isNotBlank);
        }
        return _yearId;
    }
    
    /**
     * Get all the years of a program part, search in children.
     * @param programPart The program part
     * @return A set of {@link Container} with the year nature.
     */
    public Set<Container> getYears(TraversableProgramPart programPart)
    {
        Set<Container> years = new HashSet<>();
        
        if (programPart instanceof Container container && isContainerOfTypeYear(container))
        {
            years.add(container);
        }
        
        programPart.getProgramPartChildren()
                   .stream()
                   .filter(TraversableProgramPart.class::isInstance)
                   .map(TraversableProgramPart.class::cast)
                   .map(this::getYears)
                   .flatMap(Set::stream)
                   .forEach(years::add);
        
        return years;
    }
    
    /**
     * Filter the program item to keep only container with the given nature.
     * @param programItem The program item
     * @param natureId The container nature identifier
     * @return <code>true</code> if it is a container of the given nature, <code>false</code> otherwise
     */
    public boolean isContainerOfNature(ProgramItem programItem, String natureId)
    {
        return Optional.of(programItem)
            .filter(Container.class::isInstance)
            .map(Container.class::cast)
            .map(Container::getNature)
            .map(natureId::equals)
            .orElse(false);
    }
}
