/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.search.model.impl.AbstractStaticSearchModelCriterionDefinition;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentLanguageQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * UI criteria for program item context
 */
public class ProgramItemContextCriterionDefinition extends AbstractStaticSearchModelCriterionDefinition<String>
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public Operator getOperator()
    {
        // Not suppose to be used
        return Operator.EQ;
    }
    
    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }

    @Override
    public Query getQuery(Object value, Operator customOperator, Map<String, Object> allValues, String language, Map<String, Object> contextualParameters)
    {
        List<Query> queries = new ArrayList<>();
        
        Optional<ProgramItem> program = Optional.ofNullable(contextualParameters.get("contentId"))
            .filter(String.class::isInstance)
            .map(String.class::cast)
            .map(id -> _resolver.resolveById(id));
        
        if (program.isPresent())
        {
            ProgramItem programItem = program.get();
            queries.add(new StringQuery(ProgramItem.CATALOG, programItem.getCatalog()));
            
            String lang = programItem.getLanguage();
            if (StringUtils.isNotBlank(lang))
            {
                queries.add(new ContentLanguageQuery(lang));
            }

        }
        
        return new AndQuery(queries);
    }
}
