/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.archive.ArchiveConstants;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseContainer;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

import com.opensymphony.workflow.WorkflowException;

/**
 * Component to handle ODF catalogs
 */
public class CatalogsManager extends AbstractLogEnabled implements Serviceable, Component, PluginAware, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = CatalogsManager.class.getName();

    private AmetysObjectResolver _resolver;

    private CopyCatalogUpdaterExtensionPoint _copyUpdaterEP;

    private ObservationManager _observationManager;

    private CurrentUserProvider _userProvider;

    private ContentWorkflowHelper _contentWorkflowHelper;

    private String _pluginName;

    private ODFHelper _odfHelper;

    private ContentTypeExtensionPoint _cTypeEP;
    
    private Context _context;

    private SolrIndexHelper _solrIndexHelper;
    
    private String _defaultCatalogId;

    private CurrentUserProvider _currentUserProvider;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _copyUpdaterEP = (CopyCatalogUpdaterExtensionPoint) manager.lookup(CopyCatalogUpdaterExtensionPoint.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _solrIndexHelper = (SolrIndexHelper) manager.lookup(SolrIndexHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    /**
     * Get the list of catalogs
     * @return the catalogs
     */
    public List<Catalog> getCatalogs()
    {
        List<Catalog> result = new ArrayList<>();

        TraversableAmetysObject catalogsNode = getCatalogsRootNode();
        
        AmetysObjectIterable<Catalog> catalogs = catalogsNode.getChildren();
        for (Catalog catalog : catalogs)
        {
            result.add(catalog);
        }
        
        return result;
    }

    /**
     * Get a catalog matching with the given name
     * @param name The name
     * @return a catalog, or null if not found
     */
    public Catalog getCatalog(String name)
    {
        ModifiableTraversableAmetysObject catalogsNode = getCatalogsRootNode();
        
        if (StringUtils.isNotEmpty(name) && catalogsNode.hasChild(name))
        {
            return catalogsNode.getChild(name);
        }
        
        // Not found
        return null;
    }
    
    /**
     * Returns the name of the default catalog
     * @return the name of the default catalog
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public String getDefaultCatalogName()
    {
        return Optional.ofNullable(getDefaultCatalog())
                       .map(Catalog::getName)
                       .orElse(null);
    }
    
    /**
     * Returns the default catalog
     * @return the default catalog or null if no default catalog was defined.
     */
    public synchronized Catalog getDefaultCatalog()
    {
        if (_defaultCatalogId == null)
        {
            updateDefaultCatalog();
        }
        
        if (_defaultCatalogId != null)
        {
            try
            {
                return _resolver.resolveById(_defaultCatalogId);
            }
            catch (UnknownAmetysObjectException e)
            {
                _defaultCatalogId = null;
            }
        }
        
        return null;
    }
    
    /**
     * Updates the default catalog (if it's null or if the user has updated it).
     */
    void updateDefaultCatalog()
    {
        List<Catalog> catalogs = getCatalogs();
        for (Catalog catalog : catalogs)
        {
            if (catalog.isDefault())
            {
                _defaultCatalogId = catalog.getId();
                return;
            }
        }
        
        // If no default catalog found, get the only catalog if it exists
        if (catalogs.size() == 1)
        {
            _defaultCatalogId = catalogs.get(0).getId();
        }
    }
    
    /**
     * Get the name of the catalog of a ODF content
     * @param contentId The id of content
     * @return The catalog's name
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public String getContentCatalog(String contentId)
    {
        Content content = _resolver.resolveById(contentId);
        
        if (content instanceof ProgramItem)
        {
            return ((ProgramItem) content).getCatalog();
        }
        
        // catalog can also be present on skills
        // FIXME ODF-3966 - With a new mixin for catalog aware contents, we could have same kind of API that the one existing on ProgramItem
        if (content.hasValue("catalog"))
        {
            return content.getValue("catalog");
        }
        
        // Get catalog from its parents (unecessary ?)
        AmetysObject parent = content.getParent();
        while (parent != null)
        {
            if (parent instanceof ProgramItem)
            {
                return ((ProgramItem) parent).getCatalog();
            }
            parent = parent.getParent();
        }
        
        return null;
    }
    
    /**
     * Determines if the catalog can be modified from the given content
     * @param contentId The content id
     * @return A map with success=false if the catalog cannot be edited
     */
    @Callable (rights = "ODF_Rights_EditCatalog", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> canEditCatalog(String contentId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Content content = _resolver.resolveById(contentId);
        
        if (content instanceof ProgramItem)
        {
            if (_isReferenced(content))
            {
                result.put("success", false);
                result.put("error", "referenced");
            }
            else if (_hasSharedContent((ProgramItem) content, (ProgramItem) content))
            {
                result.put("success", false);
                result.put("error", "hasSharedContent");
            }
            else
            {
                result.put("success", true);
            }
            
        }
        else
        {
            result.put("success", false);
            result.put("error", "typeError");
        }
        
        return result;
    }
    
    private boolean _isReferenced (Content content)
    {
        return !_odfHelper.getParentProgramItems((ProgramItem) content).isEmpty();
    }
    
    private boolean _hasSharedContent (ProgramItem rootProgramItem, ProgramItem programItem)
    {
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        
        for (ProgramItem child : children)
        {
            if (_isShared(rootProgramItem, child))
            {
                return true;
            }
        }
        
        if (programItem instanceof Course)
        {
            List<CoursePart> courseParts = ((Course) programItem).getCourseParts();
            for (CoursePart coursePart : courseParts)
            {
                List<ProgramItem> parentCourses = coursePart.getCourses()
                        .stream()
                        .map(ProgramItem.class::cast)
                        .collect(Collectors.toList());
                if (parentCourses.size() > 1 && !_isPartOfSameStructure(rootProgramItem, parentCourses))
                {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    private boolean _isShared(ProgramItem rootProgramItem, ProgramItem programItem)
    {
        try
        {
            List<ProgramItem> parents = _odfHelper.getParentProgramItems(programItem);
            if (parents.size() > 1 && !_isPartOfSameStructure(rootProgramItem, parents)
                || _hasSharedContent(rootProgramItem, programItem))
            {
                return true;
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // Nothing
        }
        
        return false;
    }
    
    private boolean _isPartOfSameStructure(ProgramItem rootProgramItem, List<ProgramItem> programItems)
    {
        for (ProgramItem programItem : programItems)
        {
            boolean isPartOfInitalStructure = false;
            
            List<EducationalPath> ancestorPaths = _odfHelper.getEducationalPaths(programItem);
            
            for (EducationalPath ancestorPath : ancestorPaths)
            {
                isPartOfInitalStructure = ancestorPath.resolveProgramItems(_resolver).anyMatch(p -> p.equals(rootProgramItem));
                break;
            }
            
            if (!isPartOfInitalStructure)
            {
                // The content is shared outside the program item to edit
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Set the catalog of a content. This will modify recursively the catalog of referenced children
     * @param catalog The catalog
     * @param contentId The id of content to edit
     * @throws WorkflowException if an error occurred
     */
    @Callable (rights = "ODF_Rights_EditCatalog", paramIndex = 1, rightContext = ContentRightAssignmentContext.ID)
    public void setContentCatalog(String catalog, String contentId) throws WorkflowException
    {
        Content content = _resolver.resolveById(contentId);
        
        if (content instanceof ProgramItem)
        {
            _setCatalog(content, catalog);
        }
        else
        {
            throw new IllegalArgumentException("You can not edit the catalog of the content " + contentId);
        }
    }
    
    private void _setCatalog (Content content, String catalogName) throws WorkflowException
    {
        if (content instanceof ProgramItem)
        {
            String oldCatalog = ((ProgramItem) content).getCatalog();
            if (!catalogName.equals(oldCatalog))
            {
                ((ProgramItem) content).setCatalog(catalogName);
                
                if (content instanceof WorkflowAwareContent)
                {
                    _applyChanges((WorkflowAwareContent) content);
                }
            }
        }
        else if (content instanceof CoursePart)
        {
            String oldCatalog = ((CoursePart) content).getCatalog();
            if (!catalogName.equals(oldCatalog))
            {
                ((CoursePart) content).setCatalog(catalogName);
                
                if (content instanceof WorkflowAwareContent)
                {
                    _applyChanges((WorkflowAwareContent) content);
                }
            }
        }
        
        _setCatalogToChildren(content, catalogName);
    }
    
    private void _setCatalogToChildren (Content content, String catalogName) throws WorkflowException
    {
        if (content instanceof TraversableProgramPart)
        {
            ContentValue[] children = content.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
            for (ContentValue child : children)
            {
                try
                {
                    _setCatalog(child.getContent(), catalogName);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Nothing
                }
            }
        }
        
        if (content instanceof CourseContainer)
        {
            for (Course course : ((CourseContainer) content).getCourses())
            {
                _setCatalog(course, catalogName);
            }
        }
        
        if (content instanceof CourseListContainer)
        {
            for (CourseList cl : ((CourseListContainer) content).getCourseLists())
            {
                _setCatalog(cl, catalogName);
            }
        }
        
        if (content instanceof Course)
        {
            for (CoursePart coursePart : ((Course) content).getCourseParts())
            {
                _setCatalog(coursePart, catalogName);
            }
        }
    }
    
    private void _applyChanges(WorkflowAwareContent content) throws WorkflowException
    {
        ((ModifiableDefaultContent) content).setLastContributor(_userProvider.getUser());
        ((ModifiableDefaultContent) content).setLastModified(ZonedDateTime.now());
        
        // Remove the proposal date.
        content.setProposalDate(null);
        
        // Save changes
        content.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_MODIFIED, _userProvider.getUser(), eventParams));
       
        _contentWorkflowHelper.doAction(content, 22);
    }
    
    /**
     * Get the root catalogs storage object.
     * @return the root catalogs node
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public ModifiableTraversableAmetysObject getCatalogsRootNode() throws AmetysRepositoryException
    {
        String originalWorkspace = null;
        Request request = ContextHelper.getRequest(_context);
        try
        {
            originalWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
            if (ArchiveConstants.ARCHIVE_WORKSPACE.equals(originalWorkspace))
            {
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            }
            
            ModifiableTraversableAmetysObject rootNode = _resolver.resolveByPath("/");
            ModifiableTraversableAmetysObject pluginsNode = _getOrCreateNode(rootNode, "ametys:plugins", "ametys:unstructured");
            ModifiableTraversableAmetysObject pluginNode = _getOrCreateNode(pluginsNode, _pluginName, "ametys:unstructured");
            
            return _getOrCreateNode(pluginNode, "catalogs", "ametys:unstructured");
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the ODF catalogs root node", e);
        }
        finally
        {
            if (ArchiveConstants.ARCHIVE_WORKSPACE.equals(originalWorkspace))
            {
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, originalWorkspace);
            }
        }
    }
    
    /**
     * Create a new catalog
     * @param name The unique name
     * @param title The title of catalog
     * @return the created catalog
     */
    public Catalog createCatalog(String name, String title)
    {
        Catalog newCatalog = null;
        
        ModifiableTraversableAmetysObject catalogsNode = getCatalogsRootNode();
        
        newCatalog = catalogsNode.createChild(name, "ametys:catalog");
        newCatalog.setTitle(title);
        
        if (getCatalogs().size() == 1)
        {
            newCatalog.setDefault(true);
        }
        
        newCatalog.saveChanges();
        
        return newCatalog;
    }
    
    /**
     * Get the programs of a catalog for all languages
     * @param catalog The code of catalog
     * @return The programs
     */
    public AmetysObjectIterable<Program> getPrograms (String catalog)
    {
        return getPrograms(catalog, null);
    }
    
    /**
     * Get the program's items of a catalog for all languages
     * @param catalog The code of catalog
     * @return The {@link ProgramItem}
     */
    private AmetysObjectIterable<Content> _getProgramItems(String catalog)
    {
        return _getContentsInCatalog(catalog, ProgramItem.PROGRAM_ITEM_CONTENT_TYPE);
    }

    /**
     * Get the contents of a content type in a catalog.
     * @param <T> The type of the elements to get
     * @param catalog The catalog name
     * @param contentTypeId The content type identifier
     * @return An iterable of contents with given content type in the catalog
     */
    private <T extends Content> AmetysObjectIterable<T> _getContentsInCatalog(String catalog, String contentTypeId)
    {
        List<Expression> exprs = new ArrayList<>();
        
        exprs.add(_cTypeEP.createHierarchicalCTExpression(contentTypeId));
        exprs.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        
        Expression expression = new AndExpression(exprs.toArray(Expression[]::new));
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        return _resolver.query(query);
    }
    
    /**
     * Get the programs of a catalog
     * @param catalog The code of catalog
     * @param lang The language. Can be null to get programs for all languages
     * @return The programs
     */
    public AmetysObjectIterable<Program> getPrograms (String catalog, String lang)
    {
        List<Expression> exprs = new ArrayList<>();
        exprs.add(new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE));
        exprs.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        if (lang != null)
        {
            exprs.add(new LanguageExpression(Operator.EQ, lang));
        }
        
        Expression programsExpression = new AndExpression(exprs.toArray(Expression[]::new));
        
        // Add sort criteria to get size
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        String programsQuery = QueryHelper.getXPathQuery(null, ProgramFactory.PROGRAM_NODETYPE, programsExpression, sortCriteria);
        return _resolver.query(programsQuery);
    }
    
    /**
     * Copy the programs and its hierarchy from a catalog to another.
     * The referenced courses are NOT copied.
     * @param catalog The new catalog to populate
     * @param catalogToCopy The catalog from which we copy the programs.
     * @param progressionTracker the progression tracker for catalog copy
     * @throws ProcessingException If an error occurred during copy
     */
    public void copyCatalog(Catalog catalog, Catalog catalogToCopy, ContainerProgressionTracker progressionTracker) throws ProcessingException
    {
        String catalogToCopyName = catalogToCopy.getName();
        String catalogName = catalog.getName();
        String [] handledEventIds = new String[] {ObservationConstants.EVENT_CONTENT_ADDED, ObservationConstants.EVENT_CONTENT_MODIFIED,  ObservationConstants.EVENT_CONTENT_WORKFLOW_CHANGED};
        try
        {
            Map<Content, Content> copiedContents = new HashMap<>();
            
            Set<String> copyUpdaters = _copyUpdaterEP.getExtensionsIds();
            
            AmetysObjectIterable<Program> programs = getPrograms(catalogToCopyName);
            
            SimpleProgressionTracker copyStep = (SimpleProgressionTracker) progressionTracker.getCurrentStep();
            copyStep.setSize(programs.getSize());
            
            // Do NOT commit yet to Solr in order to improve perfs
            _solrIndexHelper.pauseSolrCommitForEvents(handledEventIds);
            long start = System.currentTimeMillis();
            
            getLogger().debug("Begin to iterate over programs for copying them");
            
            for (Program program : programs)
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Start copying program '{}' (name: '{}', title: '{}')...", program.getId(), program.getName(), program.getTitle());
                }
                
                _odfHelper.copyProgramItem(program, catalogName, true, copiedContents);
                copyStep.increment();
            }

            SimpleProgressionTracker updatesAfterCopy = (SimpleProgressionTracker) progressionTracker.getCurrentStep();
            updatesAfterCopy.setSize(copyUpdaters.size());
            
            for (String updaterId : copyUpdaters)
            {
                // Call updaters after full copy of catalog
                CopyCatalogUpdater updater = _copyUpdaterEP.getExtension(updaterId);
                updater.updateContents(catalogToCopyName, catalogName, copiedContents, null);
                updater.copyAdditionalContents(catalogToCopyName, catalogName, copiedContents);
                
                updatesAfterCopy.increment();
            }

            // Workflow
            _addCopyStep(copiedContents.values(), (SimpleProgressionTracker) progressionTracker.getCurrentStep());
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("End of iteration over programs for copying them ({})", Duration.of((System.currentTimeMillis() - start) / 1000, ChronoUnit.SECONDS));
            }
            
        }
        catch (AmetysRepositoryException | WorkflowException e)
        {
            getLogger().error("Copy of items of catalog {} into catalog {} has failed", catalogToCopyName, catalogName);
            throw new ProcessingException("Failed to copy catalog", e);
        }
        finally
        {
            SimpleProgressionTracker restartCommitStep = (SimpleProgressionTracker) progressionTracker.getCurrentStep();
            restartCommitStep.setSize(1);
            _solrIndexHelper.restartSolrCommitForEvents(handledEventIds);
            restartCommitStep.increment();
        }
    }

    private void _addCopyStep(Collection<Content> contents, SimpleProgressionTracker progressionTracker) throws AmetysRepositoryException, WorkflowException
    {
        progressionTracker.setSize(contents.size());
        
        for (Content content : contents)
        {
            if (content instanceof WorkflowAwareContent workflowAwareContent)
            {
                _contentWorkflowHelper.doAction(workflowAwareContent, getCopyActionId());
            }
            progressionTracker.increment();
        }
    }
    
    /**
     * Get the workflow action id for copy.
     * @return The workflow action id
     */
    protected int getCopyActionId()
    {
        return 210;
    }
    
    /**
     * Delete catalog
     * @param catalog the catalog to delete
     * @return the result map
     */
    public Map<String, Object> deleteCatalog(Catalog catalog)
    {
        Map<String, Object> result = new HashMap<>();
        result.put("id", catalog.getId());
        
        String catalogName = catalog.getName();
        List<Content> contentsToDelete = getContents(catalogName);
        
        // Before deleting anything, we have to make sure that it's safe to delete the catalog and its programItems
        List<Content> referencingContents = _getExternalReferencingContents(contentsToDelete);
        
        if (!referencingContents.isEmpty())
        {
            for (Content content : referencingContents)
            {
                if (content instanceof ProgramItem || content instanceof CoursePart)
                {
                    getLogger().error("{} '{}' ({}) is referencing a content of the catalog {} while being itself in the catalog {}. There is an inconsistency.",
                            content.getClass().getName(),
                            content.getTitle(),
                            content.getValue("code"),
                            catalogName,
                            content.getValue("catalog", false, StringUtils.EMPTY));
                }
                else
                {
                    getLogger().warn("Content {} ({}) is referencing a content of the catalog {}. There is an inconsistency.",
                            content.getTitle(),
                            content.getId(),
                            catalogName);
                }
            }
            result.put("error", "referencing-contents");
            result.put("referencingContents", referencingContents);
            return result;
        }
        
        // Everything is fine, we can delete the courseParts, the programItems and the catalog
        String[] handledEventIds = new String[] {ObservationConstants.EVENT_CONTENT_DELETED};
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(handledEventIds);
            contentsToDelete.forEach(this::_deleteContent);
        }
        finally
        {
            _solrIndexHelper.restartSolrCommitForEvents(handledEventIds);
        }
        
        ModifiableAmetysObject parent = catalog.getParent();
        catalog.remove();
        parent.saveChanges();
        
        return result;
    }
    
    /**
     * Get all the contents to delete when deleting a catalog.
     * @param catalogName The catalog name
     * @return a {@link Stream} of {@link Content} to delete
     */
    public List<Content> getContents(String catalogName)
    {
        List<Content> contents = new ArrayList<>();
        contents.addAll(_getProgramItems(catalogName).stream().toList());
        contents.addAll(_getCourseParts(catalogName).stream().toList());
        
        for (String updaterId : _copyUpdaterEP.getExtensionsIds())
        {
            CopyCatalogUpdater updater = _copyUpdaterEP.getExtension(updaterId);
            contents.addAll(updater.getAdditionalContents(catalogName));
        }
        
        return contents;
    }
    
    /**
     * Get the course part of a catalog for all languages
     * @param catalog The code of catalog
     * @return The {@link CoursePart}s
     */
    private AmetysObjectIterable<CoursePart> _getCourseParts(String catalog)
    {
        return _getContentsInCatalog(catalog, CoursePartFactory.COURSE_PART_CONTENT_TYPE);
    }

    /**
     * Get the list of contents referencing one of the {@code ProgramItem}s in the set.
     * This method will ignore content that are included in the set and any {@code CoursePart} belonging to a {@code Course} of the set
     * @param contentsToDelete the contents to be tested
     * @return the contents referencing those program items excluding the course parts id
     */
    private List<Content> _getExternalReferencingContents(List<Content> contentsToDelete)
    {
        
        // Get all the Contents referencing one of the content to delete but not one of the content to delete
        List<Content> referencingContents = contentsToDelete.stream()
            .map(Content::getReferencingContents)   // get the referencing contents
            .flatMap(Collection::stream)            // flatten the Collection
            .distinct()                             // remove all duplicates
            .filter(content -> !contentsToDelete.contains(content))
            .collect(Collectors.toUnmodifiableList());          // collect it into a list
        return referencingContents;
    }
    
    private void _deleteContent(Content deletedContent)
    {
        try
        {
            RemovableAmetysObject content = _resolver.resolveById(deletedContent.getId());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_CONTENT, content);
            eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, content.getName());
            eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
            ModifiableAmetysObject parent = content.getParent();
            
            _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            // Remove the content.
            LockableAmetysObject lockedContent = (LockableAmetysObject) content;
            if (lockedContent.isLocked())
            {
                lockedContent.unlock();
            }
            
            content.remove();
            
            parent.saveChanges();
            
            _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETED, _currentUserProvider.getUser(), eventParams));
        }
        catch (UnknownAmetysObjectException e)
        {
            // Ignore, already removed
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("An error occured while trying to delete the content '{}' while deleting the catalog.", deletedContent.getId(), e);
            }
        }
    }
    
    private ModifiableTraversableAmetysObject _getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
}
