/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.util.Map;

import org.ametys.cms.content.CopyReport;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.workflow.copy.AbstractCreateODFContentByCopyFunction;
import org.ametys.odf.workflow.copy.CopyODFContentClientSideElement;
import org.ametys.runtime.model.ElementDefinition;

/**
 * Component used for copy of content with some override for ODF content
 * See {@link org.ametys.cms.content.CopyContentComponent}.
 */
public class CopyContentComponent extends org.ametys.cms.content.CopyContentComponent
{
    /** The avalon role */
    @SuppressWarnings("hiding")
    public static final String ROLE = CopyContentComponent.class.getName();
    
    @Override
    protected ContentValue handleLinkedContent(ElementDefinition definition, ModifiableContent content, boolean referenceMode, Map<String, Object> copyMap, CopyReport copyReport)
    {
        String path = definition.getPath();
        
        // Override for the ODF specific case for copy of a ProgramPart in reference mode
        // All child contents can be referenced EXCEPT the course lists which have to be always be copied (created by copy)
        if (path.equals(TraversableProgramPart.CHILD_PROGRAM_PARTS) && referenceMode)
        {
            if (content instanceof CourseList)
            {
                // Force copy of child course lists (force create mode)
                String targetContentId = copyLinkedContent(content, copyMap, copyReport);
                if (targetContentId != null)
                {
                    return new ContentValue(_resolver, targetContentId);
                }
            }
            else
            {
                // keep reference for other children
                return new ContentValue(content);
            }
        }
        else
        {
            return super.handleLinkedContent(definition, content, referenceMode, copyMap, copyReport);
        }
        
        return null;
    }
    
    @Override
    protected Map<String, Object> getInputsForCopy(Content baseContent, String title, Map<String, Object> copyMap, String targetContentType, CopyReport copyReport)
    {
        Map<String, Object> inputs =  super.getInputsForCopy(baseContent, title, copyMap, targetContentType, copyReport);
        
        String parentId = copyMap != null ? (String) copyMap.get(CopyODFContentClientSideElement.PARENT_KEY) : null;
        if (parentId != null)
        {
            inputs.put(AbstractCreateODFContentByCopyFunction.PARENT_KEY, parentId);
        }
        
        return inputs;
    }
}
