/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.SaxBuffer;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * SAX the structure (ie. the child program items) of a {@link ProgramItem}
 *
 */
public class ProgramItemStructureGenerator extends ServiceableGenerator implements Initializable
{
    // FIXME Accept all view names starting with "main" to be able to make the ODF and ODF orientation skins coexist (whereas they use different main views)
    private static final Set<Pattern> __ALLOWED_VIEW_NAMES = Set.of(Pattern.compile("main[a-z\\-]*"), Pattern.compile("pdf"));
    
    private static final String __REF_ITEM_CACHE_ID = ProgramItemStructureGenerator.class.getName() + "$refItems";
    private static final String __VIEW_CACHE_ID = ProgramItemStructureGenerator.class.getName() + "$view";
    private static final String __COMMON_ATTRIBUTES_CACHE_ID = ProgramItemStructureGenerator.class.getName() + "$commonAttributes";
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** Helper for ODF reference table */
    protected OdfReferenceTableHelper _odfReferenceTableHelper;
    /** The content types helper */
    protected ContentTypesHelper _cTypesHelper;
    /** The ODF skills helper */
    protected ODFSkillsHelper _odfSkillsHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    private Cache<Triple<String, String, String>, SaxBuffer> _refItemCache;
    private Cache<Content, SaxBuffer> _viewCache;
    private Cache<ProgramItem, CommonAttributes> _commonAttrCache;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
        _cTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _odfSkillsHelper = (ODFSkillsHelper) smanager.lookup(ODFSkillsHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void setup(org.apache.cocoon.environment.SourceResolver res, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(res, objModel, src, par);
        _refItemCache = _cacheManager.get(__REF_ITEM_CACHE_ID);
        _viewCache = _cacheManager.get(__VIEW_CACHE_ID);
        _commonAttrCache = _cacheManager.get(__COMMON_ATTRIBUTES_CACHE_ID);
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        _refItemCache = null;
        _viewCache = null;
        _commonAttrCache = null;
    }
    
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__REF_ITEM_CACHE_ID))
        {
            _cacheManager.createRequestCache(__REF_ITEM_CACHE_ID,
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_REF_ITEMS_LABEL"),
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_REF_ITEMS_DESCRIPTION"),
                                      false);
        }
        
        if (!_cacheManager.hasCache(__VIEW_CACHE_ID))
        {
            _cacheManager.createRequestCache(__VIEW_CACHE_ID,
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_VIEW_LABEL"),
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_VIEW_DESCRIPTION"),
                                      false);
        }
        
        if (!_cacheManager.hasCache(__COMMON_ATTRIBUTES_CACHE_ID))
        {
            _cacheManager.createRequestCache(__COMMON_ATTRIBUTES_CACHE_ID,
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_COMMON_ATTRIBUTES_LABEL"),
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAM_ITEM_STRUCTURE_COMMON_ATTRIBUTES_DESCRIPTION"),
                                      false);
        }
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Content content = (Content) request.getAttribute(Content.class.getName());
        
        if (content == null)
        {
            String contentId = parameters.getParameter("contentId", null);
            if (StringUtils.isBlank(contentId))
            {
                throw new IllegalArgumentException("Content is missing in request attribute or parameters");
            }
            content = _resolver.resolveById(contentId);
        }

        String viewName = parameters.getParameter("viewName", StringUtils.EMPTY);
        String fallbackViewName = parameters.getParameter("fallbackViewName", StringUtils.EMPTY);
        
        View view = _cTypesHelper.getViewWithFallback(viewName, fallbackViewName, content);
        
        contentHandler.startDocument();
        
        if (view != null && __ALLOWED_VIEW_NAMES.stream().anyMatch(p -> p.matcher(view.getName()).matches()))
        {
            if (content instanceof ProgramItem programItem)
            {
                XMLUtils.startElement(contentHandler, "structure");
                
                List<ProgramItem> initialAncestorPath = _getInitialAncestorPath(request, programItem);
                saxProgramItem(programItem, initialAncestorPath);
                
                XMLUtils.endElement(contentHandler, "structure");
            }
            else
            {
                getLogger().warn("Cannot get the structure of a non program item '" + content.getId() + "'");
            }
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Get the initial ancestor path from request or from root program item
     * @param request the request
     * @param rootProgramItem the root program item in saxed structure
     * @return the initial ancestor path as a list of program items
     */
    protected List<ProgramItem> _getInitialAncestorPath(Request request, ProgramItem rootProgramItem)
    {
        // First try to get ancestor path given by request
        @SuppressWarnings("unchecked")
        List<ProgramItem> ancestorPath = (List<ProgramItem>) request.getAttribute(EducationalPathHelper.PROGRAM_ITEM_ANCESTOR_PATH_REQUEST_ATTR);
        
        if (ancestorPath == null)
        {
            if (rootProgramItem instanceof SubProgram subProgram)
            {
                List<EducationalPath> subProgramPaths = subProgram.getCurrentEducationalPaths();
                if (subProgramPaths != null && subProgramPaths.size() == 1)
                {
                    // Init the ancestor paths from current educational path only if there is only one eligible educational path
                    ancestorPath = subProgramPaths.get(0).getProgramItems(_resolver);
                }
            }
            else if (rootProgramItem instanceof Course course)
            {
                List<EducationalPath> coursePaths = course.getCurrentEducationalPaths();
                if (coursePaths != null && coursePaths.size() == 1)
                {
                    // Init the ancestor paths from current educational path only if there is only one eligible educational path
                    ancestorPath = coursePaths.get(0).getProgramItems(_resolver);
                }
            }
        }
        
        // Ancestor path cannot be determine by context, initialize the ancestor path to a item itself
        return ancestorPath == null || ancestorPath.isEmpty() ? List.of(rootProgramItem) : ancestorPath;
    }
        
    /**
     * SAX a program item with its child program items
     * @param programItem the program item
     * @param ancestorPath The path of this program item in the saxed structure (computed from the initial saxed program item). Can be a partial path.
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxProgramItem(ProgramItem programItem, List<ProgramItem> ancestorPath) throws SAXException
    {
        if (programItem instanceof Program)
        {
            saxProgram((Program) programItem);
        }
        else if (programItem instanceof SubProgram)
        {
            saxSubProgram((SubProgram) programItem, ancestorPath);
        }
        else if (programItem instanceof Container)
        {
            saxContainer((Container) programItem, ancestorPath);
        }
        else if (programItem instanceof CourseList)
        {
            saxCourseList((CourseList) programItem, ancestorPath);
        }
        else if (programItem instanceof Course)
        {
            saxCourse((Course) programItem, ancestorPath);
        }
    }
    
    /**
     * SAX the child program items
     * @param programItem the program item
     * @param ancestorPath The path of parent program item in the structure (starting from the initial saxed program item)
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxChildProgramItems(ProgramItem programItem, List<ProgramItem> ancestorPath) throws SAXException
    {
        List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem childProgramItem : childProgramItems)
        {
            try
            {
                _odfHelper.switchToLiveVersionIfNeeded((DefaultAmetysObject) childProgramItem);
                List<ProgramItem> childAncestorPath = new ArrayList<>(ancestorPath);
                childAncestorPath.add(childProgramItem);
                
                saxProgramItem(childProgramItem, childAncestorPath);
            }
            catch (NoLiveVersionException e)
            {
                // Just ignore the program item
            }
        }
    }
        
    /**
     * SAX a program
     * @param program the subprogram to SAX
     * @throws SAXException if an error occurs
     */
    protected void saxProgram(Program program) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _saxCommonAttributes(program, null, attrs);
        
        XMLUtils.startElement(contentHandler, "program", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxStructureViewIfExists(program);
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxChildProgramItems(program, List.of(program));
        XMLUtils.endElement(contentHandler, "program");
    }
    
    /**
     * SAX a subprogram
     * @param subProgram the subprogram to SAX
     * @param ancestorPath The path of this program item in the saxed structure (computed from the initial saxed program item). Can be a partial path.
     * @throws SAXException if an error occurs
     */
    protected void saxSubProgram(SubProgram subProgram, List<ProgramItem> ancestorPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _saxCommonAttributes(subProgram, ancestorPath, attrs);
        
        XMLUtils.startElement(contentHandler, "subprogram", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxReferenceTableItem(subProgram.getEcts(), AbstractProgram.ECTS, subProgram.getLanguage());
        _saxStructureViewIfExists(subProgram);
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxChildProgramItems(subProgram, ancestorPath);
        
        XMLUtils.endElement(contentHandler, "subprogram");
    }
    
    /**
     * SAX a container
     * @param container the container to SAX
     * @param ancestorPath The path of this program item in the saxed structure (computed from the initial saxed program item). Can be a partial path.
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxContainer(Container container, List<ProgramItem> ancestorPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _saxCommonAttributes(container, ancestorPath, attrs);
        
        XMLUtils.startElement(contentHandler, "container", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxReferenceTableItem(container.getNature(), Container.NATURE, container.getLanguage());
        _saxReferenceTableItem(container.getPeriod(), Container.PERIOD, container.getLanguage());
        
        _saxStructureViewIfExists(container);
        
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxChildProgramItems(container, ancestorPath);
        
        XMLUtils.endElement(contentHandler, "container");
    }
    
    /**
     * SAX a course list
     * @param cl the course list to SAX
     * @param ancestorPath The path of this program item in the saxed structure (computed from the initial saxed program item). Can be a partial path.
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxCourseList(CourseList cl, List<ProgramItem> ancestorPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _saxCommonAttributes(cl, ancestorPath, attrs);
        
        XMLUtils.startElement(contentHandler, "courselist", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxStructureViewIfExists(cl);
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxChildProgramItems(cl, ancestorPath);
        
        XMLUtils.endElement(contentHandler, "courselist");
    }
    
    /**
     * SAX a course
     * @param course the container to SAX
     * @param ancestorPath The path of this program item in the saxed structure (computed from the initial saxed program item). Can be a partial path.
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxCourse(Course course, List<ProgramItem> ancestorPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _saxCommonAttributes(course, ancestorPath, attrs);
        
        XMLUtils.startElement(contentHandler, "course", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxReferenceTableItem(course.getCourseType(), Course.COURSE_TYPE, course.getLanguage());
        _saxStructureViewIfExists(course);
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxChildProgramItems(course, ancestorPath);
        
        saxCourseParts(course);
        
        XMLUtils.endElement(contentHandler, "course");
    }
    
    /**
     * SAX a course part
     * @param course The course
     * @throws SAXException if an error occurs
     */
    protected void saxCourseParts(Course course) throws SAXException
    {
        List<CoursePart> courseParts = course.getCourseParts();
        
        double totalHours = 0;
        List<CoursePart> liveCourseParts = new ArrayList<>();
        for (CoursePart coursePart : courseParts)
        {
            try
            {
                _odfHelper.switchToLiveVersionIfNeeded(coursePart);
                totalHours += coursePart.getNumberOfHours();
                liveCourseParts.add(coursePart);
            }
            catch (NoLiveVersionException e)
            {
                getLogger().warn("Some hours of " + course.toString() + " are not added because the course part " + coursePart.toString() + " does not have a live version.");
            }
        }
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("totalHours", String.valueOf(totalHours));
        XMLUtils.startElement(contentHandler, "courseparts", attrs);

        for (CoursePart coursePart : liveCourseParts)
        {
            saxCoursePart(coursePart);
        }
        
        XMLUtils.endElement(contentHandler, "courseparts");
    }
    
    /**
     * SAX a course part
     * @param coursePart The course part to SAX
     * @throws SAXException if an error occurs
     */
    protected void saxCoursePart(CoursePart coursePart) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", coursePart.getId());
        attrs.addCDATAAttribute("title", coursePart.getTitle());
        _addAttrIfNotEmpty(attrs, "code", coursePart.getCode());

        XMLUtils.startElement(contentHandler, "coursepart", attrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        _saxReferenceTableItem(coursePart.getNature(), CoursePart.NATURE, coursePart.getLanguage());
        
        _saxStructureViewIfExists(coursePart);
        
        XMLUtils.endElement(contentHandler, "attributes");
        
        XMLUtils.endElement(contentHandler, "coursepart");
    }
    
    /**
     * SAX the 'structure' view if exists
     * @param content the content
     * @throws SAXException if an error occurs
     */
    protected void _saxStructureViewIfExists(Content content) throws SAXException
    {
        SaxBuffer buffer = _viewCache.get(content);
        
        if (buffer != null)
        {
            buffer.toSAX(contentHandler);
            return;
        }
        
        View view = _cTypesHelper.getView("structure", content.getTypes(), content.getMixinTypes());
        if (view != null)
        {
            try
            {
                buffer = new SaxBuffer();
                
                content.dataToSAX(buffer, view, DataContext.newInstance().withLocale(LocaleUtils.toLocale(content.getLanguage())).withEmptyValues(false));
                
                _viewCache.put(content, buffer);
                buffer.toSAX(contentHandler);
            }
            catch (BadItemTypeException | AmetysRepositoryException e)
            {
                throw new SAXException("Fail to sax the 'structure' view for content " + content.getId(), e);
            }
        }
    }
    
    /**
     * SAX the common attributes for program item
     * @param programItem the program item
     * @param ancestorPath The path of this program item in the structure (starting from the initial saxed program item)
     * @param attrs the attributes
     */
    protected void _saxCommonAttributes(ProgramItem programItem, List<ProgramItem> ancestorPath, AttributesImpl attrs)
    {
        CommonAttributes commonAttributes = _commonAttrCache.get(programItem, k -> {
            return new CommonAttributes(programItem.getId(), ((Content) programItem).getTitle(), programItem.getCode(), programItem.getName(), _odfSkillsHelper.isExcluded(programItem));
        });
        
        attrs.addCDATAAttribute("title", commonAttributes.title());
        attrs.addCDATAAttribute("id", commonAttributes.id());
        attrs.addCDATAAttribute("code", commonAttributes.code());
        attrs.addCDATAAttribute("name", commonAttributes.name());
        boolean excludedFromSkills = commonAttributes.excludedFromSkills();
        if (excludedFromSkills)
        {
            attrs.addCDATAAttribute("excludedFromSkills", String.valueOf(excludedFromSkills));
        }
        
        if (ancestorPath != null)
        {
            attrs.addCDATAAttribute("path", EducationalPath.of(ancestorPath.toArray(ProgramItem[]::new)).toString());
        }
    }
    
    private record CommonAttributes(String id, String title, String code, String name, boolean excludedFromSkills) { }
    
    /**
     * SAX the item of a reference table
     * @param itemId the item id
     * @param tagName the tag name
     * @param lang the language to use
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxReferenceTableItem(String itemId, String tagName, String lang) throws SAXException
    {
        Triple<String, String, String> cacheKey = Triple.of(itemId, tagName, lang);
        SaxBuffer buffer = _refItemCache.get(cacheKey);
        
        if (buffer != null)
        {
            buffer.toSAX(contentHandler);
            return;
        }
        
        buffer = new SaxBuffer();
        
        OdfReferenceTableEntry item = Optional.ofNullable(itemId)
                                              .filter(StringUtils::isNotEmpty)
                                              .map(_odfReferenceTableHelper::getItem)
                                              .orElse(null);
        
        if (item != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", item.getId());
            _addAttrIfNotEmpty(attrs, "code", item.getCode());
            
            XMLUtils.createElement(buffer, tagName, attrs, item.getLabel(lang));
            
            _refItemCache.put(cacheKey, buffer);
            buffer.toSAX(contentHandler);
        }
    }
    
    /**
     * Add an attribute if its not null or empty.
     * @param attrs The attributes
     * @param attrName The attribute name
     * @param attrValue The attribute value
     */
    protected void _addAttrIfNotEmpty(AttributesImpl attrs, String attrName, String attrValue)
    {
        if (StringUtils.isNotEmpty(attrValue))
        {
            attrs.addCDATAAttribute(attrName, attrValue);
        }
    }

}
