/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content.code;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.PluginsManager;

/**
 * Implementation of a {@link DisplayCodeProvider} based on {@link Content} attributes.<br>
 * Display code will be composed by attributes' values separated by '-' character.
 * <br>
 * Expected configuration is:<br>
 *   <pre>
 *   &lt;type id="CONTENT_TYPE_ID_1">
 *       &lt;item ref="path/of/attribute1"/>
 *       &lt;item ref="path/of/attribute2"/>
 *   &lt;/type>
 *   &lt;type id="CONTENT_TYPE_ID_2">
 *       &lt;item ref="path/of/attribute1"/>
 *       &lt;item ref="path/of/attribute2"/>
 *   &lt;/type>
 *   </pre>
 */
public class AttributeBasedDisplayCodeProvider extends AbstractStaticProgramItemDisplayCodeProvider implements Serviceable
{
    private static final String __ATTRIBUTES_SEPARATOR = "-";
    
    /** The service manager */
    protected ServiceManager _smanager;
    /** The attributes composing the code by content types */
    protected Map<String, List<String>> _attributesByTypes = new HashMap<>();
    
    private ContentTypesHelper _contentTypeHelper;
    private ContentTypeExtensionPoint _cTypeEP;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _smanager = smanager;
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        if (!PluginsManager.getInstance().isSafeMode() && isActive())
        {
            // Check and fill configuration only if not in safe mode (to keep component safe)
            Configuration[] cTypesConf = configuration.getChildren("type");
            for (Configuration typeConf : cTypesConf)
            {
                String cTypeId = typeConf.getAttribute("id");
                ContentType cType = getContentTypeExtensionPoint().getExtension(cTypeId);
                if (cType == null)
                {
                    throw new ConfigurationException("Unknown content type " + cTypeId, typeConf);
                }
                
                List<String> attributePaths = new ArrayList<>();
                
                for (Configuration itemConf : typeConf.getChildren("item"))
                {
                    String dataPath = itemConf.getAttribute("ref");
                    if (!cType.hasModelItem(dataPath))
                    {
                        throw new ConfigurationException("Attribute '" + dataPath + "' is not part of the model for content type " + cTypeId, typeConf);
                    }
                    
                    ModelItem modelItem = cType.getModelItem(dataPath);
                    if (modelItem instanceof ElementDefinition definition && ModelItemTypeConstants.STRING_TYPE_ID.equals(definition.getType().getId()) && !definition.isMultiple())
                    {
                        attributePaths.add(dataPath);
                    }
                    else
                    {
                        throw new ConfigurationException("Item '" + dataPath + " is not a valid definition for display code, only non-multiple string attribute are allowed", typeConf);
                    }
                }
                
                _attributesByTypes.put(cTypeId, attributePaths);
            }
        }
    }
    
    /**
     * Get the content type extension point
     * @return the content type extension point
     */
    protected ContentTypeExtensionPoint getContentTypeExtensionPoint()
    {
        // Lazing loading to keep component safe
        if (_cTypeEP == null)
        {
            try
            {
                _cTypeEP = (ContentTypeExtensionPoint) _smanager.lookup(ContentTypeExtensionPoint.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the content type extension point", e);
            }
        }
        
        return _cTypeEP;
    }
    
    /**
     * Get content types helper
     * @return the content types helper
     */
    protected ContentTypesHelper getContentTypesHelper()
    {
        // Lazing loading to keep component safe
        if (_contentTypeHelper == null)
        {
            try
            {
                _contentTypeHelper = (ContentTypesHelper) _smanager.lookup(ContentTypesHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the content types helper", e);
            }
        }
        
        return _contentTypeHelper;
    }
    
    public Optional<String> getDisplayCode(Content content)
    {
        List<String> values = new ArrayList<>();
        
        for (String cTypeId : _attributesByTypes.keySet())
        {
            if (getContentTypesHelper().isInstanceOf(content, cTypeId))
            {
                List<String> attributePaths = _attributesByTypes.get(cTypeId);
                for (String attributePath : attributePaths)
                {
                    String value = content.getValue(attributePath, false, null);
                    if (StringUtils.isNotEmpty(value))
                    {
                        values.add(value);
                    }
                }
            }
        }
        
        return values.isEmpty() ? Optional.empty() : Optional.of(String.join(__ATTRIBUTES_SEPARATOR, values));
    }

    public boolean supports(Content content)
    {
        for (String cTypeId : _attributesByTypes.keySet())
        {
            if (getContentTypesHelper().isInstanceOf(content, cTypeId))
            {
                return true;
            }
        }
        return false;
    }
}
