/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export;

import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Set the title of the program item in response header
 */
public class SetProgramItemHeader extends ServiceableAction
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        String programItemId = parameters.getParameter("programItemId", null);
        if (StringUtils.isBlank(programItemId))
        {
            throw new IllegalArgumentException("The id of program item must be provided");
        }
        
        try
        {
            Content content = _resolver.resolveById(programItemId);
            
            Response response = ObjectModelHelper.getResponse(objectModel);
            
            String prefix = _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_PDF_NAME_PREFIX"), content.getLanguage());
            
            String fileName = prefix + " " + content.getTitle() + ".pdf";
            String encodedName = URIUtils.encodeHeader(fileName);
            
            response.setHeader("Content-Disposition", "attachment; filename=\"" + encodedName + "\";filename*=UTF-8''" + encodedName);
        }
        catch (UnknownAmetysObjectException e) 
        {
            throw new IllegalArgumentException("Unknown program item with id '" + programItemId + "'", e);
        }
        
        return EMPTY_MAP;
    }
}
