/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.TimeZone;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;

/**
 * Generator for the <code>ListIdentifiers</code> verb.
 */
public class ListIdentifiersGenerator extends AbstractOAIVerbGenerator implements Serviceable
{
    /** The OAI-PMH Sets Extension Point*/
    private OaiSetExtensionPoint _oaiSetEP;
    
    private CatalogsManager _catalogsManager;

    private ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _oaiSetEP = (OaiSetExtensionPoint) serviceManager.lookup(OaiSetExtensionPoint.ROLE);
        _catalogsManager = (CatalogsManager) serviceManager.lookup(CatalogsManager.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    protected Collection<String> getRequiredParameters()
    {
        return Arrays.asList("verb", "metadataPrefix");
    }

    @Override
    protected Collection<String> getAllowedParameters()
    {
        return Arrays.asList("verb", "from", "until", "metadataPrefix", "resumptionToken", "set");
    }

    @Override
    protected void generateVerb() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String token = request.getParameter("resumptionToken");
        
        if (StringUtils.isNotEmpty(token))
        {
            generateError("badResumptionToken ", "This repository does not support resumption tokens");
            return;
        }
        
        String metadataPrefix = request.getParameter("metadataPrefix");
        
        if (!metadataPrefix.equals("cdm") && !metadataPrefix.equals("oai_dc"))
        {
            generateError("cannotDisseminateFormat", "The value of the metadataPrefix argument is not supported by the repository.");
            return;
        }
        
        SimpleDateFormat fullFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        TimeZone tz = TimeZone.getTimeZone("UTC");
        fullFormat.setTimeZone(tz);
        format.setTimeZone(tz);

        ArrayList<Expression> expressions = new ArrayList<>();
        
        try
        {
            String fromParam = request.getParameter("from");
            
            if (fromParam != null)
            {
                Date from;
                if (fromParam.indexOf('T') == -1)
                {
                    from = format.parse(fromParam);
                }
                else
                {
                    from = fullFormat.parse(fromParam);
                }
                
                expressions.add(new DateExpression("lastModified", Operator.GE, from));
            }
            
            String untilParam = request.getParameter("until");
            
            if (untilParam != null)
            {
                Date until;
                if (untilParam.indexOf('T') == -1)
                {
                    until = format.parse(untilParam);
                }
                else
                {
                    until = fullFormat.parse(untilParam);
                }
                
                expressions.add(new DateExpression("lastModified", Operator.LE, until));
            }
        }
        catch (ParseException ex)
        {
            generateError("badArgument", "One or more aguments are malformed.");
            return;
        }
        
        OaiSet oaiSet = null;
        String set = request.getParameter("set");
        
        if (StringUtils.isNotEmpty(set))
        {
            oaiSet = _oaiSetEP.getExtension(set);
            if (oaiSet == null)
            {
                generateError("noRecordsMatch", "The set specified as an argument does not exists.");
                return;
            }
        }
        
        Expression expression = new AndExpression(expressions.toArray(new Expression[]{}));
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        try (AmetysObjectIterable<Program> programs = oaiSet != null
                ? oaiSet.getRecords(expression, sortCriteria)
                // FIXME Generate identifier for language "fr" and default catalog only
                : _odfHelper.getProgramItems(ProgramFactory.PROGRAM_CONTENT_TYPE, null, _catalogsManager.getDefaultCatalogName(), "fr", expression, sortCriteria))
        {
        
            XMLUtils.startElement(contentHandler, getTagName());
            
            for (Program program : programs)
            {
                try
                {
                    String[] labels = program.getAllLabels();
                    
                    if (Arrays.asList(labels).contains(CmsConstants.LIVE_LABEL))
                    {
                        program.switchToLabel(CmsConstants.LIVE_LABEL);
                        saxProgram(program);
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    // no "Live" label, continue
                }
            }
            
            XMLUtils.endElement(contentHandler, getTagName());
        }
    }
    
    /**
     * Returns the surrounding tag name.
     * @return the surrounding tag name.
     */
    protected String getTagName()
    {
        return "ListIdentifiers";
    }
    
    /**
     * Generate the program information.
     * @param program the program
     * @throws SAXException if an error occurs
     */
    protected void saxProgram(Program program) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "header");
        
        XMLUtils.createElement(contentHandler, "identifier", program.getCDMId());
        
        ZonedDateTime date = program.getLastModified();
        XMLUtils.createElement(contentHandler, "datestamp", DateUtils.zonedDateTimeToString(date, ZoneOffset.UTC, "yyyy-MM-dd'T'HH:mm:ssXXX"));
        
        XMLUtils.endElement(contentHandler, "header");
    }
}
