/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.orgunit;

import java.util.List;
import java.util.Optional;

import javax.jcr.Node;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.cdmfr.CDMEntity;
import org.ametys.odf.content.code.DisplayCodeProperty;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.values.ValueContext;

/**
 * OrgUnit java object
 */
public class OrgUnit extends ModifiableDefaultContent<OrgUnitFactory> implements CDMEntity
{
    /** prefix for the code attribute */
    public static final String CODE_PREFIX = "orgunit-";
    /** code RNE attribute. */
    public static final String CODE_UAI = "codeUAI";
    /** SIRET attribute. */
    public static final String SIRET = "siret";
    /** Activity number attribute. */
    public static final String ACTIVITY_NUMBER = "activityNumber";
    /** acronym attribute. */
    public static final String ACRONYM = "acronym";
    /** description attribute. */
    public static final String DESCRIPTION = "description";
    /** admission description attribute. */
    public static final String ADMISSION_INFO = "admissionInfo";
    /** regulations attribute. */
    public static final String REGULATIONS = "regulations";
    /** expenses attribute. */
    public static final String EXPENSES = "expenses";
    /** universal adjustement attribute. */
    public static final String UNIVERSAL_ADJUSTMENT = "universalAdjustment";
    /** student facilities attribute. */
    public static final String STUDENT_FACILITIES = "studentFacilities";
    /** additional data attribute. */
    public static final String ADDITIONNAL_INFOS = "additionalInfos";
    /** web link attribute. */
    public static final String WEB_LINK_LABEL = "webLinkLabel";
    /** web link  url attribute. */
    public static final String WEB_LINK_URL = "webLinkUrl";
    /** attribute holding the parent orgunit */
    public static final String PARENT_ORGUNIT = "parentOrgUnit";
    /** attribute holding the contact */
    public static final String CONTACTS = "contactsReferences";
    /** attribute holding the child orgunits */
    public static final String CHILD_ORGUNITS = "childOrgUnits";
    /** OrgUnit type */
    public static final String TYPE = "type";

    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public OrgUnit(Node node, String parentPath, OrgUnitFactory factory)
    {
        super(node, parentPath, factory);
    }

    // --------------------------------------------------------------------------------------//
    //
    // SUB ORGUNITS
    //
    // --------------------------------------------------------------------------------------//

    /**
     * Return a List of orgUnits IDs up to date, Each ID is checked to
     * remove deleted elements
     * @return List&lt;String&gt;
     */
    public List<String> getSubOrgUnits()
    {
        return ContentDataHelper.getContentIdsListFromMultipleContentData(this, CHILD_ORGUNITS);
    }

    /**
     * Get the id of parent {@link OrgUnit}
     * @return the id of parent {@link OrgUnit} or null;
     */
    public OrgUnit getParentOrgUnit()
    {
        return Optional.ofNullable((ContentValue) getValue(PARENT_ORGUNIT))
                .flatMap(ContentValue::getContentIfExists)
                .map(OrgUnit.class::cast)
                .orElse(null);
    }
    
    // --------------------------------------------------------------------------------------//
    //
    // CONTACTS
    //
    // --------------------------------------------------------------------------------------//

    /**
     * Return a List of contact IDs
     * @return a list of uuid
     */
    public List<String> getContacts()
    {
        return ContentDataHelper.getContentIdsListFromMultipleContentData(this, CONTACTS);
    }
    
    /**
     * Return a List of local contact IDs
     * @return a list of uuid
     */
    public List<String> getLocalContacts()
    {
        ValueContext localValueContext = ValueContext.newInstance().withStatus(ExternalizableDataStatus.LOCAL);
        return ContentDataHelper.getContentIdsListFromMultipleContentData(this, CONTACTS, localValueContext);
    }
    
    /**
     * Return a List of remote contact IDs
     * @return a list of uuid
     */
    public List<String> getRemoteContacts()
    {
        ValueContext externalValueContext = ValueContext.newInstance().withStatus(ExternalizableDataStatus.EXTERNAL);
        return ContentDataHelper.getContentIdsListFromMultipleContentData(this, CONTACTS, externalValueContext);
    }
    
    
    // --------------------------------------------------------------------------------------//
    //
    // GETTERS AND SETTERS
    //
    // --------------------------------------------------------------------------------------//
    /**
     * Get the code to display
     * @return the code to display
     */
    public String getDisplayCode()
    {
        return getValue(DisplayCodeProperty.PROPERTY_NAME, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the UAI code
     * @return the UAI code
     * @throws AmetysRepositoryException if failed to get metadata
     */
    public String getUAICode() throws AmetysRepositoryException
    {
        return getValue(CODE_UAI, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the SIRET
     * @return the SIRET
     * @throws AmetysRepositoryException if failed to get metadata
     */
    public String getSIRET() throws AmetysRepositoryException
    {
        return getValue(SIRET, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the orgunit type
     * @return the orgunit type
     * @throws AmetysRepositoryException if failed to get metadata
     */
    public String getType() throws AmetysRepositoryException
    {
        return ContentDataHelper.getContentIdFromContentData(this, TYPE);
    }
    
    /**
     * Return the metadata code
     * 
     * @return the acronym
     */
    public String getAcronym()
    {
        return getValue(OrgUnit.ACRONYM, false, StringUtils.EMPTY);
    }

    /**
     * Return the description
     * @return the description
     */
    public RichText getDescription()
    {
        return getValue(OrgUnit.DESCRIPTION);
    }

    /**
     * Return the metadata admission_info
     * @return the admission_info
     */
    public RichText getAdmissionInfo()
    {
        return getValue(OrgUnit.ADMISSION_INFO);
    }

    /**
     * Return the regulations
     * @return the regulations
     */
    public RichText getRegulations()
    {
        return getValue(OrgUnit.REGULATIONS);
    }

    /**
     * Return the metadata Expenses
     * @return the Expenses
     */
    public RichText getExpenses()
    {
        return getValue(OrgUnit.EXPENSES);
    }

    /**
     * Return the metadata universalAdjustment
     * @return the universalAdjustment
     */
    public RichText getUniversalAdjustment()
    {
        return getValue(OrgUnit.UNIVERSAL_ADJUSTMENT);
    }

    /**
     * Return the metadata student_facilities
     * 
     * @return the student_facilities
     */
    public RichText getStudentFacilities()
    {
        return getValue(OrgUnit.STUDENT_FACILITIES);
    }

    /**
     * Return the metadata additionnal_infos
     * 
     * @return the additionnal_infos
     */
    public RichText getAdditionnalInfos()
    {
        return getValue(OrgUnit.ADDITIONNAL_INFOS);
    }

    /**
     * Return the metadata web_link
     * 
     * @return the web_link
     */
    public String getWebLinkLabel()
    {
        return getValue(OrgUnit.WEB_LINK_LABEL, false, StringUtils.EMPTY);
    }

    /**
     * Get the web link URL
     * @return the web link URL or null
     */
    public String getWebLinkURL()
    {
        return getValue(WEB_LINK_URL, false, StringUtils.EMPTY);
    }

    // --------------------------------------------------------------------------------------//
    // CDM-fr
    // --------------------------------------------------------------------------------------//
    @Override
    public String getCDMId() 
    {
        String cdmCode = getCdmCode();
        if (StringUtils.isEmpty(cdmCode))
        {
            return "FRUAI" + _getFactory()._getRootOrgUnitUAI() + "OU" + getUAICode();
        }
        return cdmCode;
    }
    
    @Override
    public String getCdmCode()
    {
        return getValue(CDM_CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCdmCode(String cdmCode) 
    {
        setValue(CDM_CODE, cdmCode);
    }
    
}
