/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;

/**
 * Common implementation of a {@link Content} which is part of a program, traversable
 * @param <F> the actual type of factory.
 */
public abstract class AbstractTraversableProgramPart<F extends ProgramPartFactory> extends AbstractProgramPart<F> implements TraversableProgramPart
{
    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public AbstractTraversableProgramPart(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public List<ProgramPart> getProgramPartChildren()
    {
        return Arrays.stream(getValue(CHILD_PROGRAM_PARTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(ProgramPart.class::cast)
                .collect(Collectors.toList());
    }
    
    @Override
    public boolean containsProgramPart(String contentId)
    {
        return ArrayUtils.contains(ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CHILD_PROGRAM_PARTS), contentId);
    }
    
    @Override
    public boolean hasProgramPartChildren()
    {
        return !ContentDataHelper.isMultipleContentDataEmpty(this, CHILD_PROGRAM_PARTS);
    }
    
}
