/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedContent;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedPermissionContext;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * {@link AccessController} for a ODF {@link ContextualizedContent} based on ODF role
 *
 */
public abstract class AbstractODFRoleForContextualizedContentAccessController implements AccessController, Serviceable, PluginAware
{
    /** The rights profile DAO */
    protected RightProfilesDAO _rightProfileDAO;
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentsTreeHelper;
    /** The ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** The ODF right helper */
    protected ODFRightHelper _odfRightHelper;

    private String _id;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _rightProfileDAO = (RightProfilesDAO) smanager.lookup(RightProfilesDAO.ROLE);
        _odfContentsTreeHelper = (ODFContentsTreeHelper) smanager.lookup(ODFContentsTreeHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _odfRightHelper = (ODFRightHelper) smanager.lookup(org.ametys.odf.rights.ODFRightHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public String getId()
    {
        return _id;
    }
    
    public boolean supports(Object object)
    {
        return object instanceof ContextualizedContent;
    }
    
    /**
     * Get the parents of the content for rights purpose
     * @param content the content
     * @param permissionCtx the permission context
     * @return the parents of content
     */
    protected Set<Content> getParents(Content content, ContextualizedPermissionContext permissionCtx)
    {
        if (content instanceof ProgramItem programItem)
        {
            EducationalPath educationalPath = permissionCtx.getEducationalPath();
            if (educationalPath == null)
            {
                return Set.of();
            }
            
            List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
            
            ProgramItem parent = programItemsInPath.getLast();
            
            // Update educational path in permission contexte
            permissionCtx.withEducationalPath(_removeLast(educationalPath));
            
            
            Set<Content> parents = new HashSet<>();
            parents.add((Content) parent);
            
            // Add orgunits
            List<String> ouIds = programItem.getOrgUnits();
            parents.addAll(ouIds.stream()
                    .filter(Objects::nonNull)
                    .filter(_resolver::hasAmetysObjectForId)
                    .map(_resolver::resolveById)
                    .map(OrgUnit.class::cast)
                    .collect(Collectors.toSet()));
            
            return parents;
        }
        else if (content instanceof OrgUnit ou)
        {
            OrgUnit parentOrgUnit = ou.getParentOrgUnit();
            if (parentOrgUnit != null)
            {
                return Set.of(parentOrgUnit);
            }
        }
        
        return Set.of();
    }
    
    /**
     * Get the permission context
     * @param contextualizedContent the initial contextualized content
     * @return the permission context.
     */
    protected ContextualizedPermissionContext getPermissionContext(ContextualizedContent contextualizedContent)
    {
        EducationalPath educationalPath = contextualizedContent.path();
        Content initialContent = contextualizedContent.content();
        
        // Remove initial content from education path
        List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
        if (programItemsInPath.getLast().getId().equals(initialContent.getId()))
        {
            educationalPath = _removeLast(educationalPath);
        }
        return new ContextualizedPermissionContext(initialContent, educationalPath);
    }
    
    private EducationalPath _removeLast(EducationalPath educationalPath)
    {
        List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
        List<ProgramItem> subList = programItemsInPath.subList(0, programItemsInPath.size() - 1);
        if (!subList.isEmpty())
        {
            return EducationalPath.of(subList.toArray(ProgramItem[]::new));
        }
        return null;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (object instanceof ContextualizedContent contextualizedContent)
        {
            return _getPermission(user, userGroups, rightId, contextualizedContent.content(), getPermissionContext(contextualizedContent));
        }
        
        return AccessResult.UNKNOWN;
    }
    
    private AccessResult _getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Content object, ContextualizedPermissionContext permissionCtx)
    {
        List<String> rights = getRightsInTargetProfile();
        if (rights.contains(rightId))
        {
            Set<UserIdentity> allowedUsers = getLocalAllowedUsers(object);
            if (allowedUsers.contains(user))
            {
                return AccessResult.USER_ALLOWED;
            }
        }
        
        AccessResult permission = AccessResult.UNKNOWN;
        
        Set<Content> parents = getParents(object, permissionCtx);
        if (parents != null)
        {
            for (Content parent : parents)
            {
                AccessResult parentResult = _getPermission(user, userGroups, rightId, parent, permissionCtx);
                permission = AccessResult.merge(permission, parentResult);
            }
        }
        
        return permission;
    }
    
    /**
     * Get the rights hold by target profile
     * @return the rights hold by target profile
     */
    protected synchronized List<String> getRightsInTargetProfile()
    {
        String profileId = getTargetProfileId();
        return StringUtils.isNotBlank(profileId) ? _rightProfileDAO.getRights(profileId) : List.of();
    }
    
    /**
     * Get the id of target profile
     * @return the id of target profile
     */
    protected abstract String getTargetProfileId();
    
    /**
     * Get the allowed users for this content taking into account the content itself and its parents
     * @param content the ODF content (program item or orgunit)
     * @param permissionCtx the permission context
     * @return the allowed users. Empty if no user is allowed on this content
     */
    protected Set<UserIdentity> getAllowedUsers(Content content, ContextualizedPermissionContext permissionCtx)
    {
        Set<UserIdentity> allowedUsers = getLocalAllowedUsers(content);
        
        Set<Content> parents = getParents(content, permissionCtx);
        if (parents != null)
        {
            for (Content parent : parents)
            {
                allowedUsers.addAll(getAllowedUsers(parent, permissionCtx));
            }
        }
        
        return allowedUsers;
    }
    
    /**
     * Get the local allowed users for this content
     * @param content the ODF content (program item or orgunit)
     * @return the allowed users. Empty if no user is allowed on this content
     */
    protected abstract Set<UserIdentity> getLocalAllowedUsers(Content content);

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        if (object instanceof ContextualizedContent contextualizedContent)
        {
            Set<UserIdentity> allowedUsers = getAllowedUsers((Content) object, getPermissionContext(contextualizedContent));
            if (allowedUsers.contains(user))
            {
                return getRightsInTargetProfile().stream()
                    .collect(Collectors.toMap(r -> r, r -> AccessResult.USER_ALLOWED));
            }
        }
        
        return Map.of();
    }
    
    /**
     * Get the attribute path for role
     * @return the attribute path for role
     */
    protected abstract String getRoleAttributePath();

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        if (object instanceof ContextualizedContent contextualizedContent && getRightsInTargetProfile().contains(rightId))
        {
            Set<UserIdentity> allowedUsers = getAllowedUsers((Content) object, getPermissionContext(contextualizedContent));
            if (allowedUsers != null)
            {
                return allowedUsers.stream()
                    .collect(Collectors.toMap(user -> user, user -> AccessResult.USER_ALLOWED));
            }
        }
        return Map.of();
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Map.of();
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object object)
    {
        return AccessController.getDefaultAccessExplanation(getId(), AccessResult.UNKNOWN);
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        return Map.of();
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content content)
        {
            return ODFContentHierarchicalAccessController.getContentObjectLabel(content, _odfContentsTreeHelper);
        }
        throw new RightsException("Unsupported object: " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return ODFContentHierarchicalAccessController.ODF_CONTEXT_CATEGORY;
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnonymous(Object object)
    {
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnyConnected(Object object)
    {
        return Map.of();
    }
    
    public Map<UserIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByUser(Object object)
    {
        return Map.of();
    }
    
    public Map<GroupIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByGroup(Object object)
    {
        return Map.of();
    }
}
