/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.schedulable.EducationalBookletSchedulable.EducationalBookletReport.ReportStatus;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * {@link Schedulable} for educational booklet.
 */
public class EducationalBookletSchedulable extends AbstractSendingMailSchedulable
{
    /** The directory under ametys home data directory for educational booklet */
    public static final String EDUCATIONAL_BOOKLET_DIR_NAME = "odf/booklet";
    
    /** Scheduler parameter name of including of subprograms */
    public static final String PARAM_INCLUDE_SUBPROGRAMS = "includeSubPrograms";
    
    /** Scheduler parameter name of including of subprograms */
    public static final String PARAM_PROGRAM_ITEM_ID = "programItemId";
    
    /** Map key where the report is stored */
    protected static final String _EDUCATIONAL_BOOKLET_REPORT = "educationalBookletReport";

    /** The avalon source resolver. */
    protected SourceResolver _sourceResolver;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }

    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        File bookletDirectory = new File(AmetysHomeHelper.getAmetysHomeData(), EDUCATIONAL_BOOKLET_DIR_NAME);
        FileUtils.forceMkdir(bookletDirectory);
        
        Map<String, Object> pdfParameters = new HashMap<>();
        
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        pdfParameters.put(PARAM_INCLUDE_SUBPROGRAMS, jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + PARAM_INCLUDE_SUBPROGRAMS));
        
        _generateProgramItemsEducationalBooklet(context, bookletDirectory, pdfParameters);
    }

    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context)
    {
        EducationalBookletReport report = (EducationalBookletReport) context.get(_EDUCATIONAL_BOOKLET_REPORT);
        return new I18nizableText("plugin.odf", _getMailSubjectBaseKey() + report.getCurrentStatus());
    }
    
    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context)
    {
        return new I18nizableText("plugin.odf", _getMailSubjectBaseKey() + "ERROR");
    }
    
    /**
     * The base key for mail subjects.
     * @return The prefix of an I18N key
     */
    protected String _getMailSubjectBaseKey()
    {
        return "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_MAIL_SUBJECT_";
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }
    
    @Override
    protected String _getSuccessMailBody(JobExecutionContext context, String language) throws IOException
    {
        EducationalBookletReport report = (EducationalBookletReport) context.get(_EDUCATIONAL_BOOKLET_REPORT);

        List<Content> exportedProgramItems = report.getExportedProgramItems();
        List<Content> programItemsInError = report.getProgramItemsInError();
        
        try
        {
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getSuccessMailSubject(context))
                .withLanguage(language);
            
            ReportStatus status = report.getCurrentStatus();
            String i18nKey = _getMailBodyBaseKey();
            
            if (status == ReportStatus.SUCCESS || status == ReportStatus.PARTIAL)
            {
                String downloadLink = _getDownloadLink(context, report, exportedProgramItems);
                
                Map<String, I18nizableTextParameter> i18nParams = Map.of("link", new I18nizableText(downloadLink), "programItem", _getProgramItemsI18nText(exportedProgramItems));
                bodyBuilder.addMessage(new I18nizableText("plugin.odf", i18nKey + "SUCCESS" + (exportedProgramItems.size() > 1 ? "_SEVERAL" : ""), i18nParams));
                bodyBuilder.withLink(downloadLink, new I18nizableText("plugin.odf", i18nKey + "DOWNLOAD_LINK" + (exportedProgramItems.size() > 1 ? "_SEVERAL" : "")));
            }
            if (status == ReportStatus.PARTIAL || status == ReportStatus.ERROR)
            {
                Map<String, I18nizableTextParameter> i18nParams = Map.of("programItem", _getProgramItemsI18nText(programItemsInError));
                bodyBuilder.addMessage(new I18nizableText("plugin.odf", i18nKey + "ERROR" + (exportedProgramItems.size() > 1 ? "_SEVERAL" : ""), i18nParams));
            }
            
            return bodyBuilder.build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML email body for education booklet export result", e);
            return null;
        }
    }
    
    /**
     * Get the link to download PDF
     * @param context the job execution context
     * @param report the report
     * @param exportedProgramItems the exported programs
     * @return the download
     * @throws IOException if failed to build the download uri
     */
    protected String _getDownloadLink(JobExecutionContext context, EducationalBookletReport report, List<Content> exportedProgramItems) throws IOException
    {
        String downloadLink = Strings.CI.removeEnd(Config.getInstance().getValue("cms.url"), "/index.html");
        
        if (exportedProgramItems.size() > 1)
        {
            // Compress to a ZIP if there are several exported program items
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
            String zipKey = ZonedDateTime.now().format(formatter);
            _generateEducationalBookletZip(context, report.getBookletDirectory(), exportedProgramItems, zipKey);
            downloadLink += "/plugins/odf/download/educational-booklet-" + zipKey + "/educational-booklet.zip";
        }
        else
        {
            Content content = exportedProgramItems.get(0);
            downloadLink += "/plugins/odf/download/" + content.getLanguage() + "/educational-booklet.pdf?programItemId=" + content.getId();
        }
        
        return downloadLink;
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable)
    {
        List<Content> programItems = Optional.of(context)
            .map(JobExecutionContext::getJobDetail)
            .map(JobDetail::getJobDataMap)
            .map(map -> map.getString(Scheduler.PARAM_VALUES_PREFIX + "programItemIds"))
            .map(ids -> ids.split(","))
            .map(Stream::of)
            .orElseGet(() -> Stream.empty())
            .filter(StringUtils::isNotBlank)
            .map(_resolver::<Content>resolveById)
            .collect(Collectors.toList());

        try
        {
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getErrorMailSubject(context))
                .withLanguage(language);
            
            Map<String, I18nizableTextParameter> i18nParams = Map.of("programItem", _getProgramItemsI18nText(programItems));
            bodyBuilder.addMessage(new I18nizableText("plugin.odf", _getMailBodyBaseKey() + "ERROR" + (programItems.size() > 1 ? "_SEVERAL" : StringUtils.EMPTY), i18nParams));
            
            if (throwable != null)
            {
                String error = ExceptionUtils.getStackTrace(throwable);
                bodyBuilder.withDetails(null, error, true);
            }
            
            return bodyBuilder.build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML email body for education booklet export result", e);
            return null;
        }
    }

    /**
     * The base key for mail bodies.
     * @return The prefix of an I18N key
     */
    protected String _getMailBodyBaseKey()
    {
        return "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_MAIL_BODY_";
    }
    
    /**
     * Transform a list of program items in a readable list.
     * @param programItems The program items to iterate on
     * @return An {@link I18nizableText} representing the program items
     */
    protected I18nizableText _getProgramItemsI18nText(List<Content> programItems)
    {
        List<String> programItemsTitle = programItems.stream()
            .map(c -> c.getTitle())
            .collect(Collectors.toList());
        
        String readableTitles;
        if (programItemsTitle.size() == 1)
        {
            readableTitles = programItemsTitle.get(0);
        }
        else
        {
            StringBuilder sb = new StringBuilder();
            sb.append("<ul>");
            programItemsTitle.stream().forEach(t -> sb.append("<li>").append(t).append("</li>"));
            sb.append("</ul>");
            
            readableTitles = sb.toString();
        }
        
        return new I18nizableText(readableTitles);
    }
    
    /**
     * Generate educational booklet for each program items
     * @param context the context
     * @param bookletDirectory the booklet directory
     * @param pdfParameters the parameters to generate PDF
     */
    protected void _generateProgramItemsEducationalBooklet(JobExecutionContext context, File bookletDirectory, Map<String, Object> pdfParameters)
    {
        EducationalBookletReport report = new EducationalBookletReport(bookletDirectory);
        
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String idsAsString = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + "programItemIds");
        for (String programItemId : StringUtils.split(idsAsString, ","))
        {
            Content programItem = _resolver.resolveById(programItemId);
            try
            {
                _generateProgramItemEducationalBookletPDF(bookletDirectory, programItem, pdfParameters);
                report.addExportedProgramItem(programItem);
            }
            catch (IOException e)
            {
                getLogger().error("An error occurred while generating the educational booklet of program item '{}' ({}).", programItem.getTitle(), ((ProgramItem) programItem).getCode(), e);
                report.addProgramItemIhError(programItem);
            }
        }
        
        context.put(_EDUCATIONAL_BOOKLET_REPORT, report);
    }
    
    /**
     * Generate the educational booklet for one program item
     * @param bookletDirectory the booklet directory
     * @param programItem the program item
     * @param pdfParameters the parameters to generate PDF
     * @throws IOException if an error occured with files
     */
    protected void _generateProgramItemEducationalBookletPDF(File bookletDirectory, Content programItem, Map<String, Object> pdfParameters) throws IOException
    {
        File programItemDir = new File(bookletDirectory, programItem.getName());
        if (!programItemDir.exists())
        {
            programItemDir.mkdir();
        }
        
        File langDir = new File (programItemDir, programItem.getLanguage());
        if (!langDir.exists())
        {
            langDir.mkdir();
        }
        
        Map<String, Object> localPdfParameters = new HashMap<>(pdfParameters);
        localPdfParameters.put(PARAM_PROGRAM_ITEM_ID, programItem.getId());
        _generateFile(
            langDir,
            "cocoon://_plugins/odf/booklet/" + programItem.getLanguage() + "/educational-booklet.pdf",
            localPdfParameters,
            "educational-booklet",
            "pdf"
        );
    }
    
    /**
     * Generate the zip with the educational booklet for each exported program items
     * @param context the context
     * @param bookletDirectory the booklet directory
     * @param exportedProgramItems the exported program items
     * @param zipKey the zip key
     * @throws IOException if an error occured with files
     */
    protected void _generateEducationalBookletZip(JobExecutionContext context, File bookletDirectory, List<Content> exportedProgramItems, String zipKey) throws IOException
    {
        String ids = exportedProgramItems.stream()
            .map(Content::getId)
            .collect(Collectors.joining(","));
        
        _generateFile(
            bookletDirectory,
            "cocoon://_plugins/odf/booklet/educational-booklet.zip",
            Map.of("programItemIds", ids),
            "educational-booklet-" + zipKey,
            "zip"
        );
    }

    /**
     * Generate a file from the uri
     * @param bookletDirectory the booklet directory where the file are created
     * @param uri the uri
     * @param params the parameters of the uri
     * @param name the name of the file
     * @param extension the extension of the file
     * @throws IOException if an error occured with files
     */
    protected void _generateFile(File bookletDirectory, String uri, Map<String, Object> params, String name, String extension) throws IOException
    {
        Request request = ContextHelper.getRequest(_context);
        
        SitemapSource source = null;
        File pdfTmpFile = null;
        try
        {
            // Set PDF parameters as request attributes
            for (Entry<String, Object> param : params.entrySet())
            {
                request.setAttribute(param.getKey(), param.getValue());
            }
            // Resolve the export to the appropriate pdf url.
            source = (SitemapSource) _sourceResolver.resolveURI(uri, null, params);
            
            // Save the pdf into a temporary file.
            String tmpFile = name + ".tmp." + extension;
            pdfTmpFile = new File(bookletDirectory, tmpFile);
            
            try (OutputStream pdfTmpOs = new FileOutputStream(pdfTmpFile); InputStream sourceIs = source.getInputStream())
            {
                SourceUtil.copy(sourceIs, pdfTmpOs);
            }
            
            // If all went well until now, rename the temporary file
            String fileName = name + "." + extension;
            File bookletFile = new File(bookletDirectory, fileName);
            if (bookletFile.exists())
            {
                bookletFile.delete();
            }
            
            if (!pdfTmpFile.renameTo(bookletFile))
            {
                throw new IOException("Fail to rename " + tmpFile + " to " + fileName);
            }
        }
        finally
        {
            if (pdfTmpFile != null)
            {
                FileUtils.deleteQuietly(pdfTmpFile);
            }
            
            if (source != null)
            {
                _sourceResolver.release(source);
            }
            
            for (Entry<String, Object> param : params.entrySet())
            {
                request.removeAttribute(param.getKey());
            }
        }
    }
    
    /**
     * Object to represent list of programs exported and list of programs with error after PDF generation
     */
    protected static class EducationalBookletReport
    {
        /**
         * Status of export
         */
        public enum ReportStatus
        {
            
            /** All program items have been exported successfully */
            SUCCESS,
            /** Program items have been exported partially */
            PARTIAL,
            /** Error during export */
            ERROR
            
        }
        
        private File _bookletDirectory;
        private List<Content> _exportedProgramItems;
        private List<Content> _programItemsInError;
        
        /**
         * The constructor
         * @param bookletDirectory The booklet directory
         */
        public EducationalBookletReport(File bookletDirectory)
        {
            _bookletDirectory = bookletDirectory;
            _exportedProgramItems = new ArrayList<>();
            _programItemsInError = new ArrayList<>();
        }
        
        /**
         * Get the booklet directory
         * @return the booklet directory
         */
        public File getBookletDirectory()
        {
            return _bookletDirectory;
        }
        
        /**
         * Get the list of exported program items
         * @return the list of exported program items
         */
        public List<Content> getExportedProgramItems()
        {
            return _exportedProgramItems;
        }
        
        /**
         * Add a content as exported
         * @param content the content to add
         */
        public void addExportedProgramItem(Content content)
        {
            _exportedProgramItems.add(content);
        }
        
        /**
         * Set the exported program items
         * @param programItems the list of exported program items
         */
        public void setExportedProgramItems(List<Content> programItems)
        {
            _exportedProgramItems = programItems;
        }
        
        /**
         * Get the program items in error
         * @return the list of program items in error
         */
        public List<Content> getProgramItemsInError()
        {
            return _programItemsInError;
        }
        
        /**
         * Add program item as error
         * @param programItem the program item to add
         */
        public void addProgramItemIhError(Content programItem)
        {
            _programItemsInError.add(programItem);
        }
        
        /**
         * The current status of the educational booklet generation.
         * @return The report status
         */
        public ReportStatus getCurrentStatus()
        {
            if (_programItemsInError.isEmpty())
            {
                return ReportStatus.SUCCESS;
            }
            
            if (_exportedProgramItems.isEmpty())
            {
                return ReportStatus.ERROR;
            }
            
            return ReportStatus.PARTIAL;
        }
    }
}
