/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.odf.init.OdfRefTableDataExtensionPoint;
import org.ametys.odf.init.OdfRefTableDataSynchronizationAccessController;
import org.ametys.plugins.contentio.csv.ImportCSVFileHelper;
import org.ametys.plugins.contentio.csv.SynchronizeModeEnumerator.ImportMode;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Import reference table data described by {@link OdfRefTableDataExtensionPoint}.
 */
public class OdfRefTableDataSynchronizationSchedulable extends AbstractSendingMailSchedulable
{
    private static final String __CONTEXT_KEY_SYNC_REPORT = OdfRefTableDataSynchronizationSchedulable.class.getName() + "$syncReport";
    
    private static final String __RESULT_IMPORTED_COUNT = "importedCount";
    private static final String __RESULT_DETAILS = "details";
    private static final String __RESULT_ERROR = "error";
    private static final String __RESULT_PARTIAL = "partial";
    private static final String __RESULT_SUCCESS = "success";
    private static final String __RESULT_DURATION = "duration";
    private static final String __IMPORT_MODE = "importMode";

    private OdfRefTableDataExtensionPoint _odfRefTableDataEP;
    private ImportCSVFileHelper _importCSVFileHelper;
    private ContentTypeExtensionPoint _contentTypeEP;
    private SourceResolver _srcResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfRefTableDataEP = (OdfRefTableDataExtensionPoint) manager.lookup(OdfRefTableDataExtensionPoint.ROLE);
        _importCSVFileHelper = (ImportCSVFileHelper) manager.lookup(ImportCSVFileHelper.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        ZonedDateTime begin = ZonedDateTime.now();
        getLogger().info("[BEGIN] ODF reference tables synchronization");
        
        Map<String, Object> result = new HashMap<>();
        result.put(__RESULT_IMPORTED_COUNT, 0);
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            request.setAttribute(OdfRefTableDataSynchronizationAccessController.ODF_REF_TABLE_SYNCHRONIZATION, true);
            
            String language = Config.getInstance().getValue("odf.programs.lang");
            ImportMode importMode = ImportMode.valueOf((String) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + __IMPORT_MODE));
            
            Map<String, String> dataToImport = _odfRefTableDataEP.getDataToImport();
            
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("All CSV files to import: {}", dataToImport.toString());
            }
            
            AtomicInteger count = new AtomicInteger();
            Integer total = dataToImport.size();
            for (String contentTypeId : dataToImport.keySet())
            {
                getLogger().info("[{}/{}] Synchronizing contents of type {}...", count.incrementAndGet(), total, contentTypeId);
                
                Map<String, Object> details = (Map<String, Object>) result.computeIfAbsent(__RESULT_DETAILS, __ -> new HashMap<>());
                
                if (_contentTypeEP.hasExtension(contentTypeId))
                {
                    ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
                    String dataURI = dataToImport.get(contentTypeId);
                    Source source = null;
                    
                    try
                    {
                        source = _srcResolver.resolveURI(dataURI);

                        try (
                            InputStream is = source.getInputStream();
                            InputStream data = IOUtils.buffer(is);
                        )
                        {
                            Map<String, Object> csvResult = _importCSVFileHelper.importContents(data, contentType, language, importMode);
                            details.put(contentTypeId, csvResult);
    
                            if (csvResult.containsKey(ImportCSVFileHelper.RESULT_ERROR))
                            {
                                _addToListInMap(result, __RESULT_ERROR, contentTypeId);
                                getLogger().error("Error while importing ODF reference table data for content type {} with the following reason: '{}'.", contentTypeId, csvResult.get(ImportCSVFileHelper.RESULT_ERROR));
                            }
                            else
                            {
                                Integer nbImported = (Integer) csvResult.getOrDefault(ImportCSVFileHelper.RESULT_IMPORTED_COUNT, 0);
                                result.put(__RESULT_IMPORTED_COUNT, (Integer) result.get(__RESULT_IMPORTED_COUNT) + nbImported);
                                Integer nbErrors = (Integer) csvResult.getOrDefault(ImportCSVFileHelper.RESULT_NB_ERRORS, 0);
                                Integer nbWarnings = (Integer) csvResult.getOrDefault(ImportCSVFileHelper.RESULT_NB_WARNINGS, 0);
                                if (nbErrors > 0 || nbWarnings > 0)
                                {
                                    _addToListInMap(result, __RESULT_PARTIAL, contentTypeId);
                                    getLogger().warn("Some errors and warnings while importing ODF reference table data for content type {} with {} synchronized entries, {} errors and {} warnings.", contentTypeId, nbImported, nbErrors, nbWarnings);
                                }
                                else
                                {
                                    _addToListInMap(result, __RESULT_SUCCESS, contentTypeId);
                                    getLogger().info("Success while importing ODF reference table data for content type {} with {} synchronized entries.", contentTypeId, nbImported);
                                }
                            }
                        }
                    }
                    catch (IOException e)
                    {
                        getLogger().error("Error while importing ODF reference table data of content type {} from file {}.", contentTypeId, dataURI, e);
                        details.put(contentTypeId, Map.of("error", "exception", "message", e.getMessage()));
                        _addToListInMap(result, __RESULT_ERROR, contentTypeId);
                    }
                    finally
                    {
                        if (source != null)
                        {
                            _srcResolver.release(source);
                        }
                    }
                }
                else
                {
                    getLogger().warn("The content type {} is not defined.", contentTypeId);
                    details.put(contentTypeId, Map.of("error", "unexisting"));
                    _addToListInMap(result, __RESULT_ERROR, contentTypeId);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error during ODF reference tables synchronization", e);
            throw e;
        }
        finally
        {
            request.removeAttribute(OdfRefTableDataSynchronizationAccessController.ODF_REF_TABLE_SYNCHRONIZATION);
            String duration = Optional.of(ZonedDateTime.now())
                .map(end -> Duration.between(begin, end))
                .map(Duration::toMillis)
                .map(DurationFormatUtils::formatDurationHMS)
                .orElse("undefined");
            
            if (getLogger().isInfoEnabled())
            {
                StringBuilder resume = new StringBuilder("Resume: ");
                resume.append(result.get(__RESULT_IMPORTED_COUNT));
                resume.append(" synchronized entries");
                if (result.containsKey(__RESULT_SUCCESS))
                {
                    resume.append(", ");
                    resume.append(((List<String>) result.get(__RESULT_SUCCESS)).size());
                    resume.append(" successful reference tables");
                }
                if (result.containsKey(__RESULT_PARTIAL))
                {
                    resume.append(", ");
                    resume.append(((List<String>) result.get(__RESULT_PARTIAL)).size());
                    resume.append(" partial successful reference tables");
                }
                if (result.containsKey(__RESULT_ERROR))
                {
                    resume.append(", ");
                    resume.append(((List<String>) result.get(__RESULT_ERROR)).size());
                    resume.append(" failed reference tables");
                }
                getLogger().info(resume.toString());
                getLogger().info("[END] ODF reference tables synchronization in {}", duration);
            }
            
            result.put(__RESULT_DURATION, duration);
            context.put(__CONTEXT_KEY_SYNC_REPORT, result);
        }
    }

    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context) throws Exception
    {
        String subjectKey = "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_SUBJECT";
        Map<String, Object> result = (Map<String, Object>) context.get(__CONTEXT_KEY_SYNC_REPORT);
        if (result.containsKey(__RESULT_PARTIAL) || result.containsKey(__RESULT_ERROR))
        {
            subjectKey += "_WITH_ERRORS";
        }
        return new I18nizableText("plugin.odf", subjectKey);
    }

    @SuppressWarnings("unchecked")
    @Override
    protected String _getSuccessMailBody(JobExecutionContext context, String language) throws Exception
    {
        try
        {
            Map<String, Object> result = (Map<String, Object>) context.get(__CONTEXT_KEY_SYNC_REPORT);
            
            StringBuilder sb = new StringBuilder();
            
            // Duration
            sb.append(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_DURATION", List.of(result.get(__RESULT_DURATION).toString())), language));
            
            // Imported count
            sb.append(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_COUNT", List.of(result.get(__RESULT_IMPORTED_COUNT).toString())), language));
            
            // Success
            if (result.containsKey(__RESULT_SUCCESS))
            {
                sb.append(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_SUCCESS", List.of(String.valueOf(((List<String>) result.get(__RESULT_SUCCESS)).size()))), language));
            }
            
            // Partial
            if (result.containsKey(__RESULT_PARTIAL))
            {
                List<String> partialCTypes = (List<String>) result.get(__RESULT_PARTIAL);
                
                List<String> partialCTypeLabels = partialCTypes.stream()
                    .map(id -> _contentTypeEP.getExtension(id))
                    .filter(Objects::nonNull)
                    .map(ContentType::getLabel)
                    .map(label -> _i18nUtils.translate(label, language))
                    .collect(Collectors.toList());
                
                sb.append(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_PARTIAL", List.of(String.valueOf(partialCTypes.size()), StringUtils.join(partialCTypeLabels, ", "))), language));
            }
            
            // Fail
            if (result.containsKey(__RESULT_ERROR))
            {
                List<String> errorCTypes = (List<String>) result.get(__RESULT_PARTIAL);
                
                List<String> errorCTypeLabels = errorCTypes.stream()
                    .map(id -> _contentTypeEP.getExtension(id))
                    .filter(Objects::nonNull)
                    .map(ContentType::getLabel)
                    .map(label -> _i18nUtils.translate(label, language))
                    .collect(Collectors.toList());
                
                sb.append(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_FAIL", List.of(String.valueOf(errorCTypeLabels.size()), StringUtils.join(errorCTypeLabels, ", "))), language));
            }
            
            String details = sb.toString();
            
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_SUBJECT"))
                    .addMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME"))
                    .addMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_RESUME_ASK_DETAILS"))
                    .withDetails(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_SUCCESS_BODY_REPORT"), details, false)
                    .withLanguage(language)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML email body for reference table synchronization report", e);
            return null;
        }
    }

    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_ERROR_SUBJECT");
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable) throws Exception
    {
        String error = StringUtils.join(ExceptionUtils.getStackFrames(throwable), "<br/>");
        
        return StandardMailBodyHelper.newHTMLBody()
                .withTitle(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_ERROR_SUBJECT"))
                .addMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_TABLEREF_SYNC_MAIL_ERROR_BODY"))
                .withDetails(null, error, false)
                .withLanguage(language)
                .build();
    }
    
    private void _addToListInMap(Map<String, Object> map, String itemName, String itemToAdd)
    {
        @SuppressWarnings("unchecked")
        List<String> list = (List<String>) map.computeIfAbsent(itemName, __ -> new ArrayList<>());
        list.add(itemToAdd);
    }
}
