/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.services;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.ModelHelper;

/**
 * Class to parse an attribute from an ODF content in Live version.
 */
public class ODFContentFieldGenerator extends ServiceableGenerator
{
    /** The source resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Content type helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
    }

    @Override
    public void generate() throws ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String contentId = request.getParameter("contentId");
        if (StringUtils.isEmpty(contentId))
        {
            throw new ProcessingException("The content ID is empty.");
        }

        try
        {
            DefaultContent content = _resolver.resolveById(contentId);

            // Is this an ODF content ?
            if (!_contentTypesHelper.isInstanceOf(content, "org.ametys.plugins.odf.Content.odfContent"))
            {
                throw new ProcessingException(String.format("The content with the ID '%s' is not an ODF content.", contentId));
            }

            // The content Have a Live version ?
            String[] labels = content.getAllLabels();
            if (!ArrayUtils.contains(labels, CmsConstants.LIVE_LABEL))
            {
                throw new ProcessingException(String.format("The content with the ID '%s' hasn't a Live version.", contentId));
            }
            
            // Get the Live version to the content
            content.switchToLabel(CmsConstants.LIVE_LABEL);
            
            // Do we have an attribute name ?
            String attributeName = request.getParameter("metadata");
            if (StringUtils.isEmpty(attributeName))
            {
                throw new ProcessingException(String.format("No attribute has been transmitted for the content '%s'.", contentId));
            }
            
            // The attribute exists ?
            if (!ModelHelper.hasModelItem(attributeName, content.getModel()))
            {
                throw new ProcessingException(String.format("There is no attribute named '%s' for the content '%s'.", attributeName, contentId));
            }
            
            
            // Generate SAX events for the attribute
            contentHandler.startDocument();
            XMLUtils.startElement(contentHandler, "metadata");
            
            content.dataToSAX(contentHandler, attributeName);
            
            XMLUtils.endElement(contentHandler, "metadata");
            contentHandler.endDocument();
        }
        catch (AmetysRepositoryException e)
        {
            throw new ProcessingException(String.format("Unable to get the content with the ID '%s'.", contentId), e);
        }
        catch (SAXException e)
        {
            throw new ProcessingException(String.format("Error while saxing the content '%s' on the 'export' metadata set.", contentId), e);
        }
    }
}
