/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.source;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceFactory;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This factory looks for files in the WEB-INF/stylesheets directory and falls back to the default stylesheet .
 * Use: odf-view://path_to_file
 */
public class ODFViewSourceFactory extends AbstractLogEnabled implements SourceFactory, Serviceable
{
    /** The regexp for a protocol */
    protected static final Pattern __SOURCE_PATTERN = Pattern.compile("^([^:])+(:([^:]+))?://(.*)$");
    
    /** The ODF view selector */
    private ODFViewSelector _odfViewSelector;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfViewSelector = (ODFViewSelector) manager.lookup(ODFViewSelector.ROLE);
    }

    @Override
    public Source getSource(String location, Map parameters) throws IOException, MalformedURLException
    {
        Matcher m = __SOURCE_PATTERN.matcher(location);
        if (!m.matches())
        {
            throw new MalformedURLException("URI must be like protocol:<name>://path/to/resource. Location was '" + location + "'");
        }
        
        String pluginName = m.group(3);
        if (StringUtils.isEmpty(pluginName))
        {
            pluginName = "odf"; // default plugin
        }
        
        String fileLocation = m.group(4);
        
        return _odfViewSelector.getSource(pluginName, fileLocation);
    }

    @Override
    public void release(Source source)
    {
        // empty method
    }
}
