/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.translation;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;

/**
 * Translation helper: get and set translations for a content.
 */
public final class TranslationHelper
{

    /** The translations metadata name. */
    public static final String TRANSLATIONS_METADATA = "translations";
    
    private TranslationHelper()
    {
        // Hide the default constructor.
    }
    
    /**
     * Get the translations of a content.
     * @param content the Content.
     * @return the translations as a Map of language -&gt; content ID.
     * FIXME CMS-9941: Use the DataHolder Internal API
     */
    public static Map<String, String> getTranslations(Content content)
    {
        Map<String, String> translations = new HashMap<>();
        
        RepositoryData repositoryData = _getContentRepositoryData(content);
        if (repositoryData.hasValue(TRANSLATIONS_METADATA))
        {
            try
            {
                RepositoryData translationsRepositoryData = repositoryData.getRepositoryData(TRANSLATIONS_METADATA);
                for (String language : translationsRepositoryData.getDataNames())
                {
                    String contentId = translationsRepositoryData.getString(language);
                    if (StringUtils.isNotBlank(contentId))
                    {
                        translations.put(language, contentId);
                    }
                }
            }
            catch (AmetysRepositoryException e)
            {
                throw new AmetysRepositoryException("Error while getting translations for content " + content.getId(), e);
            }
        }
        
        return translations;
    }
    
    /**
     * Set the translations on a content.
     * @param content the Content.
     * @param translations the translations to set, as a Map of language -&gt; content ID.
     * FIXME CMS-9941: Use the DataHolder Internal API
     */
    public static void setTranslations(ModifiableContent content, Map<String, String> translations)
    {
        ModifiableRepositoryData repositoryData = _getContentRepositoryData(content);
        ModifiableRepositoryData translationsRepositoryData;
        if (repositoryData.hasValue(TRANSLATIONS_METADATA))
        {
            translationsRepositoryData = repositoryData.getRepositoryData(TRANSLATIONS_METADATA);
        }
        else
        {
            _addLockToken(content);
            translationsRepositoryData = repositoryData.addRepositoryData(TRANSLATIONS_METADATA, RepositoryConstants.COMPOSITE_METADTA_NODETYPE);
        }
        
        for (String language : translationsRepositoryData.getDataNames())
        {
            _addLockToken(content);
            translationsRepositoryData.removeValue(language);
        }
        
        for (String language : translations.keySet())
        {
            _addLockToken(content);
            
            String contentId = translations.get(language);
            translationsRepositoryData.setValue(language, contentId);
        }
            
        content.saveChanges();
    }
    
    private static ModifiableRepositoryData _getContentRepositoryData(Content content)
    {
        if (content instanceof SimpleAmetysObject)
        {
            return new JCRRepositoryData(((SimpleAmetysObject) content).getNode());
        }
        else
        {
            throw new IllegalArgumentException("Unable to acces to the translation of the content '" + content.getId() + "'.");
        }
    }

    private static void _addLockToken(ModifiableContent content)
    {
        if (content instanceof LockableAmetysObject lockableContent)
        {
            lockableContent.setLockInfoOnCurrentContext();
        }
    }
}
