/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow.copy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.clientsideelement.SmartContentClientSideElement;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.CopyODFUpdater;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.content.CopyODFContentUpdaterExtensionPoint;
import org.ametys.odf.course.Course;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for ODF content copy
 * 
 */
public class CopyODFContentClientSideElement extends SmartContentClientSideElement
{
    /** The key to get the duplication mode */
    public static final String DUPLICATION_MODE_KEY = "$duplicationMode";
    
    /** The key to get the parent ProgramPart's id */
    public static final String PARENT_KEY = "$parent";
    
    /** The key to tell if we keep the creation title */
    public static final String KEEP_CREATION_TITLE_KEY = "$keepCreationTitle";
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** The client side element for copy */
    protected CopyContentClientInteraction _copyClientSideInteraction;
    /** JSON utils */
    protected JSONUtils _jsonUtils;
    /** The Solr index helper */
    protected SolrIndexHelper _solrIndexHelper;
    /** The copy ODF content updater extension point */
    protected CopyODFContentUpdaterExtensionPoint _copyODFContentUpdaterEP;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _odfHelper = (ODFHelper) sManager.lookup(ODFHelper.ROLE);
        _copyClientSideInteraction = (CopyContentClientInteraction) sManager.lookup(CopyContentClientInteraction.class.getName());
        _jsonUtils = (JSONUtils) sManager.lookup(JSONUtils.ROLE);
        _solrIndexHelper = (SolrIndexHelper) sManager.lookup(SolrIndexHelper.ROLE);
        _copyODFContentUpdaterEP = (CopyODFContentUpdaterExtensionPoint) sManager.lookup(CopyODFContentUpdaterExtensionPoint.ROLE);
    }
    
    /**
     * Determines if a ODF content can be copied and linked to a target content
     * @param copiedContentId The id of copied content
     * @param targetContentId The id of target content
     * @param contextualParameters the contextual parameters
     * @return the result with success to true if copy is available
     */
    @Callable(rights = "CMS_Rights_CopyContent")
    public Map<String, Object> canCopyTo(String copiedContentId, String targetContentId, Map<String, Object> contextualParameters)
    {
        Content copiedContent = _resolver.resolveById(copiedContentId);
        Content targetContent = _resolver.resolveById(targetContentId);
           
        Map<String, Object> result = new HashMap<>();
        
        // Put the mode to copy to prevent to check shareable course fields
        contextualParameters.put("mode", "copy");
        
        List<I18nizableText> errors = new ArrayList<>();
        if (!canCopyTo(copiedContent, targetContent, errors, contextualParameters))
        {
            // Invalid target
            result.put("errorMessages", errors);
            result.put("success", false);
        }
        else
        {
            result.put("success", true);
        }
        
        return result;
    }
    
    /**
     * Determines if a ODF content can be copied and linked to a target content
     * @param copiedContent The copied content
     * @param targetContent The target content
     * @param errors The list of error messages
     * @param contextualParameters the contextual parameters
     * @return true if the relation is valid, false otherwise
     */
    protected boolean canCopyTo(Content copiedContent, Content targetContent, List<I18nizableText> errors, Map<String, Object> contextualParameters)
    {
        return _odfHelper.isRelationCompatible(copiedContent, targetContent, errors, contextualParameters);
    }
    
    /**
     * Creates a content by copy of another one.<br>
     * Also handle the possible inner duplication depending on the duplication mode for each attribute of type "content".
     * @param baseContentId The id of content to copy
     * @param newContentTitle The title of content to create
     * @param viewNameToCopy The view name to copy. Can be null
     * @param fallbackViewNameToCopy The fallback view name to use if 'viewNameToCopy' does not exist. Can be null
     * @param viewMode The view type to copy. Can be null
     * @param initActionId The init workflow action id for copy
     * @param editActionId The workflow action for editing content
     * @param duplicationModeAsString the duplication mode
     * @param parentContentId the parent id under which the duplicated content will be created. Can be null
     * @return the copy result
     * @throws Exception if an error occurred during copy
     */
    // Only do read check on the source. The right check for creation will be done by the workflow
    @Callable(rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> createContentByCopy(String baseContentId, String newContentTitle, String viewNameToCopy, String fallbackViewNameToCopy, String viewMode, int initActionId, int editActionId, String duplicationModeAsString, String parentContentId) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        result.put("locked-contents", new ArrayList<>());

        String [] handledEventIds = new String[] {ObservationConstants.EVENT_CONTENT_ADDED, ObservationConstants.EVENT_CONTENT_MODIFIED,  ObservationConstants.EVENT_CONTENT_WORKFLOW_CHANGED};
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(handledEventIds);
            
            DuplicationMode duplicationMode = StringUtils.isNotBlank(duplicationModeAsString) ? DuplicationMode.valueOf(duplicationModeAsString.toUpperCase()) : DuplicationMode.SINGLE;
            
            if (!checkBeforeDuplication(baseContentId, parentContentId, duplicationMode, result))
            {
                result.put("check-before-duplication-failed", true);
            }
            else
            {
                Map<String, Object> copyMap = new HashMap<>();
                copyMap.put(DUPLICATION_MODE_KEY, duplicationMode.toString());
                copyMap.put(PARENT_KEY, parentContentId);
                
                String jsonMap = _jsonUtils.convertObjectToJson(copyMap);
                result = _copyClientSideInteraction.createContentByCopy(baseContentId, newContentTitle, jsonMap, viewNameToCopy, fallbackViewNameToCopy, viewMode, initActionId, editActionId);
                
                if (result.containsKey("contentIds"))
                {
                    @SuppressWarnings("unchecked")
                    Map<String, String> contentIds = (Map<String, String>) result.getOrDefault("contentIds", new HashMap<>());
                    Map<Content, Content> createdContents = contentIds.entrySet()
                        .stream()
                        .collect(Collectors.toMap(
                                e -> (Content) _resolver.resolveById(e.getKey()),
                                e -> (Content) _resolver.resolveById(e.getValue())
                            )
                        );
                    
                    ProgramItem baseContent = _resolver.resolveById(baseContentId);
                    String catalog = baseContent.getCatalog();
                    Content parentContent = StringUtils.isNotBlank(parentContentId) ? _resolver.resolveById(parentContentId) : null;
                    for (String id : _copyODFContentUpdaterEP.getExtensionsIds())
                    {
                        CopyODFUpdater updater = _copyODFContentUpdaterEP.getExtension(id);
                        updater.updateContents(catalog, catalog, createdContents, parentContent);
                    }
                }
            }
        }
        finally
        {
            _solrIndexHelper.restartSolrCommitForEvents(handledEventIds);
        }
        
        return result;
    }
    
    /**
     * Check that duplication can be performed without blocking errors
     * @param contentId The content id to copy
     * @param parentContentId The parent content id
     * @param duplicationMode The duplication mode
     * @param results the results map
     * @return true if the duplication can be performed
     */
    protected boolean checkBeforeDuplication(String contentId, String parentContentId, DuplicationMode duplicationMode, Map<String, Object> results)
    {
        boolean allRight = true;
        if (StringUtils.isNotBlank(parentContentId))
        {
            // Check if the parent is locked
            Content parentContent = _resolver.resolveById(parentContentId);
            if (_isLocked(parentContent))
            {
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> lockedContents = (List<Map<String, Object>>) results.get("locked-contents");
                Map<String, Object> contentParams = getContentDefaultParameters (parentContent);
                contentParams.put("description", _getLockedDescription(parentContent));
                lockedContents.add(contentParams);
            }
        }
            
        Content content = _resolver.resolveById(contentId);
        if (content instanceof ProgramItem programItem)
        {
            if (duplicationMode == DuplicationMode.SINGLE)
            {
                // Check if the child are locked
                for (ProgramItem programItemChild : _odfHelper.getChildProgramItems(programItem))
                {
                    if (_isLocked((Content) programItemChild))
                    {
                        @SuppressWarnings("unchecked")
                        List<Map<String, Object>> lockedContents = (List<Map<String, Object>>) results.get("locked-contents");
                        Map<String, Object> contentParams = getContentDefaultParameters ((Content) programItemChild);
                        contentParams.put("description", _getLockedDescription((Content) programItemChild));
                        lockedContents.add(contentParams);
                        
                        allRight = false;
                    }
                }
            }
            else if (duplicationMode == DuplicationMode.STRUCTURE_ONLY)
            {
                // Check if course child are locked
                for (Course course : _getCourse(programItem))
                {
                    if (_isLocked(course))
                    {
                        @SuppressWarnings("unchecked")
                        List<Map<String, Object>> lockedContents = (List<Map<String, Object>>) results.get("locked-contents");
                        Map<String, Object> contentParams = getContentDefaultParameters (course);
                        contentParams.put("description", _getLockedDescription(course));
                        lockedContents.add(contentParams);
                        
                        allRight = false;
                    }
                }
            }
        }
        
        return allRight;
    }
    
    /**
     * Get all first courses in sub item of the program item
     * @param programItem the program item
     * @return a set of courses
     */
    protected Set<Course> _getCourse(ProgramItem programItem)
    {
        Set<Course> courses = new HashSet<>();
        for (ProgramItem programItemChild : _odfHelper.getChildProgramItems(programItem))
        {
            if (programItemChild instanceof Course)
            {
                courses.add((Course) programItemChild);
            }
            else
            {
                courses.addAll(_getCourse(programItemChild));
            }
        }
        
        return courses;
    }

    /**
     * Enumeration for the mode of duplication
     */
    public enum DuplicationMode
    {
        /** Duplicate the content only */
        SINGLE,
        /** Duplicate the content and its structure */
        STRUCTURE_ONLY,
        /** Duplicate the content and its structure and its courses */
        FULL
    }
}
