/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.xslt;

import java.util.Collections;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;

/**
 * Helper component to be used from XSL stylesheets for CDM-fr export
 */
public class CDMFrXSLTHelper implements Serviceable, Contextualizable
{
    private static AmetysObjectResolver _ametysObjectResolver;
    private static Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Resolve the URI of a resource
     * @param type Type name (defined by the extension to use)
     * @param uri URI depending on the type
     * @return The uri resolved, the empty string if the uri could not be resolved, or the uri itself if there is no resolver adapted
     */
    public static String resolve(String type, String uri)
    {
        return resolve(type, uri, false);
    }
    
    /**
     * Resolve the URI of a resource
     * @param type Type name (defined by the extension to use)
     * @param uri URI depending on the type
     * @param download Is this uri for download purposes.
     * @return The uri resolved, the empty string if the uri could not be resolved, or the uri itself if there is no resolver adapted
     */
    public static String resolve(String type, String uri, boolean download)
    {
        if ("explorer".equals(type))
        {
            Resource resource = (Resource) _ametysObjectResolver.resolveById(uri);
            String fullPath = resource.getPath();
            
            StringBuilder result = new StringBuilder();
            result.append((String) Config.getInstance().getValue("cms.url"))
                    .append("/_odf/_resource")
                    .append(FilenameUtils.encodePath(fullPath));
            
            return URIUtils.encodePath(result.toString());
        }
        else if ("page".equals(type))
        {
            // Force absolute
            return ResolveURIComponent.resolve(type, uri, download, true, false);
        }
        else if ("mail".equals(type) || "phone".equals(type))
        {
            // special handlig for mailto: and tel: links, as they won't be correctly handled by _resolve
            return uri;
        }
        else
        {
            // Resolve URI forcing not absolute nor internal
            String resolvedUri = ResolveURIComponent.resolve(type, uri, download, false, false);
            return _resolve(resolvedUri);
        }
    }
    
    /**
     * Resolve the URI of a resource image
     * @param type Type name (defined by the extension to use)
     * @param uri URI depending on the type
     * @param height the height
     * @param width the width
     * @return The uri resolved, the empty string if the uri could not be resolved, or the uri itself if there is no resolver adapted
     */
    public static String resolveImage(String type, String uri, int height, int width)
    {
        return resolveImage(type, uri, height, width, false);
    }
    
    /**
     * Resolve the absolute URI of a resource image
     * @param type Type name (defined by the extension to use)
     * @param uri URI depending on the type
     * @param height the height
     * @param width the width
     * @param download Is this uri for download purposes.
     * @return The uri resolved, the empty string if the uri could not be resolved, or the uri itself if there is no resolver adapted
     */
    public static String resolveImage(String type, String uri, int height, int width, boolean download)
    {
        if ("explorer".equals(type))
        {
            if (height == 0 && width == 0)
            {
                return _resolveResource(uri, download, null);
            }
            
            return _resolveResource(uri, download, "_" + height + "x" + width);
        }
        else
        {
            // Resolve URI forcing not absolute nor internal
            String resolvedUri = ResolveURIComponent.resolveImage(type, uri, height, width, download, false, false);
            return _resolve(resolvedUri);
        }
    }
    
    /**
     * Resolve the absolute URI of a resource video
     * @param type Type name (defined by the extension to use)
     * @param uri URI depending on the type
     * @param download Is this uri for download purposes.
     * @return The uri resolved, the empty string if the uri could not be resolved, or the uri itself if there is no resolver adapted
     */
    public static String resolveVideo(String type, String uri, boolean download)
    {
        if ("explorer".equals(type))
        {
            return _resolveResource(uri, download, null);
        }
        else
        {
            // Resolve URI forcing not absolute nor internal
            String resolvedUri = ResolveURIComponent.resolve(type, uri, download, false, false);
            return _resolve(resolvedUri);
        }
    }
    
    private static String _resolve (String resolvedUri)
    {
        // Redirect to "odf" workspace
        Request request = ContextHelper.getRequest(_context);
        String contextPath = request.getContextPath();
        
        StringBuilder result = new StringBuilder();
        result.append((String) Config.getInstance().getValue("cms.url"))
            .append("/_odf")
            .append(org.apache.commons.lang3.StringUtils.substringAfter(resolvedUri, contextPath));
            
        return result.toString();
    }
    
    private static String _resolveResource (String uri, boolean download, String suffix)
    {
        Resource resource = (Resource) _ametysObjectResolver.resolveById(uri);
        String fullPath = resource.getPath();
        
        int i = fullPath.lastIndexOf(".");
        String extension = i != -1 ? fullPath.substring(i) : null;
        fullPath = i != -1 ? fullPath.substring(0, i) : fullPath; 
        
        StringBuilder result = new StringBuilder();
        
        // Always use absolute url
        result.append((String) Config.getInstance().getValue("cms.url"))
                .append("/_odf/_resource")
                .append(FilenameUtils.encodePath(fullPath));
        
        if (suffix != null)
        {
            result.append(suffix);
        }
        
        if (extension != null)
        {
            result.append(extension);
        }
        
        return URIUtils.encodeURI(result.toString(), download ? Collections.singletonMap("download", "true") : null);
    }
}
