/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.notification;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.core.ui.user.ProfileImageResolverHelper;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.PageSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityFactory;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.activities.ActivityTypeExtensionPoint;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.activities.AbstractPageActivityType;
import org.ametys.web.activities.AbstractSiteAwareActivityType;
import org.ametys.web.activities.PageResourcesUpdatedActivityType;
import org.ametys.web.activities.PageUpdatedActivityType;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.rights.PageRightAssignmentContext;

/**
 * Helper for page notifications
 */
public class PageNotificationsHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role */
    public static final String ROLE = PageNotificationsHelper.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    /** The activity extension point */
    protected ActivityTypeExtensionPoint _activityTypeEP;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
        _activityTypeEP = (ActivityTypeExtensionPoint) smanager.lookup(ActivityTypeExtensionPoint.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(ROLE, 
                new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_PAGE_CACHE"),
                new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_PAGE_CACHE_DESCRIPTION"),
                true,
                null);
    }
    
    /**
     * Get the unread page notifications for each subscription of the current user
     * @param siteName the site name
     * @return the unread page notifications
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public List<Map<String, Object>> getUnreadPages(String siteName)
    {
        List<Map<String, Object>> unreadPages = new ArrayList<>();
        
        Site site = _siteManager.getSite(siteName);
        
        UserIdentity user = _currentUserProvider.getUser();
        PageSubscriptionType pageSubscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
        
        // Ignore subscription frequency for the notifications on intranet (BroadcastChannel.SITE)
        List<Subscription> subscriptions = pageSubscriptionType.getUserSubscriptions(site, null, BroadcastChannel.SITE, user, false, null);
        for (Subscription subscription : subscriptions)
        {
            String pageId = subscription.getValue(PageSubscriptionType.PAGE);
            
            Activity lastPageActivity = getLastActivity(siteName, pageId, Frequency.INSTANT);
            if (lastPageActivity != null)
            {
                ZonedDateTime lastPageActivityDate = lastPageActivity.getDate();
                ZonedDateTime lastReadDate = pageSubscriptionType.getLastReadDate(subscription, user);
                boolean hasRead = lastReadDate != null && lastReadDate.isAfter(lastPageActivityDate);
                if (!hasRead)
                {
                    UserIdentity author = lastPageActivity.getAuthor();
                    Map<String, Object> author2json = _userHelper.user2json(author);
                    author2json.put("imgUrl", ProfileImageResolverHelper.resolve(author.getLogin(), author.getPopulationId(), 64, null));
                    
                    unreadPages.add(Map.of(
                        "id", subscription.getId(),
                        "pageId", pageId,
                        "lastActivityDate", lastPageActivityDate,
                        "author", author2json
                    ));
                }
            }
        }
        
        return unreadPages;
    }
    
    /**
     * Get the last page activity 
     * @param siteName the site name
     * @param pageId the page
     * @param frequency the frequency
     * @return the last page activity
     */
    public Activity getLastActivity(String siteName, String pageId, Frequency frequency)
    {
        FrequencyTiming timing = new FrequencyTiming(FrequencyHelper.getDefaultFrequencyDay(), FrequencyHelper.getDefaultFrequencyTime());
        ZonedDateTime notificationDate = FrequencyHelper.getNotificationDate(frequency, timing);
        
        PageSubscriptionKey key = PageSubscriptionKey.of(siteName, pageId, frequency, notificationDate);
        return Optional.ofNullable(_getCache().get(key, k -> _getLastActivity(siteName, pageId, frequency, notificationDate)))
                .map(this::_resolveSilently)
                .orElse(null);
    }
    
    private String _getLastActivity(String siteName, String pageId, Frequency frequency, ZonedDateTime notificationDate)
    {
        return _getPageActivities(siteName, pageId, frequency, notificationDate)
            .stream()
            .max((a1, a2) -> a1.getDate().compareTo(a2.getDate()))
            .map(Activity::getId)
            .orElse(null);
    }
    
    private AmetysObjectIterable<Activity> _getPageActivities(String siteName, String pageId, Frequency frequency, ZonedDateTime notificationDate)
    {
        List<Expression> finalExprs = new ArrayList<>();
        
        finalExprs.addAll(FrequencyHelper.getDateExpressions(frequency, notificationDate));
        
        Expression[] activityTypeExpressions = _activityTypeEP.getExtensionsIds()
            .stream()
            .map(_activityTypeEP::getExtension)
            .filter(this::_filterActivity)
            .map(type -> new StringExpression(ActivityFactory.ACTIVITY_TYPE_ID, Operator.EQ, type.getId()))
            .toArray(Expression[]::new);
        finalExprs.add(new OrExpression(activityTypeExpressions));
        finalExprs.add(new StringExpression(AbstractPageActivityType.PAGE_ID, Operator.EQ, pageId));
        finalExprs.add(new StringExpression(AbstractSiteAwareActivityType.SITE_NAME, Operator.EQ, siteName));
        
        Expression finalExpr = new AndExpression(finalExprs.toArray(new Expression[finalExprs.size()]));
        
        String xpathQuery = ActivityHelper.getActivityXPathQuery(finalExpr);
        return _resolver.query(xpathQuery);
    }
    
    private boolean _filterActivity(ActivityType type)
    {
        return type instanceof PageUpdatedActivityType || type instanceof PageResourcesUpdatedActivityType;
    }
    
    private Activity _resolveSilently(String activityId)
    {
        try
        {
            return _resolver.resolveById(activityId);
        }
        catch (Exception e) 
        {
            getLogger().warn("Can't resolve activity with id '{}'", activityId, e);
        }
        
        return null;
    }
    
    /**
     * Mark page as read for the current user
     * @param pageId the page id
     */
    @Callable(rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = PageRightAssignmentContext.ID)
    public void markPageAsRead(String pageId)
    {
        PageSubscriptionType pageSubscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
        Page page = _resolver.resolveById(pageId);
        UserIdentity user = _currentUserProvider.getUser();
        
        PageSubscriptionContext context = PageSubscriptionContext.newInstance().withPage(page);
        List<Subscription> subscriptions = pageSubscriptionType.getUserSubscriptions(page.getSite(), null, BroadcastChannel.SITE, user, false, context);
        for (Subscription subscription : subscriptions)
        {
            pageSubscriptionType.markAsRead(subscription, user);
        }
    }
    
    private Cache<PageSubscriptionKey, String> _getCache()
    {
        return _cacheManager.get(ROLE);
    }
    
    /**
     * Clear page subscription cache 
     * @param siteName the site name
     * @param pageId the page id
     */
    public void clearCache(String siteName, String pageId)
    {
        _getCache().invalidate(PageSubscriptionKey.of(siteName, pageId));
    }
    
    static class PageSubscriptionKey extends AbstractCacheKey
    {
        PageSubscriptionKey(String siteName, String pageId, Frequency frequency, ZonedDateTime notificationDate)
        {
            super(siteName, pageId, frequency, notificationDate);
        }
        
        static PageSubscriptionKey of(String siteName, String pageId, Frequency frequency, ZonedDateTime notificationDate)
        {
            return new PageSubscriptionKey(siteName, pageId, frequency, notificationDate);
        }
        
        static PageSubscriptionKey of(String siteName, String pageId)
        {
            return new PageSubscriptionKey(siteName, pageId, null, null);
        }
    }
}
