/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.page;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.observation.AbstractNotifierAction;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;

/**
 * Action to subscribe to a page
 */
public class PageSubscribeAction extends AbstractNotifierAction
{
    /** The pattern to check emails */
    protected static final Pattern EMAIL_VALIDATOR = SendMailHelper.EMAIL_VALIDATION;
    
    /** The pattern to check text input */
    protected static final Pattern TEXT_VALIDATOR = Pattern.compile("^\\s*$");
    
    /** The ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** The subscription dao */
    protected PageSubscriptionDAO _subscriptionDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _subscriptionDAO = (PageSubscriptionDAO) smanager.lookup(PageSubscriptionDAO.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        List<I18nizableText> errors = new ArrayList<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String action = request.getParameter("page-subscribe-action");
        result.put("action", action);
        
        String email = request.getParameter("email");
        
        if (email == null || !EMAIL_VALIDATOR.matcher(StringUtils.trimToEmpty(email.toLowerCase())).matches() || TEXT_VALIDATOR.matcher(StringUtils.trimToEmpty(email)).matches())
        {
            errors.add(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_FORM_MAIL_ERROR_MAILBY"));
        }
        else
        {
            result.put("email", email);
            
            String pageId = request.getParameter("page-id");
            if (pageId == null) 
            {
                throw new IllegalArgumentException("Unable to subscribe or unsubscribe to the page: cannot determine the current page");
            }
            else
            {
                // Force default workspace
                String currentWorspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
                
                try
                {
                    Page page = _resolver.resolveById(pageId);
                    
                    
                    if (action.equals("subscribe"))
                    {
                        _subscriptionDAO.addSubscriber(page, email);
                    }
                    else if (action.equals("unsubscribe"))
                    {
                        _subscriptionDAO.removeSubscriber(page, email);
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    throw new IllegalArgumentException("Unable to subscribe or unsubscribe to the unknown page with id '" + pageId + "'");
                }
                finally
                {
                    RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorspace);
                }
            }
        }
         
        if (!errors.isEmpty())
        {
            result.put("error", errors);
        }
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        
        return EMPTY_MAP;
    }
}
